// basicSignals.c
//
// Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:This example demonstrates the basic signal functionality .
//

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>      //The header for signal functionality.
#include <time.h>        //The header for timer functionality.
#include <e32def.h>

char const *gstr = "\nPress '7'+Enter to return to the options menu or 'e'+ENTER to exit the example\n";
char const *gnote = "\nPress the Enter key to proceed with the use case\n";

/**
 Wait for a key press
*/
void PressKey()
	{
	fflush(stdout);
	getchar();
	}

/**
 Provides total time elapsed, in seconds.
 @param str Character string to be displayed.
 */
void timestamp(char *str) 
    {
    time_t t;
    time(&t);
    printf("The time %s is %s\n", str, ctime(&t));
    }

/**
 The handler for the SIGCHLD signal.
 @param signum The signal number
 */
void SIGCHLD_handler(int signum)
    {
    if(signum == SIGCHLD)
        printf("Now in SIGCHLD handler function\n");
    }

/**
 The handler for the SIGALRM signal.
 @param signum The signal number
 */
void SIGALRM_handler(int signum)
    {
    if(signum == SIGALRM)
        printf("Now in SIGALRM handler function\n");
    }

/**
 The handler for the SIGUSR1 signal.
 @param signum The signal number
 */
void SIGUSR1_handler(int signum)
    {
    if(signum == SIGUSR1)
        printf("Now in SIGUSR1 handler function\n");
    }

/**
 The handler for SIGUSR2 signal.
 @param signum The signal number
 */
void SIGUSR2_handler(int signum)
    {
    if(signum == SIGUSR2)
        printf("Now in SIGUSR2 handler function\n");
    
    sigrelse(SIGUSR1);
    printf("SIGUSR1 got released and hence its handler function will get called\n");
    }

/**
 The handler for the SIGPIPE signal.
 @param signum The signal number
 */
void SIGPIPE_handler(int signum)
    {
    if(signum == SIGPIPE)
        printf("In SIGPIPE handler function\n");
    }

/**
 Displays the menu for the signal options.
 */
void DisplaySignalOptions()
    {
    printf("\nSignal options:\n");
    printf("1) Sending and handling a signal using the default handler\n");
    printf("2) Sending and handling a signal using a customized signal handler\n");
    printf("3) Ignoring an incoming signal\n");
    printf("4) Blocking and releasing a signal\n");
    printf("5) Waiting for a signal\n");
    printf("6) Generating and handling a SIGPIPE signal\n");
    printf("Type the number of the option you wish to run followed by an enter,\n");
    printf("or type 'e'+Enter to exit the example.\n");
    }

/**
 Handles the raised signals using the default handler.
 */
void CallDefaultHandler()
    {
    pid_t my_pid;
    my_pid = getpid();
    printf("Raising a SIGCHLD signal\n");
    
    signal(SIGCHLD, SIG_DFL);
    
    //Raising SIGCHLD signal using kill command.
    kill(my_pid, SIGCHLD);
    
    printf("Default handler causes SIGCHLD to get ignored\n");
    PressKey();
    
    printf("\nRaising a SIGALRM signal\nDefault handler of SIGALRM will terminate the existing process.\n");
    printf("\nRestart the example to view the other use cases.\n");
    printf("Press the enter key to terminate the example.\n");
    PressKey();
    
    signal(SIGALRM, SIG_DFL);
    
    //Raising SIGALRM signal.
    kill(my_pid, SIGALRM);
    printf("After SIGALRM being raised\n");
    
    printf(gstr);
    }

/**
 Handle the raised signal using a customized handler 
 */
void CallSignalHandler()
    {
    struct sigaction sa1,sa2;
    sa1.sa_handler = SIGCHLD_handler;
    sigemptyset(&sa1.sa_mask);
    sa1.sa_flags = 0;  
    
    sa2.sa_handler = SIGALRM_handler;
    sigemptyset(&sa2.sa_mask);
    sa2.sa_flags = 0; 
    
    if (sigaction(SIGCHLD, &sa1, NULL) == -1)
        {
        printf("error in setting the handler for SIGCHLD\n");
        }
  
    if (sigaction(SIGALRM, &sa2, NULL) == -1)
        {
        printf("error in setting the handler for SIGALRM\n");
        }
    
    printf("\nRaising a SIGCHLD signal\n");
    kill(getpid(), SIGCHLD);                 //Raising SIGCHLD signal using kill command.
    printf("SIGCHLD has been handled\n");
    
    printf("\nRaising a SIGALRM signal\n");
    kill(getpid(), SIGALRM);                 //Raising SIGALRM signal using kill command.
    printf("SIGALRM has been handled\n");
    
    printf(gstr);
    }

/**
 The following method provides an example of how to ignore a specific signal.
 */
void IgnoreSignal()
    {
        struct sigaction sa1,sa2;   //The variables for holding signal specific settings, used in sigaction function call. 
        sa1.sa_handler = SIG_IGN;
        sigemptyset(&sa1.sa_mask);
        sa1.sa_flags = 0;  
        
        sa2.sa_handler = SIG_IGN;
        sigemptyset(&sa2.sa_mask);
        sa2.sa_flags = 0; 
        
        // Set the signal handler for SIGCHLD.
        if (sigaction(SIGCHLD, &sa1, NULL) == -1)
                   printf("sigaction for SIGCHLD failed\n");
        
        // Set the signal handler for SIGALRM.
        if (sigaction(SIGALRM, &sa2, NULL) == -1)
                printf("sigaction for SIGALRM failed\n");
        
        printf("\nRaising a SIGCHLD signal\n");
        kill(getpid(), SIGCHLD);             //Raisisng SIGCHLD signal using kill command.
        printf("SIGCHLD was ignored\n");
        
        printf("\nRaising a SIGALRM signal\n");
        kill(getpid(), SIGALRM);             //Raisisng SIGALRM signal using kill command.
        printf("SIGALRM was ignored\n");
        
        printf(gstr);
    }

/**
 The following method provides an example of how to mask and release a particular signal.
 */
void MaskSignal()
    {
    // Define a new mask set.
    sigset_t mask_set;
    
    printf("* This use case uses SIGUSR1 and SIGUSR2 signals to demonstrate the mask and the release of a signal\n");
    printf("* We mask the SIGUSR1 signal so that when it is raised it gets blocked and keeps pending until it is\nreleased.\n");
    printf("* We then raise the SIGUSR2 signal to release the blocked SIGUSR1 signal.\n");
    printf("* The custom handler of the SIGUSR2 signal releases SIGUSR1, and SIGUSR1's\ncustom handler gets called to handle it.\n");
    
    printf(gnote);
    PressKey();
    PressKey();
    
    // Set the signal handler for SIGUSR1 and SIGUSR2 signals.
    signal(SIGUSR1, SIGUSR1_handler);
    signal(SIGUSR2, SIGUSR2_handler);
   
    // Clearing the mask set so that it doesn't contain any signal numbers.
    sigemptyset(&mask_set);
    // Adding SIGUSR1 to the mask set.
    sigaddset(&mask_set, SIGUSR1);
    // Masking the signal.
    printf("Masking the SIGUSR1 signal\n");
    sigprocmask(SIG_SETMASK, &mask_set, NULL);
    
    printf("\nRaising the SIGUSR1 signal");
    printf(gnote);
    PressKey();
    
    raise(SIGUSR1);
    printf("SIGUSR1 was ignored as it was masked\n");
    
    printf("\nRaising the SIGUSR2 signal");
    printf(gnote);
    PressKey();
    
    raise(SIGUSR2);
    printf("SIGUSR2 has been handled\n");
    
    printf(gstr);
    }

/**
 The following method demonstrates how a process can wait for a particular signal. 
 */
void WaitForSignal()
    {
    int count,result;
    const int delay = 2;                //Time delay of 2 seconds.
    sigset_t waitset,outset;            // Define new mask sets.
    struct timespec timeout;            // A variable for holding timeout values.
    siginfo_t info;                     // A variable for holding signal info.
    
    signal(SIGALRM, SIGALRM_handler);   // Set the signal handler for SIGALRM.
    sigemptyset(&waitset);
    sigaddset(&waitset, SIGALRM);       // Adding SIGALRM to mask set.
    sigprocmask(SIG_BLOCK, &waitset, NULL);   
     
    printf("* This use case demonstrates how a process waits for a SIGALRM signal.\n");
    printf("* A fixed timeout period of 5 seconds is set.\n");
    printf("* The SIGALRM signal is first blocked and then we wait on it until it is raised.\n");
    printf("* The SIGALRM signal is raised twice, once after 6 seconds and another after 4 seconds.\n");
    printf("* When SIGALRM is raised after 6 seconds an error is generated as timeout has occurred.\nBut when SIGALRM is raised after 4 seconds the signal is received.\n");
    printf("* Even though the SIGALRM signal is received it cannot be handled as it is blocked.\nHence we finally unblock it and handle it.\n");
    
    printf(gnote);
    PressKey();
    PressKey();
    
    //Setting the timeout for 5 seconds.
    timeout.tv_sec = 5;
    timeout.tv_nsec = 500;
    
    for(count=0;count<2;count++)
        {
        if(count)
            {
            printf("Raising SIGALRM after 6 sec\n");
            printf(gnote);
            PressKey();
            alarm(6);
            }
        else
            {
            printf("Raising SIGALRM after 4 sec\n");
            printf(gnote);
            PressKey();
            alarm(4);
            }    
        timestamp("before sigwait()");
        result = sigtimedwait(&waitset, &info, &timeout); 
      
        if( result == SIGALRM )
            printf("sigwait() returned successful for signal %d\n", info.si_signo);
        else 
            {
            printf("sigwait() returned an error, since the alarm was generated after the specified time period.error number = %d\n", errno);
            }
        timestamp("after sigwait()");
        sleep(delay);
        }
    
    printf("Press Enter to unblock the SIGALRM signal and handle it\n");
    PressKey();
    sigemptyset(&outset);
    sigprocmask(SIG_UNBLOCK, &waitset, &outset); 
    
    printf(gstr);
    }

/**
  The following method shows how writing to a broken pipe can result in the generation of a SIGPIPE signal. 
 */
void HandleSigpipe()
    {
    int ret,fds[2];   // fds, used for holding the read and write descriptors for pipe
    printf("* This use case demonstrates how writing to a broken pipe generates a SIGPIPE signal.\n");
    printf("* We first create a pipe and obtain the read and write descriptors.\n");
    printf("* To obtain a broken pipe we close the read end of the pipe.\n");
    printf("* Once a write is performed to the broken pipe, an EPIPE error is returned\nand hence the SIGPIPE signal is generated.\n");
    
    printf(gnote);
    getchar();
    PressKey();
    
    //Set the signal handler for the SIGPIPE signal.
    signal(SIGPIPE, SIGPIPE_handler);
    
    //Create a pipe.
    ret = pipe(fds);
    if(ret != 0)
        {
        printf("Failed to create a pipe and the errno is %d\n",errno);
        }
    else
        {
        printf("The pipe was created successfully\n");
        }
    //Close the read end of a pipe.
    close(fds[0]); 
    
    //Write to a pipe whose read end is closed (a broken pipe).
    printf("Writing to a broken pipe\n");
    ret = write(fds[1], "sigpipe", 10);
    if((ret != -1) && (errno != EPIPE))
        {
        printf("Failed to generate the SIGPIPE signal on pipe\n");
        }
    else
        {
        printf("Writing to a pipe with the read end closed, has returned EPIPE and hence generated the SIGPIPE signal\n");
        }
    
    printf(gstr);
    }

/**
 Main function.
 The entry point of the example.
*/
int main()
	{
	int character;
	int val = TRUE;
	//Welcome Note.
	printf("************************************************************************\n");
	printf("*                  Welcome to the Signal Example                       *\n");
	printf("*      This example demonstrates some basic signal use cases           *\n");
	printf("************************************************************************\n");
	printf("Please refer to the overview for a better understanding of this example.\n");
	
	//Generating menu for various signal functions.
	DisplaySignalOptions();
	while(val)  
	    {
	    while((character = getchar())!= '\n')
	        {                
	        switch(character)
                {
                case '1':
                     printf("\nDemonstrating sending and handling of 2 signals by the default handler\n");
                     printf("**********************************************************************\n");
                     CallDefaultHandler();
                     break;
                case '2':
                     printf("\nDemonstrating sending and handling of 2 signals using a custom signal handler\n");
                     printf("*****************************************************************************\n");
                     CallSignalHandler();
                     break;
                case '3':
                     printf("\nDemonstrating ignoring an incoming signal\n");
                     printf("*****************************************\n");
                     IgnoreSignal();
                     break;
                case '4':
                     printf("\nDemonstrating masking and releasing of a signal\n");
                     printf("***********************************************\n");
                     MaskSignal();
                     break;
                case '5':
                     printf("\nDemonstrating waiting for a signal\n");
                     printf("**********************************\n");
                     WaitForSignal();
                     break;
                case '6':
                     printf("\nDemonstrating generating and handling a SIGPIPE signal\n");
                     printf("******************************************************\n");
                     HandleSigpipe();
                     break;
                case '7':
                     DisplaySignalOptions();
                     break;
                case 'e':
                    printf("Exiting from the main menu\n");
                    val = FALSE;
                    break;
                default:
                    printf("You chose a wrong option!!! Select again.\n");
                    break;
                }
	        }          
	    }    
	printf("Ok [Press the Enter key to exit]\n");
	PressKey();
   	return EXIT_SUCCESS;    //Returns the success code.
	}
