// Copyright (c) 2000-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Examples to demonstrate arrays
//

#include "CommonFramework.h"
#include <e32math.h>
#include <e32std.h>

_LIT(KMsgNewLine,"\n");
_LIT(KMsgPressAnyKey," (press any key to continue)\n");
_LIT(KFormat1,"X%u");

const TText* firstElement =  _S("DynamicArray: first");
const TText* secondElement = _S("Example Code: second");
const TText* thirdElement =  _S("ArrayKeys Ex: third");

//
// A T type class used in the example demonstrating the:
// CArrayFixFlat class
//
class TElement
	{
public :
	TElement();
public :
	TBuf<4> iData;
	};

TElement::TElement()
	{
	_LIT(KTextBase,"BASE");
	iData = KTextBase;
	}

class CArrayElement
	{
public :
	CArrayElement();
	TBool operator==(const CArrayElement& aElement) const;
public :
	TBuf<4> iData;
	};

CArrayElement::CArrayElement()
	{
	_LIT(KTextBase,"BASE");
	iData = KTextBase;
	}

TBool CArrayElement::operator==(const CArrayElement& aElement) const
	{
	return (iData == aElement.iData);
	}

//
// A CBase derived class used in the example demonstrating the CArrayPtrFlat
// array
//
class CElement : public CBase
	{
public :
	CElement();
	~CElement();
	void SetTextL(const TDesC& aText);
public :
	HBufC* iBuf;
	};

CElement::CElement()
	{
	}

CElement::~CElement()
	{
	delete iBuf;
	}

/**
Copies data into the CElement::iBuf heap descriptor replacing any existing data.
@param aText A descriptor.
*/
void CElement::SetTextL(const TDesC& aText)
	{
	if (!iBuf)
		{
		// The buffer is not created.
		// Hence, create the heap buffer.
		// Set the length of CElement::iBuf to the length of aText.
		iBuf  = HBufC::NewL(aText.Length());
		}

	else if(iBuf->Length() != aText.Length() )
		{
		// The Buffer exists but is a different size to the input text.
		iBuf = iBuf->ReAllocL(aText.Length());
		}
	// Copy the contents of aText to iBuf.
	*iBuf = aText;
	}

TInt Order64Bit(const TInt64& aLeft, const TInt64& aRight)
     {
     if (aLeft<aRight)
         return -1;
    if (aLeft==aRight)
         return 0;
     return 1;
     }

/**
A class to demonstrate the CArrayFix::Find() function using
a TKey derived class. An array of objects of this class is
constructed and the TKeyTIntPtr class is used to search for an element
in this array.
*/
class TIntPtrElement
	{
public :
	TIntPtrElement();
	void SetIntPtr(TInt* aPtr);
public :
	TBuf<4> iData;
	/**
	A pointer to an integer. It is used as the key to find
	an element in the array of TIntPtrElement objects.
	*/
	TInt* iIntPtr;
	};

/**
Constructor.
*/
TIntPtrElement::TIntPtrElement()
	{
	_LIT(KTextBase,"BASE");
	iData = KTextBase;
	}

/**
Sets the pointer to an integer pointer.
@param aPtr The integer pointer.
*/
void TIntPtrElement::SetIntPtr(TInt* aPtr)
	{
	iIntPtr = aPtr;
	}
/**
The TKey derived class.
An object of this class is used to find an element in the CArrayFix array.
The TKeyArrayFix::At() function is overridden in the example as the key is
stored in a pointer variable. The overridden implementation return the data
held by the pointer.
*/
class TKeyTIntPtr : public TKeyArrayFix
	{
public:
	TKeyTIntPtr(TInt aOffset, TKeyCmpText aType);
	TKeyTIntPtr(TInt aOffset, TKeyCmpNumeric aType);
	TAny* At(TInt aIndex);
	};

/**
Constructor.
This overload is used to compare text keys
such as descriptors.
*/
TKeyTIntPtr::TKeyTIntPtr(TInt aOffset, TKeyCmpText aType):TKeyArrayFix(aOffset,aType)
	{
	}

/**
Constructor.
This overload is used to compare numeric keys.
*/
TKeyTIntPtr::TKeyTIntPtr(TInt aOffset, TKeyCmpNumeric aType):TKeyArrayFix(aOffset,aType)
	{
	}

/**
Gets a pointer to the key of a specified array element.
@param aIndex The index of the array element.
@return The pointer to the key of the TIntPtrElement array element
at the specified aIndex position.
 */
TAny* TKeyTIntPtr::At(TInt aIndex)
	{
	if (aIndex==KIndexPtr)
        {
        return((TUint8 *)iPtr+iKeyOffset);
        }
	// Get the address of the array element.
	TIntPtrElement* ptr = (TIntPtrElement *)(iBase->Ptr(aIndex*sizeof(TIntPtrElement)).Ptr()+iKeyOffset);
	// The TIntPtrElement::iIntPtr points to the key.
	// Return the key.
	return(ptr->iIntPtr);
	}

/**
Prints all elements of the array of TIntPtrElement objects.
@param aFixFlat The array of TIntPtrElement objects.
*/
void PrintFixIntPtrArray(CArrayFixFlat<TIntPtrElement>& aFixFlat)
	{
	_LIT(KMsgContentsIntPtr,"Contents of the array of TIntPtrElement objects.\n");
	console->Printf(KMsgContentsIntPtr);
	for (TInt forix = 0; forix < aFixFlat.Count(); forix++)
		{
		console->Printf(KMsgNewLine);
		_LIT(KMsgBuf,"Index: %d\n Descriptor:");
		console->Printf(KMsgBuf,forix);
		console->Printf(aFixFlat[forix].iData);
		_LIT(KMsgInt,"\t\tInteger Value: %d");
		console->Printf(KMsgInt,*(aFixFlat[forix].iIntPtr));
		}
	console->Printf(KMsgNewLine);
	}

/**
Prints the contents of a CArrayFixFlat array.
@param aFixFlat The array of TElement objects.
*/
void PrintFixArray(CArrayFixFlat<TElement>& aFixFlat)
	{
	for (TInt forix = 0; forix < aFixFlat.Count(); forix++)
		{
		console->Printf(KMsgNewLine);
		console->Printf(aFixFlat[forix].iData);
		}
	console->Printf(KMsgNewLine);
	}

/**
Prints the contents of a CArrayPtrFlat array.
@param aPtrFlat The array of pointers to TElement objects.
*/
void PrintPtrArray(CArrayPtrFlat<CElement>* aPtrFlat)
	{
	_LIT(KCommonFormat4,"\n%S");
	for (TInt forix = 0; forix < aPtrFlat->Count(); forix++)
		{
		console->Printf(KCommonFormat4,((*aPtrFlat)[forix])->iBuf);
		}
	}

/**
Implements the abstract TSwap class.
*/
class TMySwap : public TSwap
	{
public:
	TMySwap(CArrayFixFlat<TElement>* aArray);
	void Swap(TInt aLeft,TInt aRight);
private:
				// The Swap() function is defined for this array.
	CArrayFixFlat<TElement>* iArray;
	};

/**
Constructor.
*/
TMySwap::TMySwap(CArrayFixFlat<TElement>* aArray):iArray(aArray)
	{
	}

/**
Implementation of the abstract TSwap::Swap() function.
@param aLeft The index of the left element of the array to be swapped.
@param aRight The index of the right element of the array to be swapped.
*/
void TMySwap::Swap(TInt aLeft,TInt aRight)
	{
	TBuf<4> tbuf(iArray->At(aLeft).iData);
	iArray->At(aLeft).iData.Copy(iArray->At(aRight).iData);
	iArray->At(aRight).iData.Copy(tbuf);
	}

/**
This function demonstrates how to append elements into a CArrayPakFlat array
and search for an element using the TKeyArrayPak class.
@param aPakFlat The array on which the CArrayPak::Find() function is performed.
*/
void DemoPakArrayFindL(CArrayPak<TText>& aPakFlat)
	{
	// Append TText strings to the aPakFlat array.

	// The length of the string to be appended is increased by one
	// to insert the '\0' character at the end of the TText string.
	// This is needed as the '\0' character is absent in descriptors.
	aPakFlat.AppendL(*firstElement,(User::StringLength(firstElement)+1)*sizeof(TText));

	// Insert the second element into the array.
	aPakFlat.AppendL(*secondElement,(User::StringLength(secondElement)+1)*sizeof(TText));

	// Insert the third element into the array.
	aPakFlat.AppendL(*thirdElement,(User::StringLength(thirdElement)+1)*sizeof(TText));

	// Print the array elements.
	_LIT(KCommonFormat7,"Array Contents");
	console->Printf(KCommonFormat7);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	TInt length = aPakFlat.Count();
	for(TInt ix = 0; ix < length; ix++)
		{
		console->Printf(TPtrC(&aPakFlat[ix]));
		console->Printf(KMsgNewLine);
		}

	// Find an element in the array.
	_LIT(KMsgFind,"\n--->Find");
	console->Printf(KMsgFind);
	_LIT(KMsgFindElement,"\nSearching the array for:");
	console->Printf(KMsgFindElement);

	// The descriptor, buf contains the string "Code".
	// The CArrayPak::Find() function will try to find this string in the array.
	console->Printf(TPtrC(thirdElement));

	// The key to find the element.
	// The length of the key is set to 12.
	// Thus, first 12 characters of all strings are compared with the key.
	TKeyArrayPak keypak(0,ECmpNormal,12);

	// Initialise the index to -1.
	TInt index = -1;

	// Perform the search.
	if(aPakFlat.Find(*thirdElement,keypak,index) == 0)
		{
		// Successful search.
		// The index variable holds the index of the element found.
		_LIT(KElementFound,"\nElement found at %d\n");
		console->Printf(KElementFound,index);
		}
	else
		{
		// Unsuccessful search.
		// The element is not found.
		_LIT(KElementNotFound,"\nElement not found");
		console->Printf(KElementNotFound);
		}
	}

/**
This function demonstrates how to append elements into a CArrayVarFlat array
and sort the array using the TKeyArrayVar class.
@param aVarFlat The array on which the CArrayVar::Sort() function is performed.
*/
void DemoVarArraySortL(CArrayVar<TText>& aVarFlat)
	{
	// Append TText strings to the aVarFlat array.

	// The length of the string to be appended is increased by one
	// to insert the '\0' character at the end of the TText string.
	// This is needed as the '\0' character is absent in descriptors.
	aVarFlat.AppendL(*firstElement,(User::StringLength(firstElement)+1)*sizeof(TText));

	// Insert the second and third elements into the array.
	aVarFlat.AppendL(*secondElement,(User::StringLength(secondElement)+1)*sizeof(TText));
	aVarFlat.AppendL(*thirdElement,(User::StringLength(thirdElement)+1)*sizeof(TText));

	// Print the array elements.
	_LIT(KCommonFormat6,"Array Contents [Before Sort]");
	console->Printf(KCommonFormat6);

	console->Printf(KMsgPressAnyKey);
	console->Getch();

	TInt length = aVarFlat.Count();
	for(TInt ix = 0; ix < length; ix++)
		{
		console->Printf(TPtrC(&aVarFlat[ix]));
		console->Printf(KMsgNewLine);
		}

	// The key to find the element.
	// The length of the key is set to 12.
	// Thus, first 12 characters of the string are compared with the key.
	TKeyArrayVar keyvar(0,ECmpFolded,12);

	// Sort the array using the TKeyArrayVar key.
	aVarFlat.Sort(keyvar);

	// Print the sorted array elements.
	_LIT(KMsgSort,"\n--->Sort");
	console->Printf(KMsgSort);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	for(TInt ix = 0; ix < length; ix++)
		{
		console->Printf(TPtrC(&aVarFlat[ix]));
		console->Printf(KMsgNewLine);
		}
	}
/**
This function demonstrates the usage of the CArrayFixFlat array.
@param aFixFlat The array of TElement objects.
*/
void DemoFlatArrayL(CArrayFixFlat<TElement>& aFixFlat)
	{
	TElement theElement;

	TInt value;
	for (value = 0; value < 3; value++)
		{
		theElement.iData.Format(KFormat1,value);
		aFixFlat.AppendL(theElement);
		}

				// Show length of each element.
				// The length of each element of the array is the size of the
				// TElement class.
	_LIT(KCommonFormat2,"\nLength()=%d");
	console->Printf(KCommonFormat2,aFixFlat.Length());

				// Show number of elements.
	_LIT(KCommonFormat3,"\nCount()=%d");
	console->Printf(KCommonFormat3,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);
				//
				// InsertL  * * * * * * * * *
				//
	_LIT(KMsgInsert,"\n\n--->InsertL");
	console->Printf(KMsgInsert);
	_LIT(KMsgPressAnyKey," (press any key to continue)\n");
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Insert elements
				// ... at the beginning,
	_LIT(KMsgBEG,"BEG");
	theElement.iData = KMsgBEG;
	aFixFlat.InsertL(0,theElement);

				// ... near the middle,
	_LIT(KMsgMID,"MID");
	theElement.iData = KMsgMID;
	aFixFlat.InsertL(2,theElement);

				// ... at the end.
				// This is the same as using the AppendL() function
	_LIT(KMsgEND,"END");
	theElement.iData = KMsgEND;
	aFixFlat.InsertL(aFixFlat.Count(),theElement);

				// Show number of elements
	_LIT(KCommonFormat5,"Count()=%d");
	console->Printf(KCommonFormat5,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);

				// Define the key to sort the array.
				// The data member of the TElement class, iData, is the key.
	TKeyArrayFix keyfix(_FOFF(TElement,iData),ECmpFolded);

				// Sort the array.
	aFixFlat.Sort(keyfix);

				// Print the sorted array.
	_LIT(KMsgSort,"\n--->Sort");
	console->Printf(KMsgSort);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	PrintFixArray(aFixFlat);

				//
		 		// Delete  * * * * * * * * *
				//
	_LIT(KMsgDelete,"\n\n--->Delete");
	console->Printf(KMsgDelete);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Delete the 2nd, 4th and 5th elements. Note:
				//
				//  1. We use position to identify the elements (i.e. offset)
				//  2. As elements are deleted, the position of other
				//	 elements changes.

	aFixFlat.Delete(1);		// delete the 2nd
	aFixFlat.Delete(2,2);   // delete what are now the 3rd and 4th

	aFixFlat.Compress();	// compress the array

				// Show number of elements
	console->Printf(KCommonFormat5,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);
				//
		 		// Reset  * * * * * * * * *
				//
	_LIT(KMsgReset,"\n\n--->Reset");
	console->Printf(KMsgReset);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// reset the array (i.e. empty it)
	aFixFlat.Reset();

				// Show number of elements
	console->Printf(KCommonFormat5,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);

				//
		 		// ExpandL & ExtendL  * * * * * * * * *
				//
	_LIT(KMsgExpandExtend,"\n\n--->ExpandL & ExtendL");
	console->Printf(KMsgExpandExtend);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// re-build the array
	_LIT(KFormat6,"%u");
	for (value = 0; value < 3; value++)
		{
		theElement.iData.Format(KFormat6,value);
		aFixFlat.AppendL(theElement);
		}
				// Expand by constructing a new element at position 1.
				// Extend the array.
				//
				// Note the use of the TElement default constructor
				// in both cases
	aFixFlat.ExpandL(1);
	aFixFlat.ExtendL();

				// Show number of elements
	console->Printf(KCommonFormat5,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);

				//
		 		// ResizeL * * * * * * * * * * * *
				//
	_LIT(KMsgResize,"\n\n--->ResizeL");
	console->Printf(KMsgResize);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Resize the array so that it only contains
				// one element
	aFixFlat.ResizeL(1);

				// Resize so that it contains 3 elements.
				// The two new elements are bit-wise copies
				// of a TElement object constructed using
				// its default constructor.
	aFixFlat.ResizeL(3);

				// Resize so that it contains 5 elements.
				// The two new elements are bit-wise copies of
				// the TElement object passed through
				// the reference.
	_LIT(KMsgXXX,"XXX");
	theElement.iData = KMsgXXX;
	aFixFlat.ResizeL(5,theElement);

				// Show number of elements
	console->Printf(KCommonFormat5,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);

				//
		 		// SetReserveL * * * * * * * * * * * *
				//
	_LIT(KMsgSetReserve,"\n\n--->SetReserveL");
	console->Printf(KMsgSetReserve);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Reserve sufficient space to append another
				// five elements.
				// This function may leave if there is
				// insufficient memory.
				// NOTE: this does NOT increase the number of
				//	   elements in the array.
	aFixFlat.SetReserveL(5);

				// We can now append five elements and be sure that
				// no leave will occur.
	_LIT(KMsgDoubleAtoE,"AABBCCDDEE");
	TBufC<10> source(KMsgDoubleAtoE);
	TInt forix;
	for (forix = 0; forix < source.Length(); forix+=2)
		{
		theElement.iData = source.Mid(forix,2);
		aFixFlat.AppendL(theElement);
		}

				// Show number of elements
	console->Printf(KCommonFormat5,aFixFlat.Count());

				// Show each element
	PrintFixArray(aFixFlat);

				// An object of the TMySwap class.
				// It is used to swap two elements of the array.
	TMySwap swap(&aFixFlat);

				// Swap the first and the last element of the array.
	swap.Swap(0,(aFixFlat.Count() - 1));

				// Print the array.
	_LIT(KMsgSwap,"\n--->Swap [First and Last element]");
	console->Printf(KMsgSwap);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	PrintFixArray(aFixFlat);
	console->Printf(KMsgPressAnyKey);
	console->Getch();
	}

/**
This function demonstrates the usage of the CArrayPtrFlat array.
@param aPtrFlat The array of pointers to CElement objects.
*/
void DemoPtrArrayL(CArrayPtrFlat<CElement>& aPtrFlat)
	{
	CElement*  ptr;
	ptr = new (ELeave) CElement;
	_LIT(KMsgFirstElement,"First Element");
	ptr->SetTextL(KMsgFirstElement);
	aPtrFlat.AppendL(ptr);


	ptr = new (ELeave) CElement;
	_LIT(KMsgSecondElement,"Second Element");
	ptr->SetTextL(KMsgSecondElement);
	aPtrFlat.AppendL(ptr);


	ptr = new (ELeave) CElement;
	_LIT(KMsgThirdElement,"Third Element");
	ptr->SetTextL(KMsgThirdElement);
	aPtrFlat.AppendL(ptr);


				// Show length of each element
	_LIT(KCommonFormat2,"\nLength()=%d");
	console->Printf(KCommonFormat2,aPtrFlat.Length());

				// Show number of elements
	_LIT(KCommonFormat3,"\nCount()=%d");
	console->Printf(KCommonFormat3,aPtrFlat.Count());

				// Show each element
	PrintPtrArray(&aPtrFlat);
				//
		 		// InsertL  * * * * * * * * *
				//
	_LIT(KMsgInsert,"\n\n--->InsertL");
	console->Printf(KMsgInsert);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Insert an element at the beginning
				// of the array.

	ptr = new (ELeave) CElement;
	_LIT(KMsgInsertedBeg,"Inserted @ beginning Element");
	ptr->SetTextL(KMsgInsertedBeg);
	aPtrFlat.InsertL(0,ptr);

				// Show number of elements
	_LIT(KCommonFormat5,"Count()=%d");
	console->Printf(KCommonFormat5,aPtrFlat.Count());

				// Show each element
	PrintPtrArray(&aPtrFlat);

				//
		 		// Delete  * * * * * * * * *
				//
	_LIT(KMsgDelete,"\n\n--->Delete");
	console->Printf(KMsgDelete);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Delete the last two elements from the array BUT
				// first we must get a reference to those elements
				// (pointers to CElement objects) otherwise
				// the CElement objects are orphaned.
				//
				// Here, we destroy those CElement objects.
				//
				// There are two alternative ways of indexing into
				// the array, using either At() or the [] operator

				// NOTE that the code below could be compressed to:
				//
				// delete (*ptrflat)[aPtrFlat.Count()-1];
				// delete (*ptrflat)[aPtrFlat.Count()-2];
				// aPtrFlat.Delete(aPtrFlat.Count()-2,2);

	TInt index = -1;
	index = aPtrFlat.Count();

	ptr = aPtrFlat[--index];
	aPtrFlat.Delete(index);
	delete ptr;

	ptr = aPtrFlat.At(--index);
	aPtrFlat.Delete(index);
	delete ptr;

				// Show number of elements
	console->Printf(KCommonFormat5,aPtrFlat.Count());

				// Show each element
	PrintPtrArray(&aPtrFlat);

				//
		 		// At & the [] operator * * * * * * * * * * * *
				//
	_LIT(KMsgAt,"\n\n--->At() and the operator []");
	console->Printf(KMsgAt);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Make a change to the object pointed to by the first element in the array
	_LIT(KMsgNewTextFirst,"New text for the first CElement");
	_LIT(KMsgNewTextSecond,"New text for the second CElement");
	aPtrFlat[0]->SetTextL(KMsgNewTextFirst);
	aPtrFlat.At(1)->SetTextL(KMsgNewTextSecond);

				// Show number of elements
	console->Printf(KCommonFormat5,aPtrFlat.Count());

				// Show each element
	PrintPtrArray(&aPtrFlat);

				//
		 		// ResetAndDestroy  * * * * * * * * *
				//
	_LIT(KMsgResetDestroy,"\n\n--->ResetAndDestroy");
	console->Printf(KMsgResetDestroy);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// destroy all of the CElement objects and reset the
				// array.
	aPtrFlat.ResetAndDestroy();

				// Show number of elements
	console->Printf(KCommonFormat3,aPtrFlat.Count());

				// Show each element
	PrintPtrArray(&aPtrFlat);

	console->Printf(KMsgNewLine);
	}

/**
This function demonstrates the use the TKey derived, TKeyTIntPtr class.
The key is used to find an element in the CArrayFixFlat<TIntPtrElement> array.
@param aFixFlat The array of TIntPtrElement objects.
*/
void DemoTKeyDerivedL(CArrayFixFlat<TIntPtrElement>& aFixFlat)
	{
	TIntPtrElement theIntPtrElement;

	// Create pointers to integers.
	TInt* ptrInt1 = new(ELeave)TInt(10);
	CleanupStack::PushL(ptrInt1);
	TInt* ptrInt2 = new(ELeave)TInt(0);
	CleanupStack::PushL(ptrInt2);
	TInt* ptrInt3 = new(ELeave)TInt(5);
	CleanupStack::PushL(ptrInt3);

	// Create an array of pointers.
	TInt* ptrInt[3] = {ptrInt1,ptrInt2,ptrInt3};

	// Append elements into the ptrElementArray array.
	TInt value;
	for (value = 0; value < 3; value++)
		{
		theIntPtrElement.iData.Format(KFormat1,value);
		theIntPtrElement.SetIntPtr(ptrInt[value]);
		aFixFlat.AppendL(theIntPtrElement);
		}

	// Print the contents of the array.
	PrintFixIntPtrArray(aFixFlat);

				// Define the key to find an element in the array.
				// Use the TKey derived class, TKeyTIntPtr to do this.
	TKeyTIntPtr keyIntPtr(_FOFF(TIntPtrElement,iIntPtr),ECmpTInt);

	_LIT(KMsgFind,"\n--->Find");
	console->Printf(KMsgFind);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	// Initialise to -1.
	// Its value will be set to 0 if the Search is successful.
	TInt index = -1;

	// Perform the search.
	TInt res = aFixFlat.Find	(
								theIntPtrElement, // The object whose
								// key is used for comparison.
								keyIntPtr, // The key object that
								// defines the property of the key.
								index
								);

	if(res == 0)
		{
		// The index variable holds the index of the element found.
		_LIT(KElementFound,"\nElement found at index: %d\n");
		console->Printf(KElementFound,index);
		}

	else
		{
		// The element is not found.
		_LIT(KElementNotFound,"\nElement not found");
		console->Printf(KElementNotFound);
		}

	CleanupStack::PopAndDestroy(3, ptrInt1); // ptrInt3, ptrInt2 and ptrInt1.
	}

/**
This function shows the use of the RArray class.
@param aArray The array of CArrayElement objects.
*/
void DemoRArrayL(RArray<CArrayElement>& aArray)
	{
	CArrayElement arrayElement;

			//
	 		// AppendL  * * * * * * * * *
			//
	_LIT(KMsgAppend,"\n\n--->AppendL");
	console->Printf(KMsgAppend);

	console->Printf(KMsgPressAnyKey);
	console->Getch();

	TInt value;
	for (value = 0; value < 3; value++)
		{
		// Formats and copies text into this descriptor, replacing any existing data.
		arrayElement.iData.Format(KFormat1,value);
		aArray.AppendL(arrayElement); // leaves with one of the system wide error codes, if the operation fails.
		}

				// Show number of elements
	_LIT(KCommonFormat3,"\nCount()=%d");
	console->Printf(KCommonFormat3,aArray.Count());

				// Show each element
	TInt forix;
	_LIT(KCommonFormat4,"\n%S");
	for (forix = 0; forix < aArray.Count(); forix++)
		{
		console->Printf(KCommonFormat4,&(aArray)[forix].iData);
		}

			//
	 		// InsertL  * * * * * * * * *
			//
	_LIT(KMsgInsert,"\n\n--->InsertL");
	console->Printf(KMsgInsert);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Inserts objects into the array at a specified position.
				// It increases the size of the dynamic array.
				// This function leaves with one of the system error codes, if the operation fails.

				// Insert arrayElements
				// ... at the beginning,
	_LIT(KMsgBEG,"BEG");
	arrayElement.iData = KMsgBEG;
	aArray.InsertL(arrayElement, 0);

				// ... near the middle,
	_LIT(KMsgMID,"MID");
	arrayElement.iData = KMsgMID;
	aArray.InsertL(arrayElement, 2);

				// ... at the end.
				// This is the same as using the AppendL() function
	_LIT(KMsgEND,"END");
	arrayElement.iData = KMsgEND;
	aArray.InsertL(arrayElement,aArray.Count());

				// Show number of elements
	_LIT(KCommonFormat5,"Count()=%d");
	console->Printf(KCommonFormat5,aArray.Count());

			//
	 		// Access elements  * * * * * * * * *
			//
	_LIT(KMsgAccess,"\n\n--->Access");
	console->Printf(KMsgAccess);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Show each element
	for (forix = 0; forix < aArray.Count(); forix++)
		console->Printf(KCommonFormat4,&(aArray)[forix].iData);

			//
	 		// Delete  * * * * * * * * *
			//
	_LIT(KMsgDelete,"\n\n--->Delete");
	console->Printf(KMsgDelete);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	TInt index2;
	index2 = aArray.Count();

			// Removes the objects at a specified position from the array.
			// It shrinks the array.

			// Delete the last element
	arrayElement = (aArray)[--index2];
	aArray.Remove(index2);

		    // Show number of elements
	console->Printf(KCommonFormat5,aArray.Count());

			// Show each element
	for (forix = 0; forix < aArray.Count(); forix++)
		console->Printf(KCommonFormat4,&(aArray)[forix].iData);

			//
	 		// Find  * * * * * * * * *
			//
	_LIT(KMsgFind,"\n\n--->Find in various ways");
	console->Printf(KMsgFind);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	TInt pos =0;

			// Find a particular element in different ways.
	arrayElement = (aArray)[--index2];

	_LIT(KCommonFormat6,"\nElement %S is found at position %d");

	// Finds the first object in the array which matches the specified object using a sequential search and a matching algorithm.
	pos=aArray.Find(arrayElement, TIdentityRelation<CArrayElement>());
			console->Printf(KCommonFormat6, &(aArray)[pos].iData, pos);

	// Finds the last object in the array which matches the specified object using a sequential search and a matching algorithm.
	pos=aArray.FindReverse(arrayElement, TIdentityRelation<CArrayElement>());
			console->Printf(KCommonFormat6, &(aArray)[pos].iData, pos);

	// Finds the object in the array which matches the specified object using a binary search technique.It assumes that objects are in signed key order
	pos=aArray.FindInSignedKeyOrder(arrayElement);
				console->Printf(KCommonFormat6,&(aArray)[pos].iData, pos);

	//	Finds the object in the array which matches the specified object using a binary search technique. It assumes that objects are in unsigned key order.
	pos=aArray.FindInUnsignedKeyOrder(arrayElement);
				console->Printf(KCommonFormat6,&(aArray)[pos].iData, pos);

	// Finds the object in the array which matches the specified object using a binary search technique.In case of more than one matching element, finds last
	// match as specified in the second parameter.It assumes that objects are in signed key order
	pos=aArray.SpecificFindInSignedKeyOrder(arrayElement, EArrayFindMode_Last);
			console->Printf(KCommonFormat6,&(aArray)[pos].iData, pos);

	// Finds the object in the array which matches the specified object using a binary search technique.In case of more than one matching element, finds last
	// match as specified in the second parameter.It assumes that objects are in unsigned key order
	pos=aArray.SpecificFindInUnsignedKeyOrder(arrayElement, EArrayFindMode_Last);
				console->Printf(KCommonFormat6,&(aArray)[pos].iData, pos);

			// Closes the array and frees all memory allocated to the array
	aArray.Close();
	}

void DemoRArraySort()
	{
			//
	 		// Sort  * * * * * * * * *
			//
	_LIT(KSortFind,"\n\n--->Sort");
	console->Printf(KSortFind);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	TLinearOrder<TInt64> Int64Order(Order64Bit);
	TInt count=10;

			// Construct two arrays of 64-bit signed integer type objects.
			// Append random objects to first array.
			// Sort all objects of first array in the second array

	RArray<TInt64> firstArray(count);
    RArray<TInt64> secondArray(count);

    for (TInt j=0; j<count; j++)
		{
        TInt64 x=(TInt64)(((TUint)Math::Random() %count));

        firstArray.Append(x);
        secondArray.InsertInOrderAllowRepeats(x,Int64Order);
		}

	// Show the initial array elements of first array.
	_LIT(KInitialArrayElements,"\nInitial array elements are:");
	console->Printf(KInitialArrayElements);

	_LIT(KCommonFormat7," %d");
	TInt forix;
	for (forix = 0; forix < count; forix++)
		console->Printf(KCommonFormat7,firstArray[forix]);

	// Sorts the objects within the array using the specified TLinearOrder
	secondArray.Sort(Int64Order);

	// Show sorted array elements of second array
	_LIT(KSortedArrayElements,"\nSorted array elements are:");
	console->Printf(KSortedArrayElements);
	for (forix = 0; forix < count; forix++)
		console->Printf(KCommonFormat7,secondArray[forix]);

	console->Getch();
	console->Printf(KMsgNewLine);

		// Closes both the arrays and frees all the memory allocated to these arrays
    firstArray.Close();
	secondArray.Close();
	}

void DemoRPointerArrayL(RPointerArray<CElement>& aPointerArray)
	{
	CElement*  elementptr;

   			//
	 		// AppendL  * * * * * * * * *
			//
	_LIT(KMsgAppend,"\n\n--->AppendL");
   	console->Printf(KMsgAppend);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

			// Append the first element.
	elementptr = new (ELeave) CElement;
	_LIT(KMsgFirstElement,"First Element");
	elementptr->SetTextL(KMsgFirstElement);
	aPointerArray.AppendL(elementptr);

			// Append the second element.
	elementptr = new (ELeave) CElement;
	_LIT(KMsgSecondElement,"Second Element");
	elementptr->SetTextL(KMsgSecondElement);
	aPointerArray.AppendL(elementptr);

			// Append the third element.
	elementptr = new (ELeave) CElement;
	_LIT(KMsgThirdElement,"Third Element");
	elementptr->SetTextL(KMsgThirdElement);
	aPointerArray.AppendL(elementptr);

				// Show number of elements
	_LIT(KCommonFormat3,"\nCount()=%d");
	console->Printf(KCommonFormat3,aPointerArray.Count());

				// Show each element
	_LIT(KCommonFormat4,"\n%S");
	TInt forix;
	for (forix = 0; forix < aPointerArray.Count(); forix++)
		{
		console->Printf(KCommonFormat4,aPointerArray[forix]->iBuf);
		}

			//
	 		// InsertL  * * * * * * * * *
			//
	_LIT(KMsgInsert,"\n\n--->InsertL");
	console->Printf(KMsgInsert);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Inserts objects into the array at a specified position.
				// It increases the size of the dynamic array.

				// Insert an element at the beginning
				// of the array
	elementptr = new (ELeave) CElement;
	_LIT(KMsgInsertedBeg,"Inserted @ beginning Element");
	elementptr->SetTextL(KMsgInsertedBeg);
	aPointerArray.InsertL(elementptr, 0);

				// Show number of elements
	_LIT(KCommonFormat5,"Count()=%d");
	console->Printf(KCommonFormat5,aPointerArray.Count());

			//
	 		// Access  * * * * * * * * *
			//
			// Show each element
	for (forix = 0; forix < aPointerArray.Count(); forix++)
		{
		console->Printf(KCommonFormat4,aPointerArray[forix]->iBuf);
		}

			//
	 		// Delete  * * * * * * * * *
			//
	_LIT(KMsgDelete,"\n\n--->Delete");
	console->Printf(KMsgDelete);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

			// Delete the last elements from the array BUT
			// first we must get a reference to the element
			// (pointers to CElement objects) otherwise
			// the CElement objects are orphaned.
			//
			// Here, we destroy the CElement object. It shrinks the array.
			//
			// Use [] operator for indexing in to the array.
	TInt index1;
	index1 = aPointerArray.Count();

	elementptr = aPointerArray[--index1];
	aPointerArray.Remove(index1);
	delete elementptr;

				// Show number of elements
	console->Printf(KCommonFormat5,aPointerArray.Count());

				// Show each element
	for (forix = 0; forix < aPointerArray.Count(); forix++)
		{
		console->Printf(KCommonFormat4,aPointerArray[forix]->iBuf);
		}

			//
	 		// Find  * * * * * * * * *
			//
	_LIT(KMsgFind,"\n--->Find");
	console->Printf(KMsgFind);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	elementptr = aPointerArray[index1-2];

	_LIT(KCommonFormat8,"\nElement found at position %d");

	// Finds the first object pointer in the array which matches the specified object pointer, using a sequential search.
	TInt result=aPointerArray.Find(elementptr);
		console->Printf(KCommonFormat8, result);

	// Finds the last object pointer in the array which matches the specified object pointer, using a sequential search.
	result=aPointerArray.FindReverse(elementptr);
		console->Printf(KCommonFormat8, result);

	// Finds the object pointer in the array that matches the specified object pointer, using a binary search technique.
	// It assumes that object pointers in the array are in address order.
	result=aPointerArray.FindInAddressOrder(elementptr);
		console->Printf(KCommonFormat8, result);

			//
	 		// Sort  * * * * * * * * *
			//
	_LIT(KSortFind,"\n\n--->Sort");
	console->Printf(KSortFind);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

			// ResetAndDestroy  * * * * * * * * *
			//
	_LIT(KMsgResetDestroy,"\n\n--->ResetAndDestroy");
	console->Printf(KMsgResetDestroy);
	console->Printf(KMsgPressAnyKey);
	console->Getch();

				// Empties the array and deletes the referenced objects.
				// It frees all memory allocated to the array and resets the internal state so that it is ready to be reused.
	aPointerArray.ResetAndDestroy();
	}

void DemoSortRPointerArray()
	{
			// Construct two arrays of TAny pointer objects.
			// Append random objects to first pointer array.
			// Sort all objects of first array in the second pointer array.
	RPointerArray<TAny> firstpointerarray;
	RPointerArray<TAny> secondpointerarray;

	TInt count = 10;
	for (TInt i=0; i<count; i++)
		{
		TAny* x=(TAny*)(((TUint)Math::Random() %count));
		firstpointerarray.Append(x);
		secondpointerarray.InsertInAddressOrderAllowRepeats(x);
		}

	// Show initial array elements of first array
	_LIT(KInitialArrayElements,"\nInitial array elements are:");
	console->Printf(KInitialArrayElements);

	_LIT(KCommonFormat7,"Array Contents %d \n");
	TInt forix;
	for (forix = 0; forix < count; forix++)
		{
		console->Printf(KCommonFormat7,firstpointerarray[forix]);
		}

	// Sorts the object pointers within the array into address order
	secondpointerarray.SortIntoAddressOrder();

	// Show sorted array elements of second array
	_LIT(KSortedArrayElements,"\nSorted array elements are:");
	console->Printf(KSortedArrayElements);
	for (forix = 0; forix < count; forix++)
		{
		console->Printf(KCommonFormat7,secondpointerarray[forix]);
		}

	// Closes the arrays and frees all memory allocated to it
	firstpointerarray.Close();
	secondpointerarray.Close();
	}
// Do the example
LOCAL_C void doExampleL()
	{
	//************************************************************
	// Demonstrate a general fixed length array using
	// a flat buffer
	//************************************************************

	CArrayFixFlat<TElement>* fixflat;
				// Construct array of TElement objects where the iData
				// data member of each element contains "X0", "X1", etc
				// Uses the AppendL() function to add the members
				// and the [] operator to access elements.
	fixflat = new (ELeave) CArrayFixFlat<TElement>(3);
	CleanupStack::PushL(fixflat);

				// Demonstrate the CArrayFixFlat array.
	DemoFlatArrayL(*fixflat);

				// Delete the fixflat array.
	CleanupStack::PopAndDestroy(fixflat); // fixflat
	//************************************************************
	// Demonstrate an array of pointers to CBase
	// derived objects. Uses the specialised array CArrayPtrFlat
	//************************************************************
	_LIT(KMsgArrayOfPointers,"\nARRAYS OF POINTERS (to CBase derived objects)\n");
	console->Printf(KMsgArrayOfPointers);

	CArrayPtrFlat<CElement>* ptrflat;

				// Construct an array of four CElement objects each
				// containing the text "First" "second" etc
				// Uses the AppendL() function to add the members
				// and the [] operator to access elements.
	ptrflat = new (ELeave) CArrayPtrFlat<CElement>(16);
	CleanupStack::PushL(ptrflat);

				// Demonstrate the CArrayPtrFlat array.
	DemoPtrArrayL(*ptrflat);

				// Delete the array.
	CleanupStack::PopAndDestroy(ptrflat); // ptrflat

	//************************************************************
	// Demonstrate Variable and Packed arrays
	// Use TKey to sort these arrays
	//************************************************************
	_LIT(KMsgVarArray,"\n\nVAR ARRAY\n\n");
	console->Printf(KMsgVarArray);

				// Create a Var array.
	CArrayVarFlat<TText>* varflat = new (ELeave) CArrayVarFlat<TText>(3);
	CleanupStack::PushL(varflat);

				// Demonstrate the CArrayVarFlat array.
	DemoVarArraySortL(*varflat);

				// Delete the array.
	CleanupStack::PopAndDestroy(varflat); // varflat

	_LIT(KMsgPakArray,"\n\nPAK ARRAY\n\n");
	console->Printf(KMsgPakArray);

				// Create a TKeyArrayPak array.
	CArrayPakFlat<TText>* pakflat = new (ELeave) CArrayPakFlat<TText>(3);
	CleanupStack::PushL(pakflat);

				// Demonstrate the CArrayPakFlat array.
	DemoPakArrayFindL(*pakflat);

				// Delete the array.
	CleanupStack::PopAndDestroy(pakflat);  // pakflat,

	//************************************************************
	// Demonstrate the CArrayFix::Find() function using the
	// TKey derived class TKeyTIntPtr.
	//************************************************************
	_LIT(KMsgTKeyDerived,"\nTKEY DERIVED CLASS\n");
	console->Printf(KMsgPressAnyKey);
	console->Getch();

	console->Printf(KMsgTKeyDerived);
	CArrayFixFlat<TIntPtrElement>* ptrElementArray;
					// Construct array of TIntPtrElement objects where the iData
					// data member of each element contains "X0", "X1", etc
					// Uses the AppendL() function to add the members
					// and the [] operator to access elements.
	ptrElementArray = new (ELeave) CArrayFixFlat<TIntPtrElement>(3);
	CleanupStack::PushL(ptrElementArray);

					// Demonstrate the use of TKey derived class.
	DemoTKeyDerivedL(*ptrElementArray);

	CleanupStack::PopAndDestroy(ptrElementArray); // ptrElementArray,

	//*****************************************************************
	// Demonstrate an array of type RArray for elements of the same size
	//*****************************************************************

	_LIT(KMsgForRArray,"\n*******Arrays of fixed length objects******");
	console->Printf(KMsgForRArray);

	RArray<CArrayElement>* array;

				// Construct array of CArrayElement objects where the iData
				// data member of each element contains "X0", "X1", etc.
				// The elements of the array are instances of a class,
				// so this is a template class. Refer to the definition of
				// element CArrayElement above.
				// Uses the AppendL() function to add the members
				// and the [] operator to access elements.
    array = new (ELeave) RArray<CArrayElement>(16);	// Initialise the array granularity to 16
	CleanupStack::PushL(array);
				// Demonstrate the use of RArray.
    DemoRArrayL(*array);

	CleanupStack::PopAndDestroy(array);
				// Demonstrates sorting of the RArray.
	DemoRArraySort();


	//******************************************************************************
	// Demonstrate a simple and efficient array of pointers to objects RPointerArray
	//******************************************************************************

	_LIT(KMsgForRPointerArray,"\n*******Array of pointers to objects******");
	console->Printf(KMsgForRPointerArray);

	RPointerArray<CElement>* ptrarray;

				// Construct an array of four CElement objects each
				// containing the text "First", "Second" etc
				// Uses the AppendL() function to add the members
				// and the [] operator to access elements.
				// It leaves with one of the system wide error codes, if the operation fails.
    ptrarray = new (ELeave) RPointerArray<CElement>(16); // Initialise the array granularity to 16
    CleanupStack::PushL(array);
				// Demonstrate the use of RPointerArray.
    DemoRPointerArrayL(*ptrarray);
	CleanupStack::PopAndDestroy(ptrarray);
				// Demonstrates sorting of the RPointerArray.
	DemoSortRPointerArray();
	}

