// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//

#include <d32dbms.h>  //Used for RDbDatabase class. 
#include "SmpExample.h"
_LIT(KDatabase1, "C:\\DBforSMP1.db");
_LIT(KDatabase2, "C:\\DBforSMP2.db");

/**
Destructor
*/
CSmpExample::~CSmpExample()
    {
    iReadThread.Close(); //Reader thread closed. 
    iWriteThread1.Close(); //Writer thread closed. 
    iWriteThread2.Close(); //Writer thread closed.
    delete iConsole;
    }

/**
NewL()
*/
CSmpExample* CSmpExample::NewL()
    {
    CSmpExample* self = new(ELeave)CSmpExample();
    CleanupStack::PushL(self);
    self->ConstructL();
    CleanupStack::Pop(self);
    return self;
    }

/**
Creates three threads; one for reading from the databases and the other two for writing to the databases. 
*/
void CSmpExample::ConstructL()
    { 
    CreateDatabaseL(KDatabase1);
    CreateDatabaseL(KDatabase2);
    
    _LIT(KTextConsoleTitle, "SmpExample");
    iConsole = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen, KConsFullScreen));  

    //Creates a reader thread to read the databases.
    _LIT(KReader, "ReaderThread");
    User::LeaveIfError(iReadThread.Create(KReader, ReadThreadFuncL, KDefaultStackSize, KMinHeapSize, 256*KMinHeapSize, this, EOwnerProcess));   

    //Creates a writer thread to write to the DBforSMP1.db database. 
    _LIT(KWriter1, "WriterThread1");
    User::LeaveIfError(iWriteThread1.Create(KWriter1, WriteThread1FuncL, KDefaultStackSize, KMinHeapSize, 256*KMinHeapSize, this, EOwnerProcess));    
    
    //Creates a writer thread to write to the DBforSMP2.db database. 
    _LIT(KWriter2, "WriterThread2");
    User::LeaveIfError(iWriteThread2.Create(KWriter2, WriteThread2FuncL, KDefaultStackSize, KMinHeapSize, 256*KMinHeapSize, this, EOwnerProcess));           


    //Sets priority to the threads. 
    iWriteThread1.SetPriority(EPriorityMuchLess);
    iWriteThread2.SetPriority(EPriorityMore);
    iReadThread.SetPriority(EPriorityNormal);
    }

/**
Starts the three threads
*/
void CSmpExample::StartThreads()
    {   
    TRequestStatus writerThread1Status; 
    TRequestStatus readerThreadStatus; 
    TRequestStatus writerThread2Status; 
    
    _LIT(KTextPressKey, "\nPress any key to start writing to and reading from the database\n");
    iConsole->Printf(KTextPressKey);
    iConsole->Getch();
  
    //Requests a notification for the ReaderThread to terminate. 
    iReadThread.Logon(readerThreadStatus);
    iReadThread.Resume();
    
    //Requests a notification for the WriterThread1 to terminate. 
    iWriteThread1.Logon(writerThread1Status);   
    iWriteThread1.Resume();
    
    //Requests a notification for the WriterThread2 to terminate. 
    iWriteThread2.Logon(writerThread2Status);   
    iWriteThread2.Resume();
    
    //Control returns to the main thread when the all the threads terminate. 
    User::WaitForRequest(writerThread1Status);
    User::WaitForRequest(readerThreadStatus);
    User::WaitForRequest(writerThread2Status);
    }

/**
Prints welcome message and purpose of the example.
*/
void CSmpExample::PrintMessage()
    {
    _LIT(KTextWelcome, "Welcome to the SmpExample.\n");
    _LIT(KTextPurposel1, "There are three threads running in the example: WriterThread1 has the lowest priority,\n");
    _LIT(KTextPurposel2, "ReaderThread has normal priority and WriterThread2 has maximum priority.\n");  
    _LIT(KTextPurposel3, "The two WriterThreads open two different databases and write some integers to them.\n");
    _LIT(KTextPurposel4, "ReaderThread reads the two databases and prints the output to the console.\n");
    
    _LIT(KTextPurposel5, "In a unicore environment, WriterThread1 will be scheduled to run last and would not have\n");
    _LIT(KTextPurposel6, "written to the database when ReaderThread starts reading the databases.\n");
    _LIT(KTextPurposel7, "But in an SMP environment, both the WriterThreads write to their respective databases simultaneously.\n");
    
    iConsole->Printf(KTextWelcome);
    iConsole->Printf(KTextPurposel1);
    iConsole->Printf(KTextPurposel2);
    iConsole->Printf(KTextPurposel3);
    iConsole->Printf(KTextPurposel4);
    iConsole->Printf(KTextPurposel5);
    iConsole->Printf(KTextPurposel6);
    iConsole->Printf(KTextPurposel7);
    }

/**
Opens the database named in the first parameter, reads it and prints the output to the console.
*/
void CSmpExample::ReadDatabaseL(const TDesC& aDbName, CConsoleBase& console)
    { 
    //Creates a file server session object before any file system manipulation. 
    RFs fsSession;
    CleanupClosePushL(fsSession);
    fsSession.Connect();
    
    //Creates Rdbs object
    RDbs dbs;
    CleanupClosePushL(dbs); 
    dbs.Connect();
    
    //Opens the named database using the RDbs object
    RDbNamedDatabase database;
    CleanupClosePushL(database);
    User::LeaveIfError(database.Open(dbs, aDbName)); 
    
    //Locks the database.
    database.Begin();
    //Prepare an SQL query to read one row of numbers from the database. 
    _LIT(KSQLStatement, "Select Number1, Number2, Number3  from Numbers order by Number1, Number2, Number3");
    
    //Creates a view on the database to read it.
    RDbView view; 
    CleanupClosePushL(view);
    
    User::LeaveIfError(view.Prepare(database, TDbQuery(KSQLStatement, EDbCompareNormal)));
    User::LeaveIfError(view.EvaluateAll());
    
    _LIT(KTextReading, "Reading the database\t %S.\n");
    console.Printf(KTextReading, &aDbName);
    
    //Boolean variable to check whether the database is empty.
    TBool isDbEmpty= EFalse;
    //Iterates through the database till the last row is read. 
    for (view.FirstL(); view.AtRow(); view.NextL())
        {       
        view.GetL();
        TInt number1 = view.ColInt(1);
        TInt number2 = view.ColInt(2);
        TInt number3 = view.ColInt(3);
        //Prepare a row formatter to print numbers in the console. 
        _LIT(KRowFormatter, "Reading  %d \t%d\t%d\n");  
        //Reads the integers from the view and display them in the console.
        console.Printf(KRowFormatter, number1, number2, number3);
        isDbEmpty= ETrue;
        }
    
    if(isDbEmpty== EFalse)
        {
        _LIT(KTextDbEmpty, "Database is empty.\n\n");
        console.Printf(KTextDbEmpty);
        }
    else
        {
        _LIT(KTextAllRead, "All the numbers in the database have been read.\n\n");
        console.Printf(KTextAllRead);
        }            
    
    CleanupStack::PopAndDestroy(&view); 
    CleanupStack::PopAndDestroy(&database);  
    CleanupStack::PopAndDestroy(&dbs);    
    CleanupStack::PopAndDestroy(&fsSession);
    }

/**
Creates a console, reads the databases and prints output to the console. 
*/
void CSmpExample::ReadBothDatabasesL()
    {
    _LIT(KTextConsoleTitle, "ReaderThread");
    CConsoleBase* console= Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen, KConsFullScreen)); 
    CleanupStack::PushL(console);

    //Reads DBforSMP1.db and prints the output to the console. 
    ReadDatabaseL(KDatabase1, *console);
    //Reads DBforSMP2.db and prints the output to the console.
    ReadDatabaseL(KDatabase2, *console);
    
    _LIT(KExit, "Press any key to exit\n");
    console->Printf(KExit);
    console->Getch();
    CleanupStack::PopAndDestroy(console);    
    }

/**
This function is called when iReadThread is resumed. It creates a cleanup stack and 
calls ReadBothDatabasesL() to read both the databases. 
*/
TInt CSmpExample::ReadThreadFuncL(TAny* /*aPtr*/)
    {
    __UHEAP_MARK;
    //Creates cleanup stack.
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }
     
    TRAPD(error, ReadBothDatabasesL()); 
    if(error != KErrNone)
        {
        _LIT(KUserPanic, "DB Read Failed");  
        User::Panic(KUserPanic, error);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }

/**
Opens the database specified in the first parameter aDbName and writes consecutive integers to it
starting from the number specified in the second parameter aNum.   
*/
void CSmpExample::WriteDbFuncL(const TDesC& aDbName, TInt aNum)
    {
    //Creates a file server session object before any file system manipulation. 
    RFs fsSession;
    CleanupClosePushL(fsSession);
    fsSession.Connect();


    //Creates Rdbs object   
    RDbs dbs;
    CleanupClosePushL(dbs);
    dbs.Connect();
    
    //Opens the database using the RDbs object
    RDbNamedDatabase database;
    CleanupClosePushL(database);
    User::LeaveIfError(database.Open(dbs, aDbName)); 
    
    //Locks the database.
    database.Begin();
    
    //Creates a view on the database. 
    RDbView view;
    CleanupClosePushL(view);
    _LIT(KSQLStatement, "Select Number1, Number2, Number3 from Numbers order by Number1, Number2, Number3");
    User::LeaveIfError(view.Prepare(database, TDbQuery(KSQLStatement, EDbCompareNormal)));
    User::LeaveIfError(view.EvaluateAll());

    for(int i=0; i<5; i++)
        {
        //Inserts a new row at the end of the database. 
        view.InsertL();
        //Fills three columns with numbers.
        view.SetColL(1, aNum++);
        view.SetColL(2, aNum++);
        view.SetColL(3, aNum++);
        view.PutL();
        }

    CleanupStack::PopAndDestroy(&view);       
    //Commits the database and unlocks it. 
    database.Commit();
    CleanupStack::PopAndDestroy(&database);  
    CleanupStack::PopAndDestroy(&dbs);   
    CleanupStack::PopAndDestroy(&fsSession);
    }


/**
This function is called when iWriteThread is resumed. It creates a cleanup stack and 
calls WriteDbFuncL() to write integers to KDatabase2. 
*/
TInt CSmpExample::WriteThread2FuncL(TAny* /*aPtr*/)
    {
    __UHEAP_MARK;
    //Creates cleanup stack.
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }
    
    //First number to be written to the database. 
    const int KFirstNumforDb2 = 500; 
    TRAPD(err, WriteDbFuncL(KDatabase2, KFirstNumforDb2));
    if(err != KErrNone)
        {
        _LIT(KUserPanic, "DB Write Failed");  
        User::Panic(KUserPanic, err);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }

/**
This function is called when iWriteThread is resumed. It creates a cleanup stack and 
calls WriteDbFuncL() to write integers to KDatabase1. 
*/
TInt CSmpExample::WriteThread1FuncL(TAny* /*aPtr*/)
    {
    __UHEAP_MARK;
    //Creates cleanup stack.
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }

    //First number to be written to the database. 
    const int KFirstNumforDb1 = 0; 
    TRAPD(err, WriteDbFuncL(KDatabase1, KFirstNumforDb1));
    if(err != KErrNone)
        {
        _LIT(KUserPanic, "DB Write Failed");  
        User::Panic(KUserPanic, err);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }

/**
Creates a named database with the name specified in the parameter,
creates a table and adds columns to it.
*/
void CSmpExample::CreateDatabaseL(const TDesC& aDbName)
    {
    //Creates a file server session object before any file system manipulation. 
    RFs fsSession;
    CleanupClosePushL(fsSession);
    fsSession.Connect();

    //Creates Rdbs object
    RDbs dbs;
    CleanupClosePushL(dbs); 
    dbs.Connect();
    
    //Creates a new database with the name specified in the parameter aDbName, 
    //if database with same name is present, it will be replaced.    
    RDbNamedDatabase database;
    CleanupClosePushL(database);  
    User::LeaveIfError(database.Replace(fsSession, aDbName)); 
    database.Close(); 
    
    //Opens the database using the RDbs object
    User::LeaveIfError(database.Open(dbs, aDbName));       
    //Creates a table definition.
    CDbColSet* columns=CDbColSet::NewLC();

    //Adds three columns each containing Int32 values. 
    _LIT(KCol1, "Number1");
    _LIT(KCol2, "Number2");
    _LIT(KCol3, "Number3");   
    columns->AddL(TDbCol(KCol1, EDbColInt32));
    columns->AddL(TDbCol(KCol2, EDbColInt32));
    columns->AddL(TDbCol(KCol3, EDbColInt32));
 

     //Creates the table, table name is "Numbers" and add the columns to it.
    _LIT(KTable, "Numbers");
    User::LeaveIfError(database.CreateTable(KTable, *columns));
    
    CDbKey* key=CDbKey::NewLC();

     //Add the key columns.
     TDbKeyCol number1(KCol1);
     key->AddL(number1);
     TDbKeyCol number2(KCol2);
     key->AddL(number2);
     TDbKeyCol number3(KCol3);
     key->AddL(number3);
    User::LeaveIfError(database.CreateIndex(KTable, KTable, *key));
    
    //Cleans up the column set.  
    CleanupStack::PopAndDestroy(key);
    CleanupStack::PopAndDestroy(columns);  
    CleanupStack::PopAndDestroy(&database);  
    CleanupStack::PopAndDestroy(&dbs);   
    CleanupStack::PopAndDestroy(&fsSession);
    }

/**
Main function which instantiates CSmpExample, prints messages on console and starts
the three threads.  
*/
static void MainL()
    {
    CSmpExample* smpExample = CSmpExample::NewL();
    CleanupStack::PushL(smpExample);    
    smpExample->PrintMessage();
    smpExample->StartThreads();
    CleanupStack::PopAndDestroy(smpExample);
    }

/**
Entry point for the example
*/
extern TInt E32Main()
    {
    //Creates cleanup stack.
    __UHEAP_MARK;
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }
    //Run application code inside a TRAP harness.
    TRAPD(mainError, MainL());
    if(mainError != KErrNone)
        {
        _LIT(KUserPanic, "Main Failed");  
        User::Panic(KUserPanic, mainError);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }

