// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This is simple code that executes the task scheduled by the task scheduler. 
// The code also checks if the task requester has the required capability.
// This window belongs to the task scheduled to execute by the main program. 
// It shows the details of the task information passed by the Task Scheduler 
// to the executed program. 
//



/**
 @file
*/
#include <schtask.h>
#include <e32cons.h>

_LIT(KTaskConsoleName, "TaskExecutor");
_LIT(KContent,"\nContents of task file:\n");
_LIT(KTask,"\nRunning task: %S");
_LIT(KTaskData,"\nThis is the task data for a");
_LIT(KTaskId,"\nThe task Id is: %d\n");
_LIT(KValidity,"Task is valid until %S\n");
_LIT(KDateString,"%H%:1%T %*E%*D%X%*N%Y %1 %2 %3");
_LIT(KPressAnyKey,"Press any key to continue \n");
_LIT(KAbort,"\nCapabilities of the task scheduler and the executor do not match. Task aborted\n");
_LIT(KSidMismatch,"SID of the task executor is not same as that of the scheduler. Task aborted\n");
// Uid of the task requester
const TUint32 KTaskSchedulerSid = 0xE80000B5;
const TCapability KTaskCapability = ECapabilityWriteDeviceData;

/**
Extracts task data, and puts up a message on the console.
Validates the security of the task requester by 
a) Checking the secure ID; and 
b) Capabilities of the task requester.
*/
void TaskExecuteL(RFile& aTaskFile)
	{
	// Construct console
	CConsoleBase* console = Console::NewL(KTaskConsoleName, TSize(KConsFullScreen, KConsFullScreen));
	CleanupStack::PushL(console);
	console->Printf(KContent);
	
	// Open the filestore
	CFileStore*	store = CDirectFileStore::FromLC(aTaskFile);//pushes store onto CleanupStack
	RStoreReadStream instream;
	
	// Open the file containing the store
	instream.OpenLC(*store,store->Root());//pushes instream onto CleanupStack

	// Get task count
	TInt count = instream.ReadInt32L();
	for (TInt i=0;i<count;i++)
		{
		CScheduledTask* task = CScheduledTask::NewLC(instream); //pushes task onto CleanupStack
		
		// Check if this is the SID of task requester
		if(task->SecurityInfo().iSecureId == KTaskSchedulerSid)
			{
			
			// Check if the requester has the necessary capability to run 
			// the scheduled tasks.
			if(task->SecurityInfo().iCaps.HasCapability(KTaskCapability))
				{
				TBuf<50> buf;
				buf.Format(KTask, &task->Info().iName);
				console->Printf(buf);
			
				HBufC* data = const_cast<HBufC*>(&(task->Data()));
		
				console->Printf(KTaskData);
				console->Printf(*data);
				
				console->Printf(KTaskId,task->Info().iTaskId);
		
				// Get the time when the task stops being valid
				TTsTime tstime = task->ValidUntil();
				const TTime time = tstime.GetLocalTime();
				TBuf<30> dateString;
				time.FormatL(dateString,(KDateString));
				console->Printf(KValidity, &dateString);
				}
			else
				{
				console->Printf(KAbort);	
				}
			}
		else
			{
			console->Printf(KSidMismatch);
			}
		console->Printf(KPressAnyKey);
		console->Getch();

		CleanupStack::PopAndDestroy(task);
			
		}
	
	CleanupStack::PopAndDestroy(3); // instream, store, console
	}


TInt Execute()
	{
	TInt err = KErrNoMemory;
	CTrapCleanup* cleanup = CTrapCleanup::New();
	if (cleanup)
		{
		RFile file;
		
		// Needs to be called early-on to allow the task scheduler server
		// to adopt the already open task file from the task scheduler
		err = file.AdoptFromCreator(TScheduledTaskFile::FsHandleIndex(),
									TScheduledTaskFile::FileHandleIndex());
		if (err != KErrNone)
			{
			return err;		
			}
			
		// Execute 
		TRAPD(err,TaskExecuteL(file));
		if(err != KErrNone)
			{
			User::Panic(_L("Failed to complete"),err);
			}
	
		// Close the file
		file.Close();		
		delete cleanup;
		}		
	return err;
	}


GLDEF_C TInt E32Main()
	{
	return Execute();
	}
