// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This is simple example code that demonstrates the use of task scheduler. 
// The code demonstrates how to,
// 1) Connect and register with the task scheduler
// 2) Create a persistent schedule
// 3) Add tasks to it
// 4) Execute the schedule
// 5) Delete the schedules and the tasks associated with them after
// verifying that they are the schedules created by us.
// 6) Create a transient schedule
// 7) Add task to the transient schedule, edit the schedule and execute it
//



/**
 @file
*/
#include "taskscheduler.h"
#include <e32std.h>
#include <schinfo.h>
#include <e32base.h>

_LIT(KTaskName,"MyTaskName\n");

_LIT(KTitle, "Task Scheduler example");
_LIT(KTextPressAKey, "\nPress any key to step through the example\n");
_LIT(KExit,"\nPress any key to exit the application");
_LIT(KPressAKey,"\nPress any key to continue\n");

_LIT(KConnect, "\nConnecting the client to the task scheduler server");
_LIT(KRegisterClient,"\nRegistering the client\n");
_LIT(KCreateSchedule,"Creating a persistent schedule\n");
_LIT(KCreateTask,"Creating task for the schedule\n");

_LIT(KPersistentWait,"Waiting for the persistent task to complete. This will take 20 seconds\n");
_LIT(KDone,"Task complete \n");
_LIT(KTransientSchedule,"A transient schedule");
_LIT(KTransientWait,"Waiting for the transient task to complete. This will take 20 seconds\n");

_LIT(KCreateTransient,"\nCreating a transient schedule with non-repeating task\n");
_LIT(KDeleteAllTasks,"\nDeleting all the persistent tasks scheduled by this exe");
_LIT(KDeleteAllSchedules,"\nDeleting all schedules created by this exe\n");
_LIT(KTask, "Number of task(s) scheduled by us is(are) %d\n");
_LIT(KExists, "The tasks scheduled exist\n");
_LIT(KOtherTask,"Error! Unexpected schedules not scheduled by this exe exist\n");

/**
Allocates and constructs a CTaskSchedule object using two phase construction
Initialises all member data to their default values.
*/	
CTaskSchedule* CTaskSchedule::NewLC()
	{
	CTaskSchedule* schedule = new(ELeave) CTaskSchedule();
	CleanupStack::PushL(schedule);
	schedule->ConstructL();
	return schedule;
	}
	
/**
Constructor
*/
CTaskSchedule::CTaskSchedule()
	{
	}	

void CTaskSchedule::ConstructL()
	{	
	iConsole = Console::NewL(KTitle,TSize(KConsFullScreen,KConsFullScreen));
	iConsole->Printf (KTextPressAKey);
	iConsole->Getch ();
	}

/**
Destructor
*/
CTaskSchedule::~CTaskSchedule()
	{
	iScheduler.Close();
	iConsole->Printf(KExit);
	iConsole->Getch();
	
	delete iConsole;
	}

/**
Connects a client to the task scheduler server by 
creating a session with that server and registers the client 
with the scheduler.
@leave system-wide error codes.
*/
void CTaskSchedule::ConnectAndRegisterL()
	{
	
	// Connect to the scheduler server
	iConsole->Printf(KConnect);
	User::LeaveIfError(iScheduler.Connect());
	
	_LIT(KTaskExec,"Taskexecutor");
	TFileName filename;
	filename.Append(KTaskExec);
	
	// Register with the scheduler
	iConsole->Printf(KRegisterClient);
	
	// A priority value
	const TInt priority = 3;
	User::LeaveIfError(iScheduler.Register(filename, priority));
	
	}

/**
Creates a persistent schedule task by calling CreatePersistentScheduleL().
Launches the task executor after adding and scheduling the task.
@leave KErrArgument
@leave system-wide error codes.
*/
void CTaskSchedule::PersistentScheduleL()
	{
	
	iConsole->Printf(KCreateSchedule);		
	TSchedulerItemRef scheduleHandle;
	TTime time;
	time.UniversalTime();
	
	// Assign an offset of 20 seconds. TTimeIntervalMinutes, 
	// TTimeIntervalHours etc can be used for longer offsets.
	time += TTimeIntervalSeconds(20);// 20 secs in future

	TTsTime time2 (time, ETrue);

	iConsole->Printf(KCreateTask);
	CreatePersistentScheduleL(scheduleHandle,time2);

	// Create the task to be scheduled
	TTaskInfo taskInfo;
	
	const TInt priority = 2; 
	const TInt numberOfRepeats = 2;
	// Name of the task
	taskInfo.iName = KTaskName;
	
	// Task priority set by the client. 
	// If the client has two tasks with different priorities, 
	// the task with the higher priority will be executed first.
	taskInfo.iPriority = priority;
	
	// Task repeats twice
	taskInfo.iRepeat = numberOfRepeats;
	
	_LIT(KScheduleType," persistent schedule");
	const TDesC* persistent = &KScheduleType;
	HBufC* data = persistent->AllocLC();
	
	// Add the task
	User::LeaveIfError(iScheduler.ScheduleTask(taskInfo, *data, scheduleHandle.iHandle));
				
	// Wait for the entry to fire
	iConsole->Printf(KPersistentWait);
	const TInt twentySecs = 20000000;
	User::After(twentySecs); // pause for 20 seconds
	
	// Delete all the persistent schedules created
	DeleteSchedulesL(scheduleHandle, EAllSchedules);

	CleanupStack::PopAndDestroy(1); // data
	
	}

	
/**
Creates a persistent time based schedule with no tasks associated
with it but merely contains start and end time information.
A persistent schedule is a schedule whose lifetime is not limited
to the lifetime of the tasks associated with it.
Persistent schedules have their information persisted to disk. 
On device reboot, this data is read back into memory by the 
task scheduler server.
@param aRef 		Reference to TSchedulerItemRef for unique identification of the schedule
@param aStartTime 	Reference to TTsTime class

@leave KErrArgument
@leave system-wide error codes.
*/	
void CTaskSchedule::CreatePersistentScheduleL(TSchedulerItemRef& aRef, const TTsTime& aStartTime)
	{
	CArrayFixFlat<TScheduleEntryInfo2>* cSchEntryInfoArray;	
	cSchEntryInfoArray = new CArrayFixFlat<TScheduleEntryInfo2>(1);
	CleanupStack::PushL(cSchEntryInfoArray);

	// Create an hourly schedule with StartTime of aStartTime
	TScheduleEntryInfo2 entry1;
	
	// Set the first instance when the entry will 
	// cause the execution of tasks. 
	entry1.SetStartTime(aStartTime);
	
	// Set the type of interval used between due times 
	// for the scheduled entry
	entry1.SetIntervalType(TIntervalType(EHourly));
	
	// Set the period for which the entry is valid. After 8 hours the 
	// tasks associated with the entry will not be eligible for execution.
	const TInt eightHours = 480; // in minutes 
	entry1.SetValidityPeriod(TTimeIntervalMinutes (eightHours));
	
	// Set the interval between execution of tasks
	// Here the interval is 1 hour because the interval type is hourly.
	const TInt interval = 1;
	entry1.SetInterval(interval);

	cSchEntryInfoArray->AppendL(entry1);
	
	User::LeaveIfError(iScheduler.CreatePersistentSchedule(aRef, *cSchEntryInfoArray));
	CleanupStack::PopAndDestroy(cSchEntryInfoArray);
	
	}
	

/**
Creates a new, transient, time based schedule, adds a task to it and then 
edits the schedule and executes the edited schedule.
Launches the task executor after scheduling the task.
A transient schedule is destroyed when the task is destroyed or power is lost.
@leave KErrArgument
@leave system-wide error codes.
*/	
void CTaskSchedule::CreateTransientScheduleL()
	{
	iConsole->ClearScreen();

	TSchedulerItemRef ref;
	CArrayFixFlat<TScheduleEntryInfo2>* cSchEntryInfoArray;
	cSchEntryInfoArray = new CArrayFixFlat<TScheduleEntryInfo2>(1);
	
	CleanupStack::PushL(cSchEntryInfoArray);
	ref.iName = KTransientSchedule;
	iConsole->Printf(KCreateTransient);
	
	// Create a schedule entry
	TScheduleEntryInfo2 entry;	
	TTime now;
	
	//	Set the date and time of this TTime to the universal time. 
	now.UniversalTime();
	
	// Assign an offset of 15 seconds. TTimeIntervalMinutes, 
	// TTimeIntervalHours etc can be used for longer offsets.
	TInt offset = 15;
	now += TTimeIntervalSeconds(offset);
	
	// Constructs a TTsTime with a TTime object. 
	// ETrue indicates that TTsTime is UTC based time
	TTsTime time (now, ETrue);
	
	// Set the above time as the first time at which 
	// the entry will cause execution of tasks. 
	entry.SetStartTime(time);
	
	// Set the type of interval used between due times for scheduled entry
	entry.SetIntervalType(TIntervalType(EHourly));
	
	// Set the period for which the entry is valid. 
	// After 2 hours the tasks associated with the entry will not be eligible for execution 
	TInt validity = 120;
	entry.SetValidityPeriod(TTimeIntervalMinutes (validity));
	
	// Set the interval between execution of tasks
	// Here the interval is 1 hour because the interval type is hourly.
	entry.SetInterval(1);
	cSchEntryInfoArray->AppendL(entry);
	
	// Create a transient task to be scheduled
	TTaskInfo taskInfo;
	// Name of the task
	taskInfo.iName = KTransientSchedule;
	// Task id
	const TInt tId = 0;
	taskInfo.iTaskId = tId;
	
	// The task repeats just once
	const TInt numberOfRepeats = 1;
	taskInfo.iRepeat = numberOfRepeats;
	
	// Task priority set by the client. 
	// Where a client has two tasks with different priorities, 
	// the task with the higher priority will be executed first.
	const TInt priority = 2; 
	taskInfo.iPriority = priority;
	
	_LIT(KScheduleType," transient schedule");
	const TDesC* transient = &KScheduleType;
	HBufC* data = transient->AllocLC();
		
	// Schedule the item
	User::LeaveIfError(iScheduler.ScheduleTask(taskInfo, *data, ref, *cSchEntryInfoArray));
	
	// Change the start time and validity duration for the schedule
	offset = 19; // 19 seconds in future
	now += TTimeIntervalSeconds(offset);
	TTsTime newTime (now, ETrue);
	entry.SetStartTime(newTime);
	validity = 300;
	entry.SetValidityPeriod(TTimeIntervalMinutes (validity));
	cSchEntryInfoArray->AppendL(entry);
	
	// Change the transient time based schedule
    User::LeaveIfError(iScheduler.EditSchedule(ref.iHandle, *cSchEntryInfoArray));
    	
	CleanupStack::PopAndDestroy(2); // data, CSchEntryInfoArray

	// Check if the tasks scheduled exist
	TBool exists;
	DoesScheduledItemExistL(ref, exists);
	if(!exists)
		{
		User::Leave(KErrNotFound);
		}
	// Wait for the task to fire and complete
	iConsole->Printf(KTransientWait);
	const TInt twentySecs = 20000000;
	User::After(twentySecs); // pause for 20 seconds
	
	iConsole->Printf(KDone);	

	// Transient schedules are deleted automatically once they are executed
	// i.e check for count to be zero in the function being called
	DoesScheduledItemExistL(ref, exists);
	if(exists)
		{
		User::Leave(KErrGeneral);
		}
	}

/**
Checks if a schedule exists
@param aRef 	Reference to TSchedulerItemRef for unique identification of the schedule
@param aExists 	TBool value, ETrue if the schedule exists, EFalse otherwise
@leave system wide error codes
*/
 void CTaskSchedule::DoesScheduledItemExistL(TSchedulerItemRef &aRef, TBool& aExists)
	// Extract schedule references from the schedule server based on a filter.
	{
	aExists = EFalse;
	CArrayFixFlat<TSchedulerItemRef>* CSchItemRefArray;
	
	CSchItemRefArray = new CArrayFixFlat<TSchedulerItemRef>(3);
	CleanupStack::PushL(CSchItemRefArray);
	
	User::LeaveIfError(iScheduler.GetScheduleRefsL(*CSchItemRefArray, EAllSchedules));
	
	TInt count = CSchItemRefArray->Count();
	for(TInt i = 0; i < count; i++)
		{
		// Get a list of schedules created by this executable
		if(aRef.iHandle == (*CSchItemRefArray)[i].iHandle)
			{
			aExists = ETrue;
			iConsole->Printf(KExists);
			iConsole->Printf(KTask, count);
			}
		else
			{
			iConsole->Printf(KOtherTask);
			}
		}

	CleanupStack::PopAndDestroy(); // CSchItemRefArray

	}

/**
Delete all tasks and schedules created by this exe before program exit
@param TSchedulerItemRef 	Reference to TSchedulerItemRef for unique identification of the schedule
@param TScheduleFilter 		Reference to a filter when listing the schedules

@leave KErrNotFound
@leave KErrAbort
@leave KErrPermissionDenied,
@leave KErrArgument
@leave system-wide error codes.
*/		
void CTaskSchedule::DeleteSchedulesL(TSchedulerItemRef &aRef, TScheduleFilter aFilter)
	{
	CArrayFixFlat<TSchedulerItemRef>* CSchItemRefArray;
	
	CSchItemRefArray = new CArrayFixFlat<TSchedulerItemRef>(3);
	CleanupStack::PushL(CSchItemRefArray);
	
	User::LeaveIfError(iScheduler.GetScheduleRefsL(*CSchItemRefArray, aFilter));
	
	TInt count = CSchItemRefArray->Count();
	iConsole->Printf(KTask, count);
	
	for(TInt i = 0; i < count; i++)
		{
		// Check if the schedules are created by this exe or
		// some other exe
		if(aRef.iHandle == (*CSchItemRefArray)[i].iHandle)
			{	
			// Delete the tasks scheduled by us (this exe)
			iConsole->Printf(KDeleteAllTasks);
			User::LeaveIfError(iScheduler.DeleteTask(aRef.iHandle));
			// Delete the schedules created by us (this exe)
			iConsole->Printf(KDeleteAllSchedules);
			User::LeaveIfError(iScheduler.DeleteSchedule(aRef.iHandle));
			
			iConsole->Printf(KPressAKey);
			iConsole->Getch();
					
			}
		else
			{
			// Not deleting the tasks or schedules as,
			// they are not created by this exe
			iConsole->Printf(KOtherTask);
			}
		}
	CleanupStack::PopAndDestroy(); // CSchItemRefArray
	
	}
	
void MainL()
	{
	// Create an Active Scheduler to handle asychronous calls
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler;
	CleanupStack::PushL(scheduler);

	// Install the active scheduler
	CActiveScheduler::Install( scheduler );
	CTaskSchedule* app = CTaskSchedule::NewLC();
	
	// Connect and register with the task scheduler
	app->ConnectAndRegisterL();
	
	// Create a persistent schedule and add tasks to it
	app->PersistentScheduleL();
		
	// Create a transient schedule and add tasks to it
	app->CreateTransientScheduleL();
	
	CleanupStack::PopAndDestroy(2); // app, scheduler

	}

TInt E32Main()
	{
    __UHEAP_MARK;
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(cleanup == NULL)
    	{
    	return KErrNoMemory;
    	}
    	
    TRAPD(err, MainL());
	if(err != KErrNone)
		{
		User::Panic(_L("Failed to complete"),err);
		}
    delete cleanup;
  
    __UHEAP_MARKEND;
    return KErrNone;
	}
