/*
 * Copyright  2008 Nokia Corporation.
 */

#ifndef __BOOKDBENGINE_H__
#define __BOOKDBENGINE_H__


#include <e32std.h>
#include <badesca.h>    // CDesCArrayFlat (cannot be forward declarated)
#include <d32dbms.h>    // RDbStoreDatabase
#include <f32file.h>    // RFs


// Forward declarations
class CFileStore;

// Constants
_LIT(KBooksTable, "Books");                /* Name of the one table       */
_LIT(KBooksAuthorCol, "Author");           /* First column                */
_LIT(KBooksTitleCol, "Title");             /* Second column               */
_LIT(KBooksDescriptionCol, "Description"); /* Third column                */
_LIT(KBooksDateCol,"PublishDate");         /* Dynamic optional column     */
_LIT(KBooksIndexName,"BooksIndex");        /* Name of the one index       */
const int KTitleMaxLength = 60;            /* Max length of title column  */
const int KDescriptionMaxLength = 128;     /* Max length of descr. column */


/**
 *  Maximum length of individual item in CDesCArrayFlat (see GetAllBooksL &
 *  GetBooksByKeyL methods).
 */
const int KBookItemMaxLength = 256;

/**
 *  Separator within individual CDesCArrayFlat item. Each item has format
 *  <Author>|<Title>|<Description>
 */
_LIT(KSeparator,"|");


/**
 * Class:       CBookDb
 *
 * Description: An instance of class CBookDb provides simple Book
 *              database access: creating & manipulating Book database
 *              files and database entries (books).
 *
 * Database definition:
 *
 *              The Book contains one table as follows:
 *
 *              Table name: Books
 *                 Column:       Type:          Max length:
 *                 ------        -----          -----------
 *                 Author        EDbColText     50  (using default)
 *                 Title         EDbColText     60  (see KTitleMaxLength)
 *                 Description   EDbColLongText 128 (see KDescriptionMaxLength)
 *    In run time: PublishDate   EDbColDateTime
 *
 *              Note that underlying database allows description to be up to
 *              2GB long, but this engine limits the size to 128 unicode
 *              characters.
 *              The PublishDate does not exist, when the database is created.
 *              It can be added on the fly (see AddDateColumnL method).
 *
 *              There is also index for Books table with name "BooksIndex". It
 *              consists of two columns:
 *
 *                   Author, Title
 *
 *              Index provides quick find.
 */
class CBookDb : public CBase
    {
public: // Creation and destruction

    /**
     * Function:    NewL
     *
     * Description: Get instance of a CBookDb object.
     */
    static CBookDb* NewL();

    /**
     * Function:    ~CBookDb
     *
     * Description: Destroy the object.
     */
    ~CBookDb();


public: // Public API for database operations

    /**
     * Function:    OpenDbL
     *
     * Description: Open existing Book database in exclusive
     *              (non-shared) mode.
     *
     * Param:       aExistingBookFile is full path to Book
     *              database file.
     *
     * Return:      KErrNone, if no error. KErrNotFound, if the file does
     *              not exist.
     *
     * Leaves:      With one of the system wide error codes, if the file
     *              is not a correct database file.
     */
    TInt OpenDb(const TFileName& aExistingBookFile);

    /**
     * Function:    CreateDbL
     *
     * Description: Creates and opens a new Book database. Creates a
     *              database file, table structure and an index. The database
     *              will be open in exclusive (non-shareable) mode. The
     *              database must be closed, when not used any more. If the
     *              database exists, it is replaced.
     *
     * Param:       aNewBookFile Name of the new database file. Is  a full
     *              path (incl. the filename). Operations following this call
     *              are performed to the new database file.
     *
     * Return:      returns always KErrNone
     *
     * Leaves:      If the file cannot be created or database initialized.
     *              Leaves with system wide error codes.
     */
    TInt CreateDb(const TFileName& aNewBookFile);

    /**
     * Function:    RemoveDbL
     *
     * Description: Removes Book database. Closes any open database,
     *              before dropping the database.
     *
     * Param:       aExistingBookFile is full path to Book
     *              database file.
     *
     * Leaves:      If the file file is not a valid database file or the
     *              database does not containt Books table, the method
     *              leaves with system wide error codes.
     */
    TInt RemoveDb(const TFileName& aExistingBookFile);

    /**
     * Function:    Close
     *
     * Description: Closes the database opened with either OpenDbL or
     *              CreateDbL. It is safe to close the database even if it
     *              is closed.
     *
     * Return:      KErrNone, if no error. KErrNotFound, if the file does
     *              not exist.
     */
    TInt Close();

    /**
     * Function:    IsOpen
     *
     * Description: Return status, whether the database is open and ready
     *              for operations.
     */
    TBool IsOpen() const;

    /**
     * Function:    AddBookWithSql
     *
     * Description: Adds a new book to Books table. The book is inserted using
     *              SQL and RDbView.
     *
     * Param:       aAuthor Author of the book. Must be shorter than
     *              the default max text length in Book API (=50). Must not be
     *              empty.
     *
     * Param:       aTitle Title of the book. Must be shorter than
     *              KTitleMaxLength. Must not be empty.
     *
     * Param:       aDescription Description of the book. It must not be longer
     *              than KDescriptionMaxLength. Must not be empty.
     */
    TInt AddBookWithSql(const TDesC& aAuthor,
                         const TDesC& aTitle,
                         const TDesC& aDescription);

    /**
     * Function:    AddBookWithCppApiL
     *
     * Description: Adds a new book to Books table. The book is inserted using
     *              RDbTable API.
     *
     * Param:       aAuthor Author of the book. Must be shorter than
     *              the default max text length in Book API (=50). Must not be
     *              empty.
     *
     * Param:       aTitle Title of the book. Must be shorter than
     *              KTitleMaxLength. Must not be empty.
     *
     * Param:       aDescription Description of the book. It must not be longer
     *              than KDescriptionMaxLength. Must not be empty.
     */
    TInt AddBookWithCppApiL(const TDesC& aAuthor,
                            const TDesC& aTitle,
                            const TDesC& aDescription);

    /**
     * Function:    GetAllBooksL
     *
     * Description: Retrieve all books from database.
     *
     * Returns:     Array of books. Each array item is represented as follows:
     *                  <Author>|<Title>|<Description>
     *              Maximum length of each item is KBookItemMaxLength
     *              Caller takes ownership of the array.
     */
    CDesCArrayFlat* GetAllBooksL();

    /**
     * Function:    GetBooksByKeyL
     *
     * Description: Retrieve books from database, which match given column
     *              and search pattern. Implementation uses SQL.
     *
     * Param:       aColumnName Name of the column to apply the search pattern.
     *              Must be either KBooksAuthorCol or KBooksTitleCol
     *
     * Param:       aSearchString Search pattern used to restrict results to.
     *
     * Returns:     Array of books. Each array item is represented as follows:
     *                  <Author>|<Title>|<Description>
     *              Maximum length of each item is KBookItemMaxLength
     *              Caller takes ownership of the array.
     */
    CDesCArrayFlat* GetBooksByKeyL(const TDesC& aColumnName,
        const TDesC& aSearchString);

    /**
     * Function:    GetABookFast
     *
     * Description: Retrieves book information for given book name. This method
     *              uses index to find first occurrence of the book.
     *              Implementation uses exact match.
     *
     * Param:       aTitle is name of the book to search for.
     *
     * Param:       aResult If there is a matching book, the complete book info
     *              is written to aResult. It is in the following format:
     *                  <Author>|<Title>|<Description>
     *              Length of the given descriptor must be KBookItemMaxLength
     */
    TInt GetABookFast(const TDesC& aTitle, TDes& aResult);

    /**
     * Function:    RemoveBooks
     *
     * Description: Deletes book(s) from database.
     *
     * Param:       aTitle is name of the book to delete. It can contain
     *              wildcard characters (% for single char, * for zero or
     *              multiple chars).
     *
     * Param:       aResultCount will contain number of deleted books.
     *
     * Returns:     KErrNone or one of the system wide error codes.
     */
    TInt RemoveBooks(const TDesC& aTitle, TInt& aResultCount);

    /**
     * Function:    RemoveAllBooksL
     *
     * Description: Deletes all books from database.
     *
     * Param:       aResultCount will contain number of deleted books.
     *
     * Returns:     KErrNone or one of the system wide error codes.
     */
    TInt RemoveAllBooks(TInt& aResultCount);


    /**
     * Function:    UpdateBookTitle
     *
     * Description: Changes the title for a book (or books, if there are
     *              multiple books with the name aOldTitleKey).
     *
     * Param:       aOldTitleKey Book title used for getting books for update.
     *
     * Param:       aNewTitle New title for the book(s).
     *
     * Returns:     KErrNone or one of the system wide error codes.
     */
    TInt UpdateBookTitle(const TDesC& aOldTitleKey, const TDesC& aNewTitle);



    /**
     * Function:    ColumnNamesAndSizes
     *
     * Description: Get array of colum names in the Books table. The result
     *              array includes also the size of the textual columns.
     *              This is here just to demonstrate iterating column names
     *              from a table.
     *
     * Returns:     Array of column names. Caller takes ownership.
     */
    CDesCArrayFlat* ColumnNamesAndSizesL();

    /**
     * Function:    HasDateColumn
     *
     * Description: Tests whether the Books table has date column.
     *
     * Returns:     KErrNone or one of the system wide error codes.
     */
    TInt HasDateColumn(TBool& aReturnValue);

    /**
     * Function:    AddDateColumn
     *
     * Description: Adds date column to Books table. This here just to
     *              demonstrate how to alter table definition.
     *              This fails, if the date column already exists.
     *
     * Returns:     KErrNone or one of the system wide error codes.
     */
    TInt AddDateColumn();

    /**
     * Function:    RemoveDateColumn
     *
     * Description: Removes date column from Books table. This here just to
     *              demonstrate how to alter table definition.
     *              This fails, if the date column does not exist.
     *
     * Returns:     KErrNone or one of the system wide error codes.
     */
    TInt RemoveDateColumn();

private: // Construction

    /**
     * Function:    ConstructL
     *
     * Description: Perform the second phase construction of a CBookDb
     *              object
     */
    void ConstructL();

    /**
     * Function:    CBookDb
     *
     * Description: Perform the first phase of two phase construction.
     */
    CBookDb();

private: // Helpers

    void CreateBooksTableL();     /* Create the Books table                  */
    void CreateBooksIndexL();     /* Create the BooksIndex for Books table   */
    void DropBooksTable();       /* Drop Books table (DDL example)          */

private: // Member data

    RFs              iFsSession;  /* For use of iFileStore                   */
    RDbStoreDatabase iBookDb;/* For database operations                 */
    CFileStore*      iFileStore;  /* For creating and opening database files */
    TBool            iOpen;       /* Flag indicating iBookDb open status*/
    };

#endif // __BOOKDBENGINE_H__


