/*
 * Copyright  2008 Nokia Corporation.
 */

#include <avkon.hrh>
#include <badesca.h>
#include <bautils.h>
#include <eikmenup.h>
#include <eikapp.h>
#include <aknnotewrappers.h> // CAknInformationNote
#include <aknquerydialog.h>
#include "DBMSAppUi.h"
#include "DBMSAppView.h"
#include "DBMSListboxView.h"
#include "DBMSEditorView.h"
#include "DBMS.hrh"
#include <DBMS.rsg>  // R_MAIN_MENU

#define KEnableSkinFlag 0x1000

// Implementation constants
const TInt KArrayGranularity = 5;

//Text constants
_LIT(KCDrive, "C:");

// ---------------------------------------------------------------------------
// CDBMSAppUi::ConstructL()
//
// Perform second phase construction. This is allowed to leave.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::ConstructL()
    {
	BaseConstructL(EAknEnableSkin);

    iCurrentView = ENone;

    ChangeViewL(EMainView);

	iDatabaseFile = DatabaseDriveAndPathL();

    iDatabaseFile.Append(KDatabaseFile);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::CDBMSAppUi()
//
// Constructor of the AppUi. This may not leave.
// ---------------------------------------------------------------------------
//
CDBMSAppUi::CDBMSAppUi() : iAppView(NULL), iListboxView(NULL),
    iBookEditorView(NULL), iBookDb(NULL)
    {
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::~CDBMSAppUi()
//
// Destructor. Release reserved resources
// ---------------------------------------------------------------------------
//
CDBMSAppUi::~CDBMSAppUi()
    {
    if (iAppView)
        {
        if(iCurrentView == EMainView)
            iEikonEnv->RemoveFromStack(iAppView);
        delete iAppView;
        iAppView = NULL;
        }

    if (iListboxView)
        {
        if(iCurrentView == EListView || iCurrentView == EColumnsView)
            iEikonEnv->RemoveFromStack(iListboxView);
        delete iListboxView;
        iListboxView = NULL;
        }

    if (iBookEditorView)
        {
        if(iCurrentView == EEditorView)
            iEikonEnv->RemoveFromStack(iBookEditorView);
        delete iBookEditorView;
        iBookEditorView = NULL;
        }

    if(iBookDb) // Just in case
        {
        iBookDb->Close();
        delete iBookDb;
        iBookDb = NULL;
        }
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::ChangeView()
//
// Switch view.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::ChangeViewL(TViewId aNewView)
    {

    if(aNewView == iCurrentView)
        return;

    // Delete previous view (window owning control)
    switch(iCurrentView)
        {
        case EMainView:
            RemoveFromStack(iAppView);
            delete iAppView;
            iAppView = NULL;
            break;

        case EListView:
            RemoveFromStack(iListboxView);
            delete iListboxView;
            iListboxView = NULL;
            break;

        case EColumnsView:
            RemoveFromStack(iListboxView);
            delete iListboxView;
            iListboxView = NULL;
            break;

        case EEditorView:
            RemoveFromStack(iBookEditorView);
            delete iBookEditorView;
            iBookEditorView = NULL;
            break;

        case ENone:
        default:
            break;
        }
    // Create new view (window owning control)
    switch(aNewView)
        {
        case EMainView:
            iAppView = CDBMSAppView::NewL(ClientRect());
            AddToStackL(iAppView);
            break;

        case EListView:
        case EColumnsView:
            iListboxView = CDBMSListboxView::NewL(ClientRect());
            AddToStackL(iListboxView);
            break;

        case EEditorView:
            iBookEditorView = CDBMSEditorView::NewL(ClientRect());
            AddToStackL(iBookEditorView);
            break;

        case ENone:
        default:
            break;
        }
    iCurrentView = aNewView;
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::DynInitMenuPaneL(...)
//
// Initialize menu before it is shown. This overrides default implementation
// in base class.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::DynInitMenuPaneL(TInt aResourceId,
    CEikMenuPane* aMenuPane)
    {
    if (aResourceId != R_MAIN_MENU)
        {
        return;
        }

    // First hide all menu items
    aMenuPane->SetItemDimmed(EOpenCmd, ETrue);
    aMenuPane->SetItemDimmed(ECreateCmd, ETrue);
    aMenuPane->SetItemDimmed(ERemoveDbCmd, ETrue);
    aMenuPane->SetItemDimmed(EAddBookCmd, ETrue);
    aMenuPane->SetItemDimmed(EBackCmd, ETrue);
    aMenuPane->SetItemDimmed(EAddBookAPICmd, ETrue);
    aMenuPane->SetItemDimmed(EAddBookSQLCmd, ETrue);
    aMenuPane->SetItemDimmed(ERemoveBookCmd, ETrue);
    aMenuPane->SetItemDimmed(ERemoveAllBooksCmd, ETrue);
    aMenuPane->SetItemDimmed(EChangeTitleCmd, ETrue);
    aMenuPane->SetItemDimmed(EGetAllBooksCmd, ETrue);
    aMenuPane->SetItemDimmed(ESearchBooksCmd, ETrue);
    aMenuPane->SetItemDimmed(EQuickFindCmd, ETrue);
    aMenuPane->SetItemDimmed(EAddDateCmd, ETrue);
    aMenuPane->SetItemDimmed(ERemoveDateCmd, ETrue);
    aMenuPane->SetItemDimmed(EColumnNamesCmd, ETrue);
    aMenuPane->SetItemDimmed(ECloseCmd, ETrue);

    // Show appropriate menu items depending on the current view
    switch(iCurrentView)
        {
        case EMainView:
            {
            // Db is not open. Allow creating, opening or removing the database
            if(BaflUtils::FileExists(CCoeEnv::Static()->FsSession(),
                iDatabaseFile))
                {
                aMenuPane->SetItemDimmed(EOpenCmd, EFalse);
                aMenuPane->SetItemDimmed(ERemoveDbCmd, EFalse);
                }
            aMenuPane->SetItemDimmed(ECreateCmd, EFalse);
            break;
            }

        case EListView:
            {
            // DB is open, allow / show db operations
            if(iBookDb && iBookDb->IsOpen())
                {
                aMenuPane->SetItemDimmed(EAddBookCmd, EFalse);
                aMenuPane->SetItemDimmed(ERemoveBookCmd, EFalse);
                aMenuPane->SetItemDimmed(ERemoveAllBooksCmd, EFalse);
                aMenuPane->SetItemDimmed(EChangeTitleCmd, EFalse);
                aMenuPane->SetItemDimmed(EGetAllBooksCmd, EFalse);
                aMenuPane->SetItemDimmed(ESearchBooksCmd, EFalse);
                aMenuPane->SetItemDimmed(EQuickFindCmd, EFalse);
                aMenuPane->SetItemDimmed(EColumnNamesCmd, EFalse);
                aMenuPane->SetItemDimmed(ECloseCmd, EFalse);
                }
            break;
            }

        case EColumnsView:
            {
            if(iBookDb && iBookDb->IsOpen())
                {
                // Get existence status of date column
                TBool hasDateColumn(EFalse);
                User::LeaveIfError(iBookDb->HasDateColumn(hasDateColumn));

                aMenuPane->SetItemDimmed(EBackCmd, EFalse);
                if(!hasDateColumn)
                    aMenuPane->SetItemDimmed(EAddDateCmd, EFalse);
                else
                    aMenuPane->SetItemDimmed(ERemoveDateCmd, EFalse);
                aMenuPane->SetItemDimmed(ECloseCmd, EFalse);
                }
            break;
            }

        case EEditorView:
            {
            if(iBookDb && iBookDb->IsOpen())
                {
                aMenuPane->SetItemDimmed(EBackCmd, EFalse);
                aMenuPane->SetItemDimmed(EAddBookAPICmd, EFalse);
                aMenuPane->SetItemDimmed(EAddBookSQLCmd, EFalse);
                aMenuPane->SetItemDimmed(ECloseCmd, EFalse);
                }
            break;
            }

        case ENone:
        default:
            break;
        }
    }

// ---------------------------------------------------------------------------
// CDBMSAppUi::HandleCommandL(...)
//
// Handle menu commands.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::HandleCommandL(TInt aCommand)
    {

    switch(aCommand)
        {

    case EOpenCmd:            // Open existing database
        OpenDatabaseL();
        break;

    case ECreateCmd:          // Create (replace) and open a database
        CreateDatabaseL();
        break;

    case ERemoveDbCmd:        // Drop database. Remove the database file
        RemoveDatabaseL();
        break;

    case EBackCmd:            // Move back from editor or columns view to
        ShowAllBooksL();      // list view
        break;

    case EAddBookCmd:         // Show a Book editor for adding a Book
        ShowBookEditorViewL();
        break;

    case EAddBookAPICmd:      // Add a Book using Book API
        AddBookL(EFalse);
        break;

    case EAddBookSQLCmd:      // Add a Book using SQL
        AddBookL(ETrue);
        break;

    case ERemoveBookCmd:      // Remove a named Book from database
        RemoveBookL();
        break;

    case ERemoveAllBooksCmd:  // Remove all Books from database
        RemoveAllBooksL();
        break;

    case EChangeTitleCmd:     // Update a Book title
        UpdateBookTitleL();
        break;

    case EGetAllBooksCmd:     // Read all Books from database
        ShowAllBooksL();
        break;

    case ESearchBooksCmd:     // Search Books with given search pattern
        SearchBooksL();
        break;

    case EQuickFindCmd:       // Find a Book using index
        IndexFindL();
        break;

    case EAddDateCmd:         // Add a date column to Books table
        AddDateColumnL();
        break;

    case ERemoveDateCmd:      // Remove the date column from Books table
        RemoveDateColumnL();
        break;

    case EColumnNamesCmd:     // Show column names of database
        ShowColumnsL();
        break;

    case ECloseCmd:           // Close database
        CloseDatabaseL();
        break;

    case EAknSoftkeyExit:     // From CBA
    case EEikCmdExit:         // From operating system / framework
        Exit();
        break;

    default:
        break;
        }
    }

// ---------------------------------------------------------------------------
// CDBMSAppUi::OpenDatabaseL()
//
// Create instance of iDBMSDb and open existing database.
// ---------------------------------------------------------------------------
void CDBMSAppUi::OpenDatabaseL()
    {
    iBookDb = CBookDb::NewL();
    iBookDb->OpenDb(iDatabaseFile);
    ShowAllBooksL(); // Change to list view
    }

// ---------------------------------------------------------------------------
// CDBMSAppUi::CreateDatabaseL()
//
// Create instance of iDBMSDb and create a new database.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::CreateDatabaseL()
    {
    _LIT(KMessage,"Database created.");
    iBookDb = CBookDb::NewL();
    iBookDb->CreateDb(iDatabaseFile); // replaces, if exists
    ShowAllBooksL(); // Change to list view
    ShowNoteL(KMessage);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::RemoveDatabaseL()
//
// Remove database file from the system.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::RemoveDatabaseL()
    {
    _LIT(KMessage,"Database removed.");
    iBookDb = CBookDb::NewL();
    iBookDb->RemoveDb(iDatabaseFile);
    delete iBookDb;
    iBookDb = NULL;
    ShowNoteL(KMessage);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::CloseDatabaseL()
//
// Close an open database. Database opened with OpenDatabaseL or
// CreateDatabaseL must be closed, when not used any more.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::CloseDatabaseL()
    {
    if(iBookDb && iBookDb->IsOpen())
        {
        iBookDb->Close();
        delete iBookDb;
        iBookDb = NULL;
        }
    ChangeViewL(EMainView);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::ShowDBMSEditorView()
//
// Activate DBMS editor view
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::ShowBookEditorViewL()
    {
    ChangeViewL(EEditorView);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::AddDBMSL()
//
// Add a Book to database. Query details from DBMS editor view.
//
// There are two variants for insertion. See DBEngine.h fordetails.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::AddBookL(TBool aUseSql)
    {

    _LIT(KErrorMsg,"Failed. Make sure the fields are not empty.");
    TInt err(KErrNone);

    // Lengths are from DBEngine.h. Author uses default length (50)
    TBuf<50> author;
    TBuf<KTitleMaxLength> title;
    TBuf<KDescriptionMaxLength> description;
    

    iBookEditorView->GetAuthorL(author);
    iBookEditorView->GetTitleL(title);
    iBookEditorView->GetDescriptionL(description);

    if(aUseSql)
        err = iBookDb->AddBookWithSql(author,
                                            title,
                                            description);
    else
        err = iBookDb->AddBookWithCppApiL(author,
                                               title,
                                               description);

    if(err)
        ShowNoteL(KErrorMsg);
    else
        ShowAllBooksL(); // Change back to listbox view
        
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::RemoveDBMSL()
//
// Remove selected Book from the database.
//
// Implementation removes named Book from the database. If there are multiple
// matches for the Book title, the are all removed.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::RemoveBookL()
    {

    TBuf<KBookItemMaxLength> selectedBookTitle;
    if(iListboxView->GetSelectedItem(selectedBookTitle) != KErrNone)
        {
        _LIT(KErrorMsg,"Failed. Book not selected.");
        ShowNoteL(KErrorMsg);
        return;
        }

    TInt resultCount;
    // Wildcards are also allowed, like 'Title*', or '?itle"
    iBookDb->RemoveBooks(selectedBookTitle, resultCount);
    ShowAllBooksL();
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::RemoveAllDBMSsL()
//
// Remove all Books from database.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::RemoveAllBooksL()
    {
    _LIT(KInfoText,"All Books removed.");
    TInt resultCount;
    iBookDb->RemoveAllBooks(resultCount);
    ShowNoteL(KInfoText);
    ShowAllBooksL();
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::UpdateDBMSTitleL()
//
// Change the title of a selected Book.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::UpdateBookTitleL()
    {
    _LIT(KQuery,"New title(s) for: ");
    TBuf<KBookItemMaxLength> selectedBookTitle;
    if(iListboxView->GetSelectedItem(selectedBookTitle) != KErrNone)
        {
        _LIT(KErrorMsg,"Failed. Book not selected.");
        ShowNoteL(KErrorMsg);
        return;
        }
    TBuf<KBookItemMaxLength+32> prompt;
    TBuf<KBookItemMaxLength> newTitle;

    prompt.Append(KQuery);
    prompt.Append(selectedBookTitle);

    if(QueryTextL(prompt, newTitle))
        {
        newTitle.Trim(); // Remove specific characters from the beginning
                         // and the end of the string
        iBookDb->UpdateBookTitle(selectedBookTitle, newTitle);
        ShowAllBooksL();
        }
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::ShowAllDBMSsL()
//
// Get list of all Books in the database. Show the titles in the listbox
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::ShowAllBooksL()
    {
    _LIT(KAllBooksTitle,"Books in DBMS:");

    // Get array of full Book infos from the database. Construct array of
    // titles and show them in the listbox view.

    CDesCArrayFlat* Books = iBookDb->GetAllBooksL();
    CleanupStack::PushL(Books);
    CDesCArrayFlat* titles = TitlesArrayL(Books);
    CleanupStack::PushL(titles);

    ChangeViewL(EListView); // construct the listbox view
    iListboxView->SetCaptionL(KAllBooksTitle);
    CleanupStack::Pop(titles);
    iListboxView->SetListItemsL(titles); // Takes ownership
    CleanupStack::PopAndDestroy(Books);

    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::SearchBooksL()
//
// Query Book search string from the user and perform Book search. Show the
// results in the list.
//
// Implementation finds Books according to a KBooksTitleCol column name
// and the queried search pattern.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::SearchBooksL()
    {
    _LIT(KSearchResult, "Search result:");
    _LIT(KBookSearch, "Book search");
    _LIT(KWildCard, "*");

    TBuf<KTitleMaxLength> titlePattern;
    TBuf<32> prompt(KBookSearch);

    QueryTextL(prompt, titlePattern);
    titlePattern.Append(KWildCard);

    ChangeViewL(EListView);
    iListboxView->SetCaptionL(KSearchResult);

    // Get array of matching DBMSs. Construct array of titles from the
    // DBMSs array and show it in the listbox view.

    CDesCArrayFlat* Books =
        iBookDb->GetBooksByKeyL(KBooksTitleCol, titlePattern);
    CleanupStack::PushL(Books);

    CDesCArrayFlat* titles = TitlesArrayL(Books);
    iListboxView->SetListItemsL(titles); // Takes ownership

    CleanupStack::PopAndDestroy(Books);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::IndexFindL()
//
// Find a Book using index. Show full info for the Book.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::IndexFindL()
    {
    _LIT(KDetailsCaption,"Book details:");

    // Query the title of the selected Book in the listbox
    TBuf<KBookItemMaxLength> selectedBook;
    if(iListboxView->GetSelectedItem(selectedBook) != KErrNone)
    {
        _LIT(KErrorMsg,"Failed. Book not selected.");
        ShowNoteL(KErrorMsg);
        return;
    }

    // Query Book details from the database using Book title
    TBuf<KBookItemMaxLength> result;
    TInt err = iBookDb->GetABookFast(selectedBook, result);
    if(err==KErrNotFound)
        {
        _LIT(KNotFoundMsg,"Book not found.");
        ShowNoteL(KNotFoundMsg);
        return;
        }
    iEikonEnv->InfoWinL(KDetailsCaption, result);
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::AddDateColumn()
//
// Adds a date column to the Books table - if the column does not exist already
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::AddDateColumnL()
    {
        iBookDb->AddDateColumn();
        ShowColumnsL();
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::RemoveDateColumnL()
//
// Removes the date column from Books table - if the column exists
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::RemoveDateColumnL()
    {
        iBookDb->RemoveDateColumn();
        ShowColumnsL();
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::ShowColumns()
//
// Show the columns of the Book (Books table)
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::ShowColumnsL()
    {
    _LIT(KColumnsCaption,"Columns in Book:");
    ChangeViewL(EColumnsView); // construct the listbox view
    iListboxView->SetCaptionL(KColumnsCaption);
    CDesCArrayFlat* tmp = iBookDb->ColumnNamesAndSizesL();
    iListboxView->SetListItemsL(tmp); // takes ownership
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::ApplicationDriveAndPathL()
//
// Get the application path and drive. It must be done differently in the
// development environment and in the device.
// ---------------------------------------------------------------------------
//
TFileName CDBMSAppUi::ApplicationDriveAndPathL() const
    {
    TFileName appfullname(Application()->AppFullName());
    TParse parse;

#ifdef __WINS__   // See macro definition in DBMSDb.mmp

    // On development environment the AppFullName points to z drive.
    // Replace it to point to C drive, which is writable by our application.

    parse.Set(KCDrive, &appfullname, NULL);

#else // In device use the application fullname directly.

    parse.Set(appfullname, NULL, NULL);

#endif

    TFileName fn = parse.DriveAndPath();
    // Make sure the path exists (create if not). This is needed in EMULATOR.
    BaflUtils::EnsurePathExistsL(CCoeEnv::Static()->FsSession(), fn);
    return fn;
    }

// ---------------------------------------------------------------------------
// CDBMSAppUi::DatabaseDriveAndPathL()
// ---------------------------------------------------------------------------
//
TFileName CDBMSAppUi::DatabaseDriveAndPathL() const
{	
	RFs fsSession;
	User::LeaveIfError(fsSession.Connect());
	CleanupClosePushL(fsSession);
	
    TFileName appfullname, fn;
    appfullname = Application()->AppFullName();
	fsSession.PrivatePath(fn);

#ifdef __WINS__   // See macro definition in DBMS.mmp
	fn.Insert(0,KCDrive);

#else // In device use the application fullname directly.
    TParse parse;
    parse.Set(appfullname, NULL, NULL);
    fn.Insert(0,parse.Drive());   
#endif

    BaflUtils::EnsurePathExistsL(fsSession, fn);
	CleanupStack::PopAndDestroy(&fsSession);
	
    return fn;
}

// ---------------------------------------------------------------------------
// CDBMSAppUi::ShowNoteL()
//
// Show a note. Note that successive frequent calls to this method results in
// showing the latest message only.
// ---------------------------------------------------------------------------
//
void CDBMSAppUi::ShowNoteL(const TDesC& aMessage) const
    {

    CAknInformationNote* note = new(ELeave)CAknInformationNote;
    note->ExecuteLD(aMessage); // Deletes itself, when returns
    }


// ---------------------------------------------------------------------------
// CDBMSAppUi::TitlesArrayL()
//
// Build an array of Book titles from an array having Books with full info.
// ---------------------------------------------------------------------------
//
CDesCArrayFlat* CDBMSAppUi::TitlesArrayL(
    const CDesCArrayFlat* aFullDBMSInfoArray) const
    {
    // Assume the items within aFullDBMSInfoArray are in the format:
    // <Author>|<Title>|<Description>

    CDesCArrayFlat* resultArray =
        new (ELeave)CDesC16ArrayFlat(KArrayGranularity);
    CleanupStack::PushL(resultArray);

    TPtrC16 sourceRow;
    TInt startPos = 0;
    TInt endPos = 0;

    // Iterate through the DBMSs.
    // From each DBMS row, parse the <Title> and append it to result array.
    for(TInt i=0; i<aFullDBMSInfoArray->MdcaCount(); i++)
        {
        sourceRow.Set(aFullDBMSInfoArray->MdcaPoint(i));
        startPos = sourceRow.Locate('|') + 1; // exclude '|' from result
        endPos = sourceRow.LocateReverse('|');
        resultArray->AppendL(sourceRow.Mid(startPos, endPos-startPos));
        }
    CleanupStack::Pop(resultArray);
    return resultArray;
    }

// ---------------------------------------------------------------------------
// CDBMSAppUi::QueryTextL()
//
// Show simple text query dialos for the user
// ---------------------------------------------------------------------------
//
TBool CDBMSAppUi::QueryTextL(TDesC& aPrompt,
    TDes& aResultText) const
    {
   // Note: aPrompt cannot be const TDesC&, because CAknTextQueryDialog
   //       does not accept const TDesC& as a second parameter.

    CAknTextQueryDialog* dlg = new(ELeave)
        CAknTextQueryDialog( aResultText, // result is placed here
                             aPrompt,
                             CAknTextQueryDialog::ENoTone );

    dlg->SetMaxLength(aResultText.MaxLength());
    return dlg->ExecuteLD(R_SIMPLE_TEXT_QUERY);
    }

// ---------------------------------------------------------
// CDBMSAppView::HandleStatusPaneSizeChange()
// Called by framework when resource is changed.
// ---------------------------------------------------------
//
void CDBMSAppUi::HandleStatusPaneSizeChange()
	{
	CAknAppUi::HandleStatusPaneSizeChange(); //call to upper class

    if(iAppView)
    	iAppView->SetRect( ClientRect() );
    if(iListboxView)
       	iListboxView->SetRect( ClientRect() );
    if(iBookEditorView)
       	iBookEditorView->SetRect( ClientRect() );
	}
	
//EOF
