// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This class contains methods to create a database of integers and periodically read /write integers
// from/ to the database. The macro USE_SEMAPHORE is used at various places to show thread synchronization
// using a semaphore. 

#include "CDatabase.h"

//Declare a delay of 1000000 microseconds i.e. 1 sec for the periodic functions. 
const int KDelay = 1000000; 

//The last number in the database.
const int KLastNum = 50; 

/**
NewL()
*/
CDatabase* CDatabase::NewL()
    {
    CDatabase* self=new(ELeave)CDatabase();
    CleanupStack::PushL(self);
    self->ConstructL();
    CleanupStack::Pop(self);
    return self;
    }

/**
Destructor
*/
CDatabase::~CDatabase()
    {   
    iFsSession.Close();  //close the file server session. 
    delete iPeriodic;    //delete the periodic object.
    delete iConsole;     //delete the console. 
    #ifdef USE_SEMAPHORE
    iSemWrite.Close(); 
    #endif
    }

/**
Constructs a CPeriodic object, a database , opens the global semaphore.  
*/
void CDatabase::ConstructL()
    {
    iPeriodic = CPeriodic::NewL(CActive::EPriorityUserInput);
    
    _LIT(KDbCreateDb,"exampleDatabase.db");
      
    iFsSession.Connect();
    
    //The database exampleDatabase.db is created as a permanent file store 
    //in the app's private directory.
    User::LeaveIfError(iFsSession.CreatePrivatePath(RFs::GetSystemDrive()));

    User::LeaveIfError(iFsSession.PrivatePath(iDbname));
    iDbname.Append(KDbCreateDb); 

    #ifdef USE_SEMAPHORE
    //Open the global semaphore which was created in CSemaphoreExample. 
    _LIT(KSemaphoreWrite, "WriteSemaphore");
    User::LeaveIfError(iSemWrite.OpenGlobal(KSemaphoreWrite, EOwnerProcess)); 
    #endif
    }

/**
Creates database in the store and create a table and add columns to it. 
*/
void CDatabase::CreateDatabaseL()
    {
    //Construct a file store object - the file to contain the
    //database replaces any existing file of the same name.
    CFileStore* store = CPermanentFileStore::ReplaceLC(iFsSession, iDbname, EFileRead|EFileWrite);
    //Complete file store creation.
    store->SetTypeL(store->Layout());
    
    //Create a database in the store.  
    TStreamId id= iDatabase.CreateL(store);
    
    //Keep database id as root of store.
    store->SetRootL(id);
    
    //Complete database creation by commiting the store.
    store->CommitL();
    
    
    //Create a table definition.
    CDbColSet* columns=CDbColSet::NewLC();
    
    //Add three columns each containing Int32 values. 
    _LIT(KCol1, "Number1");
    _LIT(KCol2, "Number2");
    _LIT(KCol3, "Number3");   
    columns->AddL(TDbCol(KCol1,EDbColInt32));
    columns->AddL(TDbCol(KCol2,EDbColInt32));
    columns->AddL(TDbCol(KCol3,EDbColInt32));
    
    
    //Create the table, table name is "Numbers" and add the columns to it.
    _LIT(KTable, "Numbers");
    User::LeaveIfError(iDatabase.CreateTable(KTable,*columns));
    
    CDbKey* key=CDbKey::NewLC();
    
    //Add the key columns.
    TDbKeyCol number1(KCol1);
    key->AddL(number1);
    TDbKeyCol number2(KCol2);
    key->AddL(number2);
    TDbKeyCol number3(KCol3);
    key->AddL(number3);
    User::LeaveIfError(iDatabase.CreateIndex(KTable,KTable,*key));
    
    iDatabase.Close();
    //Cleanup the column set.  
    CleanupStack::PopAndDestroy(key);
    CleanupStack::PopAndDestroy(columns);
    CleanupStack::PopAndDestroy(store);
    }

/**
This function is called by the WriteThread in the SemaphoreExample and it 
starts a periodic function to write data in the database at an interval of 1 sec.  
*/
void CDatabase::WritetoDatabaseL()
    {
    iPeriodic->Start(0, KDelay, TCallBack(PeriodicWriteL, this));
    CActiveScheduler::Start();
    }

/**
The call back function associated with the CPeriodic object of the CDatabase class and is 
called by the WritetoDatabaseL() function.
*/
TInt CDatabase::PeriodicWriteL(TAny* aPtr)
    {
    CDatabase* ptr = static_cast<CDatabase*> (aPtr);
    //Invoke the RThread::Resume() function on the consumer thread repeatedly.
    ptr->PeriodicWriteFuncL();
    return KErrNone;
    }

/**
This function is called periodically at an interval of 1 sec, it opens the database, 
inserts one row and fills it with integer values. 
*/
void CDatabase::PeriodicWriteFuncL()
    {
    //Declare an integer which is used to fill numbers in the database. 
    static int num = 0 ;  
    
     //Open the file store.
    _LIT(KSQLStatement,"select Number1, Number2, Number3 from Numbers order by Number1,Number2,Number3");
    CFileStore* store = CFileStore::OpenLC(iFsSession,iDbname,EFileRead|EFileWrite);
    
    
    //Open the database from the root stream.
    iDatabase.OpenL(store,store->Root());
    
    //Lock the database before use. 
    iDatabase.Begin();
    
    //Create a view on the database view to read/write numbers into it. 
    RDbView view;
    User::LeaveIfError(view.Prepare(iDatabase,TDbQuery(KSQLStatement,EDbCompareNormal)));
    User::LeaveIfError(view.EvaluateAll());

    
    //Insert a new row at the end of the database. 
    view.InsertL();

    //Fill three coloumns with numbers.
    view.SetColL(1, num++);
    view.SetColL(2, num++);
    view.SetColL(3, num++);
    view.PutL();
    
    //Close the view. 
    view.Close();
    
    //Unlock the database after writing. 
    iDatabase.Commit();
    
    // close the database.
    iDatabase.Close();

    //Do not commit store: database has taken control of commit.
    CleanupStack::PopAndDestroy(store);
    
    //Cancel the periodic function once 50 numbers have been written to the database.
    if(num > KLastNum )
        {
        iPeriodic->Cancel();
        CActiveScheduler::Stop();
        }
    
    #ifdef USE_SEMAPHORE
    /*Signal the semaphore and the PeriodicReadFunc() which was 
      waiting on the semaphore is scheduled to run. 
      */
    iSemWrite.Signal();
    #endif
    }
    
/**
This function is called by the ReaderThread in SemaphoreExample. It creates a console to display 
the data read from the database and starts a periodic function to read the database at an interval 
of 1 second. 
*/
void CDatabase::ReadfromDatabaseL()
    { 
    _LIT(KTextConsoleTitle, "ReaderThread");
    if(!iConsole)
        {
         iConsole = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen)); 
         _LIT(KReading, "Reading the database.\n");
         iConsole->Printf(KReading);
        }   
    iPeriodic->Start(0, KDelay, TCallBack(PeriodicReadL,this));
    CActiveScheduler::Start();
    }

/**
The call back function associated with the CPeriodic object of the CDatabase class and is called
by the ReadfromDatabaseL() function. 
*/
TInt CDatabase::PeriodicReadL(TAny* aPtr)
    {
    CDatabase* ptr = static_cast<CDatabase*> (aPtr);
    ptr->PeriodicReadFuncL();
    return KErrNone;
    }

/**
This function is called periodically at an interval of 1 second, it opens the database , reads 
one row and displays the numbers to the console. 
*/
void CDatabase::PeriodicReadFuncL()
    {
    #ifdef USE_SEMAPHORE
    //Semaphore waits till it gets a signal from the PeriodicWrite() function. 
    iSemWrite.Wait();
    #endif
    
    CFileStore* store = CFileStore::OpenLC(iFsSession,iDbname,EFileRead);  
    
    //Open the database from the root stream.
    iDatabase.OpenL(store,store->Root());
    
    //Lock the database.
    iDatabase.Begin();
    
    //Prepare an SQL query to read three numbers from the database. 
    _LIT(KSQLStatement,"select Number1,Number2,Number3  from Numbers order by Number1,Number2, Number3");

    //Create a view on the database.
    RDbView view; 
    User::LeaveIfError(view.Prepare(iDatabase,TDbQuery(KSQLStatement,EDbCompareNormal)));
    User::LeaveIfError(view.EvaluateAll());
    
    //Declare integers to iterate through the database rows. 
    static int count = 0; 
    TInt iter = 0; 
    
    /*Iterate through the database to read only the row which was most recently  
      written by the PeriodicWriteFunc.
      */
    view.FirstL();
    while( iter < count)
        {
        view.NextL();
        iter++; 
        }
    count ++ ; 
     
    //Read the integers from the view and display in console. 
    view.GetL();
    TInt number1 = view.ColInt(1);
    TInt number2 = view.ColInt(2);
    TInt number3 = view.ColInt(3);
    //Prepare a row formatter to print the numbers to the console. 
    _LIT(KRowFormatter, "\n reading  %d \t%d\t%d");
    
    iConsole->Printf(KRowFormatter,number1,number2, number3);
    
    //Cancel the periodic function after it has read the last number in the database.  
    if(number3 >= KLastNum)
        {
        iPeriodic->Cancel();
        CActiveScheduler::Stop();
        _LIT(KAllRead, "\nAll the numbers in the database have been read\n");
        iConsole->Printf(KAllRead);
        _LIT(KExit, "Press a key to exit\n");
        iConsole->Printf(KExit);
        iConsole->Getch();
        }     
    //Close the view. 
    view.Close();
    
    //Unlock the database. 
    iDatabase.Commit();
    
    iDatabase.Close();
    CleanupStack::PopAndDestroy(store);
    }
