// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the E32Main() function, which executes the example. The macro USE_SEMAPHORE is used at various
// places to show synchronization mechanism using semaphores.

 
#include "SemaphoreExample.h"

/**
Destructor
*/
CSemaphoreExample::~CSemaphoreExample()
    {
    iReadThread.Close();//Reader thread closed. 
    iWriteThread.Close();//Writer thread closed. 
    delete iConsole;
    }

/**
NewL()
*/
CSemaphoreExample* CSemaphoreExample::NewL()
    {
    CSemaphoreExample* self=new(ELeave)CSemaphoreExample();
    CleanupStack::PushL(self);
    self->ConstructL();
    CleanupStack::Pop(self);
    return self;
    }

/**
Creates and resumes two threads, one for reading from the database and other for writing into the database. 
Prints messages to console depending on whether the USE_SEMAPHORE macro is enabled or disabled. 
*/
void CSemaphoreExample::ConstructL()
    { 
    //Create a reader thread to read the database.
    _LIT(KReader,"ReaderThread");
    User::LeaveIfError(iReadThread.Create(KReader,ReadThreadFuncL,KDefaultStackSize,KMinHeapSize,256*KMinHeapSize,this,EOwnerProcess));   

    //Create a writer thread to write into the database. 
    _LIT(KWriter,"WriterThread");
    User::LeaveIfError(iWriteThread.Create(KWriter,WriteThreadFuncL,KDefaultStackSize,KMinHeapSize,256*KMinHeapSize,this,EOwnerProcess));           

    //Sets the priorities of the threads when not using a semaphore, but this is not a good practice to achieve serialized execution; the preferred way
    //should be using semaphores. If USE_SEMAPHORE is used, create a global semaphore. 
    #ifndef USE_SEMAPHORE
    iWriteThread.SetPriority(EPriorityMore);
    iReadThread.SetPriority(EPriorityLess);
    #else
    _LIT(KSemaphoreWrite, "WriteSemaphore");
    User::LeaveIfError(iSem.CreateGlobal(KSemaphoreWrite, 0, EOwnerProcess));
    #endif
    }

/**
Starts iReadThread and iWriteThread. 
*/
void CSemaphoreExample::StartThreads()
    {
    TRequestStatus status1, status2; 
    
    _LIT(KPressKey, "\nPress any key to start writing to and reading from the database.\n");
    iConsole->Printf(KPressKey);
    iConsole->Getch();
    //Request a notification for iReadThread to terminate. 
    iReadThread.Logon(status2);
    iReadThread.Resume();
    
    //Request a notification for iWriteThread to terminate. 
    iWriteThread.Logon(status1);   
    iWriteThread.Resume();
    
    //Control returns to the main thread when the two threads terminate. 
    User::WaitForRequest(status1);
    User::WaitForRequest(status2);    
    }

void CSemaphoreExample::PrintMessage()
    {
    _LIT(KTextConsoleTitle, "SemaphoreExample");
    iConsole = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen));  
    _LIT(KWelcome, "Welcome to the SemaphoreExample.\n");
    
    iConsole->Printf(KWelcome);
    #ifdef USE_SEMAPHORE 
    _LIT(KSemDefined, "The USE_SEMAPHORE macro is defined.\n"); 
    iConsole->Printf(KSemDefined); 
    #else  
    _LIT(KSemUnDefined, "The USE_SEMAPHORE macro is undefined.\n"); 
    iConsole->Printf(KSemUnDefined); 
    #endif
    
    _LIT(KPurposel1, "There are two threads running in the example: WriterThread calls a periodic function\n");
    _LIT(KPurposel2, "which adds rows of integers to a database at an interval of 1 second.\n");  
    
    iConsole->Printf(KPurposel1);
    iConsole->Printf(KPurposel2);
    
    
    #ifdef USE_SEMAPHORE  
    
    _LIT(KPurposel4, "ReaderThread waits for the WriterThread to complete inserting one row of integers\n");
    _LIT(KPurposel5, "and then reads the data just written.\n");
    
    iConsole->Printf(KPurposel4);
    iConsole->Printf(KPurposel5);
    #else   
    _LIT(KPurposel31, "ReaderThread also calls a periodic function to read each row of integers\n");
    _LIT(KPurposel32, "and displays them in the console.\n");
    iConsole->Printf(KPurposel31);
    iConsole->Printf(KPurposel32);
    #endif
    
    #ifdef USE_SEMAPHORE  
    _LIT(KSemaphoreNote, "RSemaphore is used to synchronize the two threads.\n"); 
    iConsole->Printf(KSemaphoreNote);
    #else
    _LIT(KNoSemaphoreNotel, "This implementation does not use Semaphores, instead it uses thread priorities and doesn't\n");
    _LIT(KNoSemaphoreNote2, "guarantee correct thread synchronization.\n");
    iConsole->Printf(KNoSemaphoreNotel);
    iConsole->Printf(KNoSemaphoreNote2);
    #endif
    }

/**
This function installs an active scheduler , creates a CDatabase object to read the 
database. 
*/
static void ReadDbFuncL()
    {
    CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
    CleanupStack::PushL(scheduler);
    CActiveScheduler::Install(scheduler);
    
    CDatabase* db = CDatabase::NewL();
    CleanupStack::PushL(db);
    
    //Start reading the database. 
    db->ReadfromDatabaseL();
    
    CleanupStack::PopAndDestroy(db);
    CleanupStack::PopAndDestroy(scheduler);
    }

/**
This function is called when iReadThread is resumed. It creates a cleanup stack and 
calls ReadDbFunc(). 
*/
TInt CSemaphoreExample::ReadThreadFuncL(TAny* /*aPtr*/)
    {
    __UHEAP_MARK;
    //Create cleanup stack.
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }
     
    TRAPD(error, ReadDbFuncL()); 
    if(error != KErrNone)
        {
        _LIT(KUserPanic,"Failed");  
        User::Panic(KUserPanic, error);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }

/**
This function installs an active scheduler , creates a CDatabase object to write into the 
database. 
*/
static void WriteDbFuncL()
    {
    CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
    CleanupStack::PushL(scheduler);
    CActiveScheduler::Install(scheduler);
    
    CDatabase* db = CDatabase::NewL(); 
    CleanupStack::PushL(db);
    //Create a database.
    db->CreateDatabaseL();
    
    //Write into the database. 
    db->WritetoDatabaseL();
    
    CleanupStack::PopAndDestroy(db);
    CleanupStack::PopAndDestroy(scheduler);
    }

/**
This function is called when iWriteThread is resumed. It creates a cleanup stack and 
calls WriteDbFunc(). 
*/
TInt CSemaphoreExample::WriteThreadFuncL(TAny* /*aPtr*/)
    {
    __UHEAP_MARK;
    //Create cleanup stack.
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }

    TRAPD(err, WriteDbFuncL());
    if(err != KErrNone)
        {
        _LIT(KUserPanic,"Failed");  
        User::Panic(KUserPanic, err);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }

/**
Main function which creates an object of CSemaphoreExample. 
*/
static void MainL()
    {
    CSemaphoreExample* semExample = CSemaphoreExample::NewL();
    CleanupStack::PushL(semExample);  
    semExample->PrintMessage();
    semExample->StartThreads();
    CleanupStack::PopAndDestroy(semExample);
    }

/**
Entry point for the example
*/
extern TInt E32Main()
    {
    //Create cleanup stack.
    __UHEAP_MARK;
    CTrapCleanup* cleanup = CTrapCleanup::New();
    if(!cleanup)
        {
        return KErrNoMemory;
        }
    //Run application code inside a TRAP harness.
    TRAPD(mainError, MainL());
    if(mainError != KErrNone)
        {
        _LIT(KUserPanic,"Failed");  
        User::Panic(KUserPanic, mainError);
        }
    delete cleanup;
    __UHEAP_MARKEND;
    return KErrNone;
    }
