// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This example program demonstrates how to use the resizable buffer descriptor class RBuf.
//



/**
 @file 
*/
#include "rbufexample.h"

const TInt KLargeBufSize = 15;
const TInt KSmallBufSize = 3;
const TInt KMedBufSize = 7;

_LIT(KFailed, "\n Error occurred");
_LIT(KPressAKey, "\n\n Press any key to continue the example\n");
_LIT(KLength,"\n  Length= %d");
_LIT(KMaxLength,"\n  MaxLength= %d");
_LIT(KBuffer1,"11111");
_LIT(KBuffer2,"22222");

/**
Constructor
*/
CRBufExample::CRBufExample()
	{
	}

void CRBufExample::ConstructL()
	{
	_LIT(KTitle," RBuf Example");
	iConsole = Console::NewL(KTitle, TSize(KConsFullScreen, KConsFullScreen));
	
	_LIT(KWelcome," Welcome to the RBuf example application");
	iConsole->Printf(KWelcome);
	
	_LIT(KPressAKeyMsg, "\n\n Press any key to step through the example\n");
	iConsole->Printf(KPressAKeyMsg );
	iConsole->Getch();
	}

/**
Destructor
*/
CRBufExample::~CRBufExample()
	{
	delete iConsole;
	}

/**
Allocates and constructs a CRBufExample object.
@return A CRBufExample object.
*/
CRBufExample* CRBufExample::NewL()
	{
	CRBufExample* self=new(ELeave)CRBufExample();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}
    
/**
Creates an empty resizable buffer descriptor using Create() and CreateMax().
*/
void CRBufExample::CreateRBufL()
	{
	_LIT(KCreate,"\n Creating an RBuf: \n");

	iConsole->Printf(KCreate);
	
	RBuf buf;
	
	// Create an empty buffer descriptor
	// The length is zero, the maxlength is 3
	User::LeaveIfError(buf.Create(KSmallBufSize));
	CleanupClosePushL(buf);
	
	// Get the length of the buffer
	TInt length= buf.Length();
	
	// Get the maximum length of the buffer
	TInt maxLength= buf.MaxLength();
	
	// Check that length and maximum length are not equal because it was created using Create()
	if(length != maxLength)
		{
		_LIT(KUsingCreate, "\n Using Create() API: ");
		iConsole->Printf(KUsingCreate);
		iConsole->Printf(KLength, length);
		iConsole->Printf(KMaxLength, maxLength);
		}
	else
		{
		iConsole->Printf(KFailed);
		}	

	// Deallocate the memory assigned to the buffer
	CleanupStack::PopAndDestroy(&buf);
	
	// Create buffer descriptor
	// The length is 3, the maxlength is 3
	User::LeaveIfError(buf.CreateMax(KSmallBufSize));
	CleanupClosePushL(buf);
	
	// Get the length of the buffer.
	length= buf.Length();
	
	// Get the maximum length of the buffer.
	maxLength=buf.MaxLength();
	
	// Check that length and maximum length are equal because it was created using CreateMax().
	if(length == maxLength)
		{
		_LIT(KCreateInAdvance, "\n Using CreateMax() API: ");
		iConsole->Printf(KCreateInAdvance);	
		iConsole->Printf(KLength, length);
		iConsole->Printf(KMaxLength, maxLength);
		}
	else
		{
		iConsole->Printf(KFailed);
		}	
	
	// Deallocate the memory assigned to the buffer.
	CleanupStack::PopAndDestroy(&buf);
	} 
	
/**
Creates a resizable buffer descriptor with data copied from another descriptor.
*/
void CRBufExample::CreateRBufFromExistingDesL()
	{
	RBuf buf;
	
	// Create a buffer descriptor, initialised with some text
	buf.CreateL(KBuffer1);
	
	_LIT(KCreateFromExistingDes, "\n From an existing descriptor ");
	iConsole->Printf(KCreateFromExistingDes);

	// Deallocates the memory assigned to the buffer
	buf.Close();
	} 
	
/**
Creates a resizable buffer descriptor by assigning data from an HBufC.
*/
void CRBufExample::CreateRBufFromHBufCL()
	{
	// Create a pointer to a heap descriptor
	HBufC* hptr = HBufC::NewL(KBuffer1().Length());
	
	// Assign some data to the heap descriptor.
	*hptr = KBuffer1; 
	
	RBuf buf;
	
	// Assign the HBufC to the buffer. This transfers ownership of the heap descriptor.
	buf.Assign(hptr);
	
	_LIT(KCreateFromHBufC, "\n From HBufC ");
	iConsole->Printf(KCreateFromHBufC);
	
	// Deallocate the memory assigned to the buffer. 
	// There is no need to free the HBufC because Assign() transferred ownership.
	buf.Close(); 
	}
	
/**
Creates a resizable buffer descriptor from another RBuf.
*/
void CRBufExample::CreateRBufFromAnotherRBufL()
	{
	RBuf buf;
	
	// Create a buffer descriptor, initialised with some text
	User::LeaveIfError(buf.Create(KBuffer1));
		
	RBuf targetBuf;
	
	// Assign one buffer to the other. 
	// Transfers ownership of the source buffer's memory to the target buffer.
	// Note that targetBuf should have no memory allocated to it beforehand
	// otherwise a memory leak will occur.
	targetBuf.Assign(buf);
	
	_LIT(KCreateFromAnotherRBuf, "\n From another RBuf");
	iConsole->Printf(KCreateFromAnotherRBuf);

	// Deallocate the memory assigned to the buffer.
	targetBuf.Close();
	}
		
/**
Creates a resizable buffer descriptor by reading data from an RReadStream.
*/
void CRBufExample::CreateRBufUsingRReadStreamL()
	{
	// A handle to a file server session. 
	// This is used to create and write to/read from a file.
	RFs fs;
	
	// Connect to the file server.
	User::LeaveIfError(fs.Connect());
	CleanupClosePushL(fs);
	// Create the session path at the process's private path on the system drive.
	User::LeaveIfError(fs.CreatePrivatePath(RFs::GetSystemDrive()));

	// Set the session path to point to the process's private path on the system drive.
	User::LeaveIfError(fs.SetSessionToPrivate(RFs::GetSystemDrive()));
	
	// Declare a file stream to write to.
	RFileWriteStream wStream;
	
	_LIT(KFileName,"stream.dat");
	
	// Create a file, associates it with the stream, and prepares the stream for writing.
	User::LeaveIfError(wStream.Replace(fs, KFileName, EFileWrite));
	CleanupClosePushL(wStream);
	
	_LIT(KText, "RBuf Example");

	// Write some text to the stream.
	wStream << KText();
	wStream.CommitL();
	CleanupStack::PopAndDestroy(&wStream);
	
	// Declare a file stream to read from.
	RFileReadStream rStream;
	
	// Open the file
	User::LeaveIfError(rStream.Open(fs, KFileName, EFileRead));
	CleanupClosePushL(rStream);
	
	RBuf buf;
	CleanupClosePushL(buf);
	
	// Create the buffer by passing the open stream object to CreateL().
	buf.CreateL(rStream, KLargeBufSize);
	
	_LIT(KCreateUsingRReadStream, "\n Using RReadStream ");
	iConsole->Printf(KCreateUsingRReadStream);
	
	// Deallocate the memory assigned to the buffer, close the stream and close the file server connection
	CleanupStack::PopAndDestroy(&buf);
	CleanupStack::PopAndDestroy(&rStream);
	CleanupStack::PopAndDestroy(&fs);
	}

/**
Creates a resizable buffer descriptor RBuf from some previously allocated memory.
*/	
void CRBufExample::CreateRBufFromAllocatedMemoryL()
	{
	RBuf buf;
	
	// Allocate some memory.
	TUint16* allocatedMemory = static_cast<TUint16*>( User::Alloc( KLargeBufSize * sizeof( TUint16) ) ) ;

	// Transfer ownership of the memory to the descriptor.
	buf.Assign(allocatedMemory, KLargeBufSize);
	
	_LIT(KCreateFromAllocatedMemory, "\n By transferring ownership of allocated memory");
	iConsole->Printf(KCreateFromAllocatedMemory);
	
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	
	// Deallocate the memory assigned to the buffer.
	// There is no need to free memory here because Assign() transferred ownership.
	buf.Close(); 
	}

/**
Swaps the contents of two resizable buffer descriptors.
*/	
void CRBufExample::SwapTwoRBufsL()
	{
	_LIT(KSwapAndCopy,"\n Swapping and copying data: ");
	iConsole->Printf(KSwapAndCopy);
	
	RBuf buf1;
	
	// Create a buffer descriptor, initialised with some text
	User::LeaveIfError(buf1.Create(KBuffer1));
	CleanupClosePushL(buf1);
	
	RBuf buf2;
	
	// Create a second buffer descriptor, initialised with some text
	User::LeaveIfError(buf2.Create(KBuffer2));
	CleanupClosePushL(buf2);
	
	_LIT(KBeforeSwapping, "\n Before swapping: ");
	iConsole->Printf(KBeforeSwapping);
	
	// Print out the contents of the 2 descriptors
	_LIT(KPrintFirstdata,"\n  Data present in first descriptor is: ");
	iConsole->Printf(KPrintFirstdata);	
	iConsole->Printf(buf1);
	
	_LIT(KPrintSecondData,"\n  Data present in second descriptor is: ");
	iConsole->Printf(KPrintSecondData);	
	iConsole->Printf(buf2);
	
	// Swap them and print out the new contents
	buf1.Swap(buf2);
	
	_LIT(KAfterSwapping, "\n After swapping: ");
	iConsole->Printf(KAfterSwapping);
	
	iConsole->Printf(KPrintFirstdata);	
	iConsole->Printf(buf1);
	
	iConsole->Printf(KPrintSecondData);	
	iConsole->Printf(buf2);
	
	_LIT(KSwap, "\n Swapping between two RBufs is successful");
	iConsole->Printf(KSwap);
			
	// Deallocate memory assigned to the 2 buffer descriptors.
	CleanupStack::PopAndDestroy(2); // buf1, buf2
	}
	
/**
Copies data using the assignment operator from one descriptor to another 
*/	
void CRBufExample::CopyDataUsingAssignmentOperatorL()
	{
	RBuf buf1;
	
	// Create a buffer descriptor, initialised with some text
	buf1.Create(KBuffer1);
	CleanupClosePushL(buf1);

	RBuf buf2;
	
	// Create a second buffer descriptor, initialised with some text
	User::LeaveIfError(buf2.Create(KBuffer2));
	CleanupClosePushL(buf2);

	// Copy data using assignment operator from one to the other. 
	// Target buffer must be long enough, otherwise a panic occurs.
	buf2= buf1;
	
	// Check the value of the buffers .
	if (buf1==buf2)
		{
		_LIT(KCopyDataUsingAssignmentOperator,"\n Copying data using assignment operator from descriptor, and RBuf is successful");
		iConsole->Printf(KCopyDataUsingAssignmentOperator);
		}
	else
		{
		iConsole->Printf(KFailed);	
		}	
	
	iConsole->Printf(KPressAKey);
    iConsole->Getch();
	
	// Deallocate memory assigned to the 2 buffer descriptors.
	CleanupStack::PopAndDestroy(2); // buf1, buf2
	}
	
/**
Reallocates and frees the resizable buffer descriptor.
*/
void CRBufExample::ReallocateAndFreeTheMemoryBufferL()
	{
	_LIT(KReAllocAndFree,"\n Realloc and free:");
	iConsole->Printf(KReAllocAndFree);
	
	RBuf buf;
	
	// Create empty buffer descriptor with max length of KSmallBufSize.
	User::LeaveIfError(buf.Create(KSmallBufSize));
	
	// Get the length of the buffer.
	TInt length= buf.Length();
	
	// Get the maximum length of the buffer.
	TInt maxLength= buf.MaxLength();
	
	_LIT(KBeforeReAlloc,"\n Before ReAlloc: ");
	iConsole->Printf(KBeforeReAlloc);
	
	// Print out the length and max length
	iConsole->Printf(KLength, length);
	iConsole->Printf(KMaxLength, maxLength);	

	// Re-allocate the buffer descriptor. Length doesnt change but maxlength does.
	User::LeaveIfError(buf.ReAlloc(KMedBufSize));
	
	// Get the length of the buffer.
	length= buf.Length();
	
	// Get the maximum length of the reallocated buffer.
	maxLength= buf.MaxLength();
	
	_LIT(KAfterReAlloc,"\n After ReAlloc: ");
	iConsole->Printf(KAfterReAlloc);

	// Print out the length and new max length
	iConsole->Printf(KLength, length);
	iConsole->Printf(KMaxLength, maxLength);	
	
	// Free the memory buffer. It sets the length and maxlength to zero.
	User::LeaveIfError(buf.ReAlloc(0));
	
	// Get the length of the buffer.
	length= buf.Length();
	
	// Get the maximum length of the buffer.
	maxLength= buf.MaxLength();
	
	_LIT(KFreeBuffer,"\n After free: ");
	iConsole->Printf(KFreeBuffer);
	
	iConsole->Printf(KLength, length);
	iConsole->Printf(KMaxLength, maxLength);
	}
	
/**
Replaces and modifies a portion of the data present in an RBuf.
*/
void CRBufExample::ReplaceAndModifyTheDataL()
	{
	RBuf buf;
	
	// Create a buffer descriptor, initialised with some text
	User::LeaveIfError(buf.Create(KBuffer1)); 
	CleanupClosePushL(buf);
			
	_LIT(KReplaceAndModify,"\n Replace and modify: ");
	iConsole->Printf(KReplaceAndModify);
	
	// Print the original data present in the buffer.
	_LIT(KOriginalData,"\n Data present in RBuf is: ");
	iConsole->Printf(KOriginalData);
 	iConsole->Printf(buf);
 	
 	TInt pos= 1;
 	
 	_LIT(KReplacedMessage,"22");
 	
 	TInt length= KReplacedMessage().Length();

 	// Replace a portion of the data in the buffer descriptor.
 	buf.Replace(pos,length, KReplacedMessage);
 	
 	// Print the buffer's new contents.
 	_LIT(KReplacedData,"\n After replacement, data held in RBuf is: ");
 	iConsole->Printf(KReplacedData);
 	iConsole->Printf(buf);
 	
 	pos=3;
 	length=1;
 	
 	// Delete the replacement text.
    buf.Delete(pos,length);
    
    // Print the buffer's contents again.
    _LIT(KModifiedData,"\n After modification, data held in RBuf is: ");
    iConsole->Printf(KModifiedData);
    iConsole->Printf(buf);
    
    _LIT(KReplaceAndModifyData,"\n Replacing and modifying the data held in RBuf is successful");
    iConsole->Printf(KReplaceAndModifyData);
    
    // Deallocate the memory assigned to the buffer. 
	CleanupStack::PopAndDestroy(&buf);
	}
	
/**
Demonstrates cleanup for resizable buffer descriptors.
*/
void CRBufExample::CleanUpRulesL()
	{
	RBuf buf;
	
	// Create a buffer descriptor, initialised with some text
	User::LeaveIfError(buf.Create(KBuffer1));
	
	// To avoid memory leaks use RBuf::CleanupClosePushL() to push a cleanup item for the RBuf onto the cleanup stack.
	// The effect is to cause Close() to be called on the buffer descriptor if a leave occurs, 
	// or when CleanupStack::PopAndDestroy() is called.
	buf.CleanupClosePushL();
	
	// Causes Close() to be called to deallocate the memory assigned to the buffer.
	CleanupStack::PopAndDestroy(&buf);
	
	_LIT(KCleanUp,"\n RBuf cleanup is successful");
	iConsole->Printf(KCleanUp);
	
	_LIT(KExitMsg, "\n\n Press any key to exit the example");
    iConsole->Printf(KExitMsg);
	iConsole->Getch();
	}	

void MainL()
	{
	CRBufExample* app= CRBufExample::NewL();
	CleanupStack::PushL(app);
 	
 	// Creates empty RBuf.
	app->CreateRBufL();
	
	// Creates RBuf from an existing descriptor.
	app->CreateRBufFromExistingDesL(); 
	
	// Creates RBuf from an HBufC.
	app->CreateRBufFromHBufCL();
	
	// Creates RBuf from another RBuf.
	app->CreateRBufFromAnotherRBufL();
	
	// Create RBuf using RReadStream
	app->CreateRBufUsingRReadStreamL();
	
	// Creates RBuf from allocated memory.
	app->CreateRBufFromAllocatedMemoryL();
		
	// Swaps two RBufs.
	app->SwapTwoRBufsL();
	
	// Copies data using assignment operator.
	app->CopyDataUsingAssignmentOperatorL();
	
	// Reallocates and frees the memory buffer.
	app->ReallocateAndFreeTheMemoryBufferL();
	
	// Replaces and modifies the data in the descriptor.
	app->ReplaceAndModifyTheDataL();
	
	// Demonstrates cleanup.
	app->CleanUpRulesL();
	
	CleanupStack::PopAndDestroy(app);
	} 

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;

	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	TRAPD(err, MainL());
	delete cleanup;

	if(err !=KErrNone)
		{
		User::Panic(KFailed, err);
		}	
		
	__UHEAP_MARKEND;
	return KErrNone;
	}
  
  
  
  
 
 
 
