// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This example program demonstrates the use of hash table classes.
//



/**
 @file 
*/
#include "hashtableexample.h"
#include <e32hashtab.h>
#include <e32math.h>

/**
Constructor
*/
CHashTableExample::CHashTableExample()
	{
	}

void CHashTableExample::ConstructL()
	{
	iConsole = Console::NewL(KTitle, TSize(KConsFullScreen, KConsFullScreen));
	iConsole->Printf(KWelcome);
	iConsole->Printf(KPressAKeyMsg );
	iConsole->Getch();
	}

/**
Destructor
*/
CHashTableExample::~CHashTableExample()
	{
	delete iConsole;
	iPointerArray.ResetAndDestroy();
	}

/**
Allocates and constructs a CHashTableExample object.
Initialises all member data to their default values.
@return A CHashTableExample object
*/
CHashTableExample* CHashTableExample::NewL()
	{
	CHashTableExample* self=new(ELeave)CHashTableExample();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}
    
/**
Constructs a hash set using the default hash and identity functions for integers
and for descriptors. 
*/
void CHashTableExample::ConstructDefaultHashSet()
	{
	// Construct hash set using default hash and identity functions for integer
	RHashSet<TInt> hashSetInt();
  
	// Construct hash set using default hash and identity functions for 8bit descriptor
	RHashSet<TDesC8> hashSetDes8();
  
	// Construct hash set using default hash and identity functions for 16bit descriptor
	RHashSet<TDesC16> hashSetDes16();
  
	iConsole->Printf(KHashSet);
	iConsole->Printf(KConstruct);
	iConsole->Printf(KConstructDefaultHashSet);
	} 
   
/**
Creates a custom structure containing two integer variables 
*/
struct TMyOwnObject
	{
	TInt iVar1;
	TInt iVar2;
	};
 
/**
This is a simple implementation of a hash function for a compound object. 
It is not intended as an example of an optimum hash function implementation
@param aObject Custom structure object
@return Integer value of hash function for two values of a specified custom structure objects 
*/
TUint32 MyHashFunction(const TMyOwnObject& aObject)
	{
	return	DefaultHash::Integer(aObject.iVar1) + DefaultHash::Integer(aObject.iVar2 );
	}
    
/**
Custom identity function in arbitary implementation to check the identity relation between the 
values of two custom structure objects.
@param aObj1 Custom structure object
@param aObj2 Custom structure object
@return Boolean value of identity function for two custom structure objects
*/
TBool MyIdentityFunction(const TMyOwnObject& aObject1, const TMyOwnObject& aObject2)
	{
	return aObject1.iVar1 == aObject2.iVar1 && aObject1.iVar2 != aObject2.iVar2;
	}
    
/**
Constructs a hash set using the custom hash and identity functions to demonstrate the options. 
The objects declared aren't used elsewhere in the example.
*/
void CHashTableExample::ConstructOwnHashSet()
	{
	// Creates an object of custom hash function by using our created structure
	THashFunction32<TMyOwnObject> ownHashFunction(MyHashFunction);

	// Creates an object of identity function by using our created structure
	TIdentityRelation<TMyOwnObject> ownIdentityFunction(MyIdentityFunction);	
 
	// Construct hash set by providing custom hash and identity function
	RHashSet<TMyOwnObject> ownHashSet(ownHashFunction, ownIdentityFunction);
	iConsole->Printf(KConstructOwnHashSet);
	}
 
/**
Applies various operations to hash set:
- Inserting items
- Finding items
- Iterating items
- Removing items
- closing and cleanup items
@leave KErrNoMemory An attempt to allocate memory has failed
@leave KErrNotFound Item not found
*/
void CHashTableExample::OperationsToHashSetL()	
	{
	// Declare a const integer value for the initial item to be stored
    const TInt startItem=1;
    
    // Declare a const integer value for the last item to be stored	
	const TInt endItem=100;
	TInt itemToBeFound=120;
	TInt itemToBeRemoved=200;
	TInt64 items=1;
  
	iConsole->Printf(KOperation);
	
	// Creates an object of hash set using the template class RHashSet 
	RHashSet<TInt> hashSet;
	
	// Push hash set on to the cleanup stack
	CleanupClosePushL(hashSet);
 
	// Insert random items to hash set
	for (TInt i=startItem; i<=endItem; ++i)
		{
		TInt res = Math::Rand(items);
		hashSet.InsertL(res);
		}
   
	iConsole->Printf(KInsertItemsToHashSet);
  
	// Search the set for a specified item 
	TInt* result= hashSet.Find(itemToBeFound);
	
	// result is NULL if specified item is not found in the set
	if(result)
		{
		iConsole->Printf(KItemPresentInHashSet);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInHashSet);
		}
 
	// Creates an object of TIter to iterate over the elements of hash set
	RHashSet<TInt>::TIter hashSetIter(hashSet);
  
	// Iterate over the items in the set
	// Creates an iterator object of type TIter
	for ( ; ;)
		{
		const TInt* res = hashSetIter.Next();
		
		// Next() moves the iterator to the next item and returns it
		// Returns NULL if there are no more items 
		if (!res)
			{
			break;
			}
		}
  
	iConsole->Printf(KIterateItemsFromHashSet);

	// Remove an item from hash set
	TInt res = hashSet.Remove(itemToBeRemoved);
	// Check if the item was successfully removed
	if(res)
		{
		iConsole->Printf(KItemPresentInHashSet);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInHashSet);
		}
  
	iConsole->Printf(KRemoveItemsFromHashSet);
  
	// Close and cleanup hash set
	CleanupStack::PopAndDestroy(&hashSet);
	
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	}
  
/**
Constructs a hash set of pointers using the default hash and identity functions for integers
and for descriptors
*/
void CHashTableExample::ConstructDefaultPtrHashSet()
	{
	// Construct hash set of pointers using default hash and identity functions for integer
	RPtrHashSet<TInt> ptrHashSetInt();
  
	// Construct hash set of pointers using default hash and identity functions for 8bit descriptor
	RPtrHashSet<TDesC8> ptrHashSetDes8();
  
	// Construct hash set of pointers using default hash and identity functions for 16bit descriptor
	RPtrHashSet<TDesC16> ptrHashSetDes16();
 
	iConsole->Printf(KPtrHashSet);
	iConsole->Printf(KConstruct);
	iConsole->Printf(KConstructDefaultPtrHashSet);
	}
  
/**
Constructs hash set of pointers using the custom hash and identity functions to demonstrate the options. 
The objects declared aren't used elsewhere in the example.
*/
void CHashTableExample::ConstructOwnPtrHashSet()
	{
	// Creates an object of custom hash function by using our created structure
	THashFunction32<TMyOwnObject> ownHashFunction(MyHashFunction);
 
	// Creates an object of identity relation by using our created structure
	TIdentityRelation<TMyOwnObject> ownIdentityFunction(MyIdentityFunction);	
 
	// Construct hash set of pointers by providing custom hash and identity functions
	RPtrHashSet<TMyOwnObject> ownPtrHashSet(ownHashFunction, ownIdentityFunction);
	iConsole->Printf(KConstructOwnPtrHashSet);	
	}
 
/**
Converts the specified number into words
@param aNum Integer to convert
@param aDes Reference to the descriptor containing the result in words
*/
void FindNumberInWords(const TInt& aNum, TDes& aDes)
	{
	TInt number = aNum;
	const TInt bufferSize=256;
	const TText* numbers[] = {_S("zero"), _S("one"), _S("two"),_S("three"),_S("four"),_S("five"),_S("six"),_S("seven"), 
							_S("eight"),_S("nine"),_S("ten"),_S("eleven"),_S("twelve"),_S("thirteen"), 
							_S("fourteen"),_S("fifteen"), _S("sixteen"),_S( "seventeen"),_S( "eighteen"),
							_S("nineteen"),_S( "twenty"),_S( "thirty"),_S( "forty"),_S( "fifty"),_S("sixty"),
							_S("seventy"),_S( "eighty"), _S("ninety"), _S("hundred"), _S("thousand")  };
                           
	// Converts the words if the number is less than 20
	if (number<20)
		{
		aDes.Copy(reinterpret_cast<const TUint16*> (numbers[number]));
		}
    
	// Converts the words if the number between 20 and 100 
	if (number<100 && number>=20)
		{
		TInt tens = number/10;
		TInt units = number%10;
		aDes.Copy(reinterpret_cast<const TUint16*> (numbers[tens-2+20]));
		if (units)
			{
			aDes.Append(' ');
			aDes.Append(TPtrC16(reinterpret_cast<const TUint16*> (numbers[units])));
			}
		}
   
	// Converts the words if the number is between 100 and 1000 
	if (number<1000 && number>=100)
		{
		TInt hundreds = number/100;
		aDes.Copy(reinterpret_cast<const TUint16*> (numbers[hundreds]));
		aDes.Append(' ');
		aDes.Append(TPtrC16(reinterpret_cast<const TUint16*> (numbers[28])));
		number%=100;
		if (number)
			{
			TBuf<bufferSize> buf;
			TDes& des1= buf;
			FindNumberInWords(number, des1);
			aDes.Append(KAnd);
			aDes+=des1;
			}
		}
  
	// Converts the words if the number is greater than or equal to 1000
	if(number>=1000)
		{
		TInt hundreds = number/1000;
		aDes.Copy(reinterpret_cast<const TUint16*> (numbers[hundreds]));
		aDes.Append(' ');
		aDes.Append(TPtrC16(reinterpret_cast<const TUint16*> (numbers[29])));
		number%=1000;
		if (number)
			{
			TBuf<bufferSize> buf;
			TDes& des1= buf;
			FindNumberInWords(number, des1);
			aDes.Append(KAnd);
			aDes+=des1;
			}
		}
	}
 
/**
Applies various operations to hash set of pointers:
- Inserting items
- Finding items
- Iterating items
- Removing items.
- Closing and cleanup items.   
@leave KErrNoMemory An attempt to allocate memory has failed
@leave KErrNotFound Item not found
*/
void CHashTableExample::OperationsToPtrHashSetL()	
	{
	// 	
	iConsole->Printf(KOperation);
    
    // First, set up some sample data to store in the hash
    // We're going to use strings for the numbers 0 to 1999 
     	
	// Populate a string array temporarily with the sample data
	// We'll use the array to populate the set, and later on use it also
	// to populate a map
    const TInt KMaxItem=1200;
    const TInt KMaxBufferSize=256;
    TInt i=0;
	for (i=0; i<KMaxItem; ++i)
		{
	   	HBufC* hbuf = HBufC::NewLC(KMaxBufferSize);
 		TPtr buf = hbuf->Des();
		// FindNumberInWords gets a string representation of the specified integer
		FindNumberInWords(i, buf);		
		iPointerArray.AppendL(hbuf);
		CleanupStack::Pop(hbuf);
		}
	
	// Now create a set and populate it with the data from the array

	// Creates an object of hash set of pointers for 16 bit data using the template class RPtrHashSet
	RPtrHashSet<TDesC16> ptrHashSet;	
	// Push hash set of pointers on to the cleanup stack
	CleanupClosePushL(ptrHashSet);
	// Insert items to hash set of pointers from 16 bit descriptor array
	for (i=0; i<KMaxItem; ++i)
		{
		ptrHashSet.InsertL(iPointerArray[i]);
		}  
	iConsole->Printf(KInsertItemsToPtrHashSet);
  
	// Search the set for a specified item 
	TDesC16* item1 = ptrHashSet.Find(KFindItem);
	// item1 is NULL if KFindItem is not found in the set
	if (item1)
		{
		iConsole->Printf(KItemPresentInPtrHashSet);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInPtrHashSet);
		}	
 
 	// Iterate over the items in the set
	// Creates an iterator object of type TIter
	RPtrHashSet<TDesC16>::TIter ptrHashSetIter(ptrHashSet);
	// Loop through the items 
	for ( ; ; )
		{
		const TDesC16* resNext = ptrHashSetIter.Next();
		// Next() moves the iterator to the next item and returns it
		// Returns NULL if there are no more items  
		if (!resNext)
			{ 
			break;
			}
		}
	iConsole->Printf(KIterateItemsFromPtrHashSet);
  	
	// Remove an item from hash set
	TInt err = ptrHashSet.Remove(&KRemoveItem);
	// Check if the item was successfully removed
	if (err == KErrNone)
		{
		iConsole->Printf(KItemPresentInPtrHashSet);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInPtrHashSet);
		}	
	iConsole->Printf(KRemoveItemsFromPtrHashSet);
  
	// Close and cleanup hash set of pointers
	CleanupStack::PopAndDestroy(&ptrHashSet);
 
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	}
 
/**
Constructs a hash map using the default hash and identity functions for integers
and for descriptors.
*/
void CHashTableExample::ConstructDefaultHashMap()
	{
	// Construct hash map using default hash and identity functions for integer
	RHashMap<TInt, TInt> hashMapInt();
  
	// Construct hash map using default hash and identity functions for 8 bit descriptor
	RHashMap<TDesC8, TDesC8> hashMapDes8();
  
	// Construct hash map using default hash and identity functions for 16bit descriptor
	RHashMap<TDesC16, TDesC16> hashMapDes16();
  
	iConsole->Printf(KHashMap);
	iConsole->Printf(KConstruct);
	iConsole->Printf(KConstructDeafultHashMap);	
	}

/** 
Constructs a hash map using the custom hash and identity functions to demonstrate the options.
The objects declared aren't used elsewhere in the example.
*/
void CHashTableExample::ConstructOwnHashMap()
	{
	// Creates an object of custom hash function by using our created structure
	THashFunction32<TMyOwnObject> ownHashFunction(MyHashFunction);
 
	// Creates an object of identity function by using our created structure
	TIdentityRelation<TMyOwnObject> ownIdentityFunction(MyIdentityFunction);
 
	// Construct hash map by providing custom hash and identity function	
	RHashMap<TMyOwnObject, TMyOwnObject> ownHashMap(ownHashFunction, ownIdentityFunction);
	iConsole->Printf(KConstructOwnHashMap);		
	}
   
/**
Applies various operations to hash map:
- Inserting items
- Finding items
- Iterating items
- Removing items
- Closing and cleanup items.
@leave KErrNoMemory An attempt to allocate memory has failed
@leave KErrNotFound Item not found
*/ 
void CHashTableExample::OperationsToHashMapL()
	{
	TInt maxItem=300;
	TInt itemToBeFound=150;
	TInt itemToBeRemoved=200;
	TInt64 items;
	
	iConsole->Printf(KOperation);
  
	// Creates an object of hash map using the template class RHashMap
	RHashMap<TInt, TInt> hashMap;
	
	// Push hash map on to the cleanup stack
	CleanupClosePushL(hashMap);
    
	// Insert items to hash map
	for (TInt i=0; i<maxItem; i++)
		{
		TInt res = Math::Rand(items);
		hashMap.InsertL(res*res, res);
		}
  	
	iConsole->Printf(KInsertItemsToHashMap);
  
	// Search the map for a specified item
	TInt* result= hashMap.Find(itemToBeFound);
	
	// result is NULL if specified item is not found in the map
	if(result)
		{
		iConsole->Printf(KItemPresentInHashMap);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInHashMap);
		}
  
	// Iterate over the items in the map
	// Creates an iterator object of type TIter
	RHashMap<TInt, TInt>::TIter hashMapIter(hashMap);
	
	for ( ; ; )
		{
		const TInt* resNext = hashMapIter.NextKey();
		if (!resNext)
			{ 
			break;
			}
		}

	iConsole->Printf(KIterateItemsFromHashMap);
 
	// Remove an item from hash map
	TInt res = hashMap.Remove(itemToBeRemoved);
	
	// Check if the item was successfully removed
	if(res)
		{
		iConsole->Printf(KItemPresentInHashMap);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInHashMap);
		}
 
	iConsole->Printf(KRemoveItemsFromHashMap);
  
	// Close and cleanup hash map
	CleanupStack::PopAndDestroy(&hashMap);
  
	iConsole->Printf(KPressAKey);
	iConsole->Getch();
	}

/**
Constructs hash map of pointers using the default hash and identity functions for integers
and for descriptors.
*/
void CHashTableExample::ConstructDefaultPtrHashMap()
	{
	// Construct hash map of pointers using default hash and identity functions for integer
	RPtrHashMap<TInt, TInt> ptrHashMapInt();
  
	// Construct hash map of pointers using default hash and identity functions for 8bit descriptor
	RPtrHashMap<TDesC8, TDesC8> ptrHashMapDes8();
  
	// Construct hash map of pointers using default hash and identity functions for 16bit descriptor
	RPtrHashMap<TDesC16, TDesC16> ptrHashMapDes16();
     
	iConsole->Printf(KPtrHashMap);
	iConsole->Printf(KConstruct);
	iConsole->Printf(KConstructDeafultPtrHashMap);	
	}
 
/**
Constructs hash map of pointers using the custom hash and identity functions to demonstrate the options.
The objects declared aren't used elsewhere in the example.
*/
void CHashTableExample::ConstructOwnPtrHashMap()
	{
	// Creates an object of custom hash function by using our created structure
	THashFunction32<TMyOwnObject> ownHashFunction(MyHashFunction);
  
	// Creates an object of identity function by using our created structure
	TIdentityRelation<TMyOwnObject> ownIdentityFunction(MyIdentityFunction);
  
	// Construct hash map of pointers by providing custom hash and identity function
	RPtrHashMap<TMyOwnObject, TMyOwnObject> ownPtrHashMap(ownHashFunction, ownIdentityFunction);
	iConsole->Printf(KConstructOwnPtrHashMap);
	}
 
/**
Applies various operations to hash map of pointers:
- Inserting items
- Finding items
- Iterating items
- Removing items
- Closing and cleanup items.
@leave KErrNoMemory An attempt to allocate memory has failed
@leave KErrNotFound Item not found
*/
void CHashTableExample::OperationsToPtrHashMapL()
	{
	TInt i;
	TInt maxItem=200;
 
	iConsole->Printf(KOperation);

	// Creates an object of hash map of pointers using the template class RPtrHashMap
	RPtrHashMap<TDesC16, TDesC16> ptrHashMap;
	
	// Push hash map of pointers on to the cleanup stack
	CleanupClosePushL(ptrHashMap);
   
	// Insert items to hash map of pointers 
	for (i=0; i<maxItem; ++i)
		{
	    ptrHashMap.InsertL(iPointerArray[i], iPointerArray[i+1]);
		}
  
	iConsole->Printf(KInsertItemsToPtrHashMap);
  
	// Search the set for a specified item 
	TDesC16* item1= ptrHashMap.Find(KFindItem);
	
	// item1 is NULL if KFindItem is not found in the map
	if(item1)
		{
		iConsole->Printf(KItemPresentInPtrHashMap);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInPtrHashMap);
		}

	// Iterate over the items in the map
	// Creates an iterator object of type TIter
	RPtrHashMap<TDesC16, TDesC16>::TIter ptrHashMapIter(ptrHashMap);
 
	for ( ; ; )
		{
		const TDesC16* resNext = ptrHashMapIter.NextKey();
		// Next() moves the iterator to the next item and returns it
		// Returns NULL if there are no more items 
		if (!resNext)
			{ 
			break;
			}
		}
	
	iConsole->Printf(KIterateItemsFromPtrHashMap);

	// Remove an item from hash map of pointers 
    TInt res = ptrHashMap.Remove(&KRemoveItem);
    
    // Check if the item was successfully removed
	if(res)
		{
		iConsole->Printf(KItemPresentInPtrHashMap);
		}
	else
		{
		iConsole->Printf(KItemNotPresentInPtrHashMap);
		}	

	iConsole->Printf(KRemoveItemsFromPtrHashMap);
	iConsole->Printf(KExitMsg); 
	iConsole->Getch();	
  
	// Close and Cleanup hash map of pointers
	CleanupStack::PopAndDestroy(&ptrHashMap);
	}
 
void MainL()
	{
	CHashTableExample* app= CHashTableExample::NewL();
	CleanupStack::PushL(app);
 
	// Hash set 
	app->ConstructDefaultHashSet();
	app->ConstructOwnHashSet();
	app->OperationsToHashSetL();
  
	// Hash set of pointers
	app->ConstructDefaultPtrHashSet();
	app->ConstructOwnPtrHashSet();
	app->OperationsToPtrHashSetL();
  
	// Hash map
	app->ConstructDefaultHashMap();
	app->ConstructOwnHashMap();
	app->OperationsToHashMapL();
 
	// Hash map of pointers
	app->ConstructDefaultPtrHashMap();
	app->ConstructOwnPtrHashMap();
	app->OperationsToPtrHashMapL();
 
	CleanupStack::PopAndDestroy(app);
	} 

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;

	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	TRAPD(err, MainL());
	if(err !=KErrNone)
		{
		User::Panic(KFailed, err);
		}	
	delete cleanup;

	__UHEAP_MARKEND;
	return KErrNone;
	}
  
  
  
  
 
 
 
