/*
 * Copyright  2009 Nokia Corporation.
 */

// INCLUDE FILES
#include <aknquerydialog.h> // for input dialogs
#include <utf.h>
#include "Common.h"
#include "BluetoothPMPExampleEngine.h"
#include "DeviceDiscoverer.h"
#include "ServiceAdvertiser.h"
#include "ServiceDiscoverer.h"
#include "Listener.h"
#include "Connector.h"
#include <BtPmpEx.rsg>

CBluetoothPMPExampleEngine* CBluetoothPMPExampleEngine::NewL(
    CBluetoothPMPExampleAppUi& aAppUi)
    {
    CBluetoothPMPExampleEngine* self =
        CBluetoothPMPExampleEngine::NewLC(aAppUi);
    CleanupStack::Pop(self);
    return self;
    }


CBluetoothPMPExampleEngine* CBluetoothPMPExampleEngine::NewLC(
    CBluetoothPMPExampleAppUi& aAppUi)
    {
    CBluetoothPMPExampleEngine* self =
        new (ELeave) CBluetoothPMPExampleEngine(aAppUi);
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
    }


// Standard EPOC 2nd phase constructor
void CBluetoothPMPExampleEngine::ConstructL()
    {
    // get socket server session
    User::LeaveIfError(iSocketServ.Connect());

    // init listener
    iListener = CListener::NewL(*this, iSocketServ);
    // init device discoverer
    iDeviceDiscoverer = CDeviceDiscoverer::NewL(iSocketServ, *this);

#ifdef ENABLE_LIAC
    // Set default LIAC to OFF
    iLIAC = ETrue;
    // set current LIAC state
    iDeviceDiscoverer->SetLIAC( iLIAC );
#endif
    // init service advertiser
    iServiceAdvertiser = CServiceAdvertiser::NewL();
    // init service discoverer
    iServiceDiscoverer = CServiceDiscoverer::NewL(*this);

    // clean connections table to begin with
    for ( TInt idx=0; idx<KMaxConnectedDevices; idx++ )
        {
        iConnectedDevices[idx] = NULL;
        }
    
    #ifdef ENABLE_LIAC
    // The key to control whether the local device is in Limited Discoverable 
    // mode The P&S value will contain a boolean: ETrue if in limited discoverable mode,
    // otherwise EFalse
    TInt attErr = iProperty.Define(KPropertyUidBluetoothControlCategory, 
            KPropertyKeyBluetoothSetLimitedDiscoverableStatus,RProperty::EInt);
    if (attErr != KErrNone && attErr != KErrAlreadyExists)
        {
        User::Leave(attErr);
        }
    
    
    // Set LIAC value
    SetLIAC(iLIAC);
    #endif
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::CBluetoothPMPExampleEngine(
//     CBluetoothPMPExampleAppUi* aAppUi)
//
// constructor
// ----------------------------------------------------------------------------
CBluetoothPMPExampleEngine::CBluetoothPMPExampleEngine(
    CBluetoothPMPExampleAppUi& aAppUi):
    iIsSlave(EFalse),
    iAppUi(aAppUi)
    {
    //Nothing to do
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::~CBluetoothPMPExampleEngine()
//
// destructor
// ----------------------------------------------------------------------------
CBluetoothPMPExampleEngine::~CBluetoothPMPExampleEngine()
    {
    #ifdef ENABLE_LIAC
    SetLIAC(EFalse);
    iProperty.Close();
    #endif

    // disconnect all devices and clean up connections table
    DisconnectDevices();
    // stop and kill helpers

    delete iServiceAdvertiser;
    iServiceAdvertiser=NULL;

    delete iListener;
    iListener=NULL;
    
    iDeviceDiscoverer->Cancel();
    delete iDeviceDiscoverer;
    iDeviceDiscoverer = NULL;

    delete iServiceDiscoverer;
    iServiceDiscoverer=NULL;

    iSocketServ.Close();

    // wipe device data list
    iDevDataList.ResetAndDestroy();
    }



// ----------------------------------------------------------------------------
// ShowMessageL(
//     const TDesC& aMsg, TBool aDrawLine=EFalse)
//
// displays text referenced by aMsg in the label, will append the aMsg in the
// existing text
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::ShowMessageL(const TDesC& aMsg,
                                              TBool aDrawLine=EFalse)
    {
    if (iAppUi.Container())
        {
        if( aDrawLine )
            iAppUi.Container()->DrawLineL();
        
        iAppUi.Container()->ShowMessageL(aMsg);
        }
    }



// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::DiscoverDevicesL()
//
// discover bluetooth devices within range.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::DiscoverDevicesL()
    {
    //timing:
    iStartTime.HomeTime();
    ShowMessageL(KDiscDevicesTxt, ETrue);
    TRAPD(err,iDeviceDiscoverer->DiscoverDevicesL(&iDevDataList));
    if (err)
        {
        HandleDeviceDiscoveryComplete(err);
        }
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::DiscoverServicesL()
//
// discover services provided by the discovered devices.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::DiscoverServicesL()
    {
    iStartTime.HomeTime();
    ShowMessageL(KDiscServicesTxt, ETrue);
    iServiceDiscoverer->DiscoverServicesL(&iDevDataList);
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::StartSlaveL()
//
// set application in slave mode.  the device will be set to listen to
// a bluetooth channel, and advertise its service on the channel.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::StartSlaveL()
    {
    if ( iIsSlave )
        return;

    ShowMessageL(KSlaveInitTxt, ETrue);

    TInt channel;
    iListener->StartListenerL(channel);
    TBuf<KThirty> msg;
    msg.AppendFormat(KListeningTxt, channel);
    ShowMessageL(msg, EFalse);

    iServiceAdvertiser->StartAdvertiserL(channel);
    ShowMessageL(KSlaveInitCompTxt, EFalse);
    iIsSlave=ETrue;
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::StopSlaveL()
//
// Stop slave mode
// 
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::StopSlaveL()
    {
    if ( !iIsSlave )
        return;

    ShowMessageL(KNullDesC, ETrue);

    iListener->StopListener();
    iServiceAdvertiser->StopAdvertiserL();
    iIsSlave=EFalse;
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::DisconnectDevices()
//
// disconnect connected devices and clean up connections table
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::DisconnectDevices()
    {
    for ( TInt idx=0; idx<KMaxConnectedDevices; idx++ )
        {
        if ( iConnectedDevices[idx] )
            {
            delete iConnectedDevices[idx];
            iConnectedDevices[idx]=NULL;
            }
        }
    ShowMessageL(KDisconnDevicesTxt, ETrue);
    iIsMaster = EFalse;
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::ConnectDevicesL()
//
// attempt to connect on all discovered devices (up to 7)
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::ConnectDevicesL()
     {
     // close and delete all existing connections
     DisconnectDevices();

     iIsMaster = ETrue;
     ShowMessageL(KConnectingTxt, ETrue);
     // now attempt to connect
     for ( TInt idx=0; idx<(iDevDataList.Count()); idx++ )
         {
         if ( iConnectedDeviceCount>=KMaxConnectedDevices )
             return;

         TDeviceData *dev = iDevDataList[idx];

         // if matching service on remote device was found, the service port
         // is set and will be > 0.  if so, we can attempt to connect.
         if ( dev->iDeviceServicePort>0 )
             {
             CConnector* connector = CConnector::NewLC(*this, iSocketServ);
             if ( (connector->ConnectL(dev->iDeviceName,
                                       dev->iDeviceAddr,
                                       dev->iDeviceServicePort))==KErrNone )
                 {
                 iConnectedDevices[iConnectedDeviceCount] = connector;
                 iConnectedDeviceCount++;
                 CleanupStack::Pop(connector);
                 }
             else
                 {
                 // connection to device failed!!
                 CleanupStack::PopAndDestroy();
                 }
             }
         }

     ShowConnectedDevicesL();
     }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::ShowConnectedDevicesL()
//
// display the devices we are connected to
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::ShowConnectedDevicesL()
    {
    TInt count=0;
    ShowMessageL(KConnDevicesTxt, ETrue);
    for (TInt idx=0; idx<KMaxConnectedDevices; idx++)
        {
        if ( iConnectedDevices[idx] )
            {
            THostName name;
            name = iConnectedDevices[idx]->iName;
            name.Append(KNewLine);
            ShowMessageL(name, EFalse);
            count++;
            }
        }
    if ( count==0 )
        {
        // no connections!
        // this may be because of no devices has been discovered,
        // or no devices are offering the requested service.
        ShowMessageL(KNoConns, EFalse);
        }
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::SendMessageL()
//
// send a message to all connected devices.  user will be prompted to enter
// the message text.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::SendMessageL()
    {
    iMsgtext.Zero();
    CAknTextQueryDialog* dlg = CAknTextQueryDialog::NewL(iMsgtext);
    if(!dlg->ExecuteLD(R_BLUETOOTHEXAMPLE_MESSAGEINPUT))
        {
        // Cancel
        return;
        }
    
    // explicitly convert from 16bit to 8bit character set
    CnvUtfConverter::ConvertFromUnicodeToUtf8(iMsgtext8, iMsgtext);

    TBuf<KEighty> msg;
    if ( iIsSlave )
        {
        // slave sending data
        iListener->SendData(iMsgtext8);
        msg.Format(KFormatStr3, &iMsgtext);
        ShowMessageL(msg, EFalse);
        }
    else
        {
        // master sending data
        for (TInt idx=0; idx<KMaxConnectedDevices; idx++)
            {
            if ( iConnectedDevices[idx])
                {
                iConnectedDevices[idx]->SendData(iMsgtext8);
                THostName name;
                name=iConnectedDevices[idx]->iName;
                msg.Format(KFormatStr2, &name, &iMsgtext);
                ShowMessageL(msg, EFalse);
                }
            }
        }
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleListenerDataReceivedL(TDesC& aData)
//
// display data the slave listener receives from the master.  this is a
// callback that CListener class will invoke when it receives new data.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleListenerDataReceivedL(const TDesC& aData)
    {
    // display received message
    TBuf<KEighty> msg;
    msg.Format(KFormatStr1, &aData);
    ShowMessageL(msg, EFalse);
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleListenerConnectedL()
//
// a callback received from CListener to indicate that it has been connected to
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleListenerConnectedL()
    {
    ShowMessageL(KConnMsg, ETrue);
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleListenerDisconnectedL()
//
// a callback received from CListener to indicate that it has been disconnected
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleListenerDisconnectedL()
    {
    if ( !iIsSlave )
        return;

    // Stop listener and advertiser
    // Set app to init mode
    StopSlaveL();
    ShowMessageL(KDisconMsg, ETrue);
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleConnectorDataReceivedL(THostName aName,
//                                                          TDesC& aData)
//
// display data the master receives from a connected slave.  this is a
// callback that CConnector class will invoke when it receives data from slave.
// Also echo the message to other slaves.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleConnectorDataReceivedL(THostName aName,
                                                              const TDesC& aData)
    {
    // display received message
    TBuf<KEighty> msg;
    msg.Format(KFormatStr, &aName, &aData);
    ShowMessageL(msg, EFalse);

    // echo the message to other slaves
    _LIT8(KSeparator, ":");

    TBuf8<KEighty> buf; //should use HBufC so the size will be big enough
    TPtr8 msgtext8((TUint8*)buf.Ptr(), aData.Size()+KSeparator().Length() + aName.Size());
    CnvUtfConverter::ConvertFromUnicodeToUtf8(msgtext8, aData);

    //convert name to UTF8 so other slaves see
    //the sender name
    TBuf8<KEighty> bufName;
    TPtr8 name8((TUint8*)bufName.Ptr(), aName.Size());
    CnvUtfConverter::ConvertFromUnicodeToUtf8(name8, aName);

    //add the separator and name in the beginning;
    msgtext8.Insert(0, KSeparator );
    msgtext8.Insert(0, name8);

    for (TInt idx=0; idx<KMaxConnectedDevices; idx++)
        {
        if ( iConnectedDevices[idx])
            {
            THostName name;
            name=iConnectedDevices[idx]->iName;

            //echo to other slaves than the sender
            if( name.Compare(aName) != 0)
                iConnectedDevices[idx]->SendData(msgtext8);
            }
        }
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleConnectorErrorL(THostName aName,TInt aError)
//
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleConnectorErrorL(THostName aName, TInt aError)
    {
    if (aError)
        {
        // display received message
        TBuf<KEighty> msg;
        msg.Format(KDeviceDisconMsg, &aName);
        ShowMessageL(msg, EFalse);

        for (TInt idx=0; idx<KMaxConnectedDevices; idx++)
            {
            if ( iConnectedDevices[idx])
                {
                THostName name;
                name=iConnectedDevices[idx]->iName;

                //echo to other slaves than the sender
                if( name.Compare(aName) == 0)
                    {
                    delete iConnectedDevices[idx];
                    iConnectedDevices[idx] = NULL;
                    }
                }
            }

        // Is there more connection running?
        iIsMaster = EFalse;
        for (TInt idx=0; idx<KMaxConnectedDevices; idx++)
            {
            if ( iConnectedDevices[idx])
                {
                iIsMaster = ETrue;
                break;
                }
            }
        }
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleDeviceDiscoveryComplete()
//
// a callback received from device discoverer to indicate that the device
// discovery has completed.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleDeviceDiscoveryComplete(TInt aError)
    {
    if (aError)
        {
        iIsMaster = EFalse;
        }
    else
        {
        iIsMaster = ETrue;
        }
    
    iEndTime.HomeTime();

    TTimeIntervalSeconds seconds;
    iEndTime.SecondsFrom(iStartTime, seconds);

    TInt time = seconds.Int();
    TBuf<KTwelve> temp = KTimeTxt();
    temp.AppendNum(time);
    temp.Append(KSecTxt);
    TRAPD(err,ShowMessageL(temp,EFalse));

    // iterate and display found devices, if any
    if ( iDevDataList.Count()> 0 )
        {
        TBuf<KTwenty> count = KFoundTxt();
        count.AppendNum( iDevDataList.Count() );
        count.Append( KDevices );
        TRAP(err,ShowMessageL(count));
        }
    else
        {
        // no devices found
        TRAP(err,ShowMessageL(KNoDevFound, EFalse));
        }
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HandleServiceDiscoveryCompleteL()
//
// a callback received from service discoverer to indicate that the service
// discovery has completed.
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::HandleServiceDiscoveryCompleteL()
    {
    iEndTime.HomeTime();

    TTimeIntervalSeconds seconds;
    iEndTime.SecondsFrom(iStartTime, seconds);

    TInt time = seconds.Int();
    TBuf<KTwelve> temp = KTimeTxt();
    temp.AppendNum(time);
    temp.Append(KSecTxt);
    ShowMessageL(temp,ETrue);

    TInt count=0;
    // display devices with service we can use
    for ( TInt idx=0; idx<(iDevDataList.Count()); idx++ )
        {
        TDeviceData *dev = iDevDataList[idx];
        if ( dev->iDeviceServicePort>0 )
            {
            THostName name = dev->iDeviceName;
            name.Append(KNewLine);
            ShowMessageL(name, EFalse);
            count++;
            }
        }
    if ( count==0 )
        {
        ShowMessageL(KNoServiceFound, EFalse);
        }
    else
        {
        ShowMessageL(KServiceFound, EFalse);
        }
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::HasConnections()
//
// returns true if master has any connections to slave(s)
// ----------------------------------------------------------------------------
TBool CBluetoothPMPExampleEngine::HasConnections()
    {
    TBool exists = EFalse;
    for (TInt idx=0; idx<KMaxConnectedDevices; idx++)
        {
        if ( iConnectedDevices[idx])
            {
            exists = ETrue;
            break;
            }
        }
    return exists;
    }


// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::DeviceDiscovered
// ----------------------------------------------------------------------------
//a callback to indicate that a device has been found
//main reason for this is that the UI can react so user
//knows that something is going on and the app is not "frozen"
void CBluetoothPMPExampleEngine::DeviceDiscovered(const TDeviceData &aDevice)
    {
    TBuf<KForty> name = aDevice.iDeviceName;
    name.Trim();

    if( name.Length() == 0 )
        name.Append(KDeviceWithNoName);

    TRAPD(err,ShowMessageL(name, EFalse));
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::ReportServiceDiscoveryErrorL
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::ReportServiceDiscoveryErrorL(TInt aError)
    {
    TBuf<KThirty> discError = KServiceDiscoveryError();
    discError.AppendNum(aError);
    iAppUi.Container()->ShowMessageL(discError);
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::TurnBtOnL
// ----------------------------------------------------------------------------
//Uses the Notifier API to ask the user to turn on Bluetooth
//if it's not on already.
void CBluetoothPMPExampleEngine::TurnBtOnL()
    {
    //the constant is from btnotifierapi.h which is not in all SDKs
    //so it's hard coded here
    const TUid KPowerModeSettingNotifierUid = {0x100059E2};
    //const TUid KBTPowerStateNotifierUid = {0x101F808E}; //S80 and 7710
    
    RNotifier notifier;
    User::LeaveIfError( notifier.Connect() );
    TPckgBuf<TBool> dummy(ETrue);
    TPckgBuf<TBool> reply(EFalse);
    TRequestStatus stat;
    notifier.StartNotifierAndGetResponse(stat, KPowerModeSettingNotifierUid, dummy, reply);
    User::WaitForRequest(stat);
    notifier.CancelNotifier(KPowerModeSettingNotifierUid);
    notifier.Close();
    }

#ifdef ENABLE_LIAC
// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::SetLIAC
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::SetLIAC( TBool aState )
    {
    TInt err = KErrNone;

    // Set LIAC for the system    
    err = iProperty.Set(KPropertyUidBluetoothControlCategory,
            KPropertyKeyBluetoothSetLimitedDiscoverableStatus, 
            aState);
    
    TBuf<KEighty> myMessage;

    myMessage.Zero();

    if ( KErrNone == err )
        {
        if ( aState )
            {
            myMessage.Append( KLIACOn );
            }
        else
            {
            myMessage.Append( KLIACOff );
            }        
        iLIAC = aState;

        iDeviceDiscoverer->SetLIAC( iLIAC );

        }
    else
        {
        myMessage.AppendFormat( KLIACError, err );
        }

    TRAP( err, ShowMessageL( myMessage, ETrue ));
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::LIAC
// ----------------------------------------------------------------------------
TBool CBluetoothPMPExampleEngine::LIAC()
    {
    return iLIAC;
    }
#endif

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::IsDiscoveryActive
// ----------------------------------------------------------------------------
TBool CBluetoothPMPExampleEngine::IsDiscoveryActive()
    {
    return iDeviceDiscoverer->IsActive();
    }

// ----------------------------------------------------------------------------
// CBluetoothPMPExampleEngine::StopDiscovery
// ----------------------------------------------------------------------------
void CBluetoothPMPExampleEngine::StopDiscovery()
    {
    if ( IsDiscoveryActive() )
        {
        iDeviceDiscoverer->StopDiscovery();
        } 
    iIsMaster = EFalse;
    }
