// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//



#include "rconnection.h"

_LIT(KContinueNote,"Press any key to continue\n");
_LIT(KOpenErr,"Open() unsuccessful\n");
_LIT(KStartErr,"RConnection::Start() unsuccessful\n");
_LIT(KInterfaceUp,"Connection has started and interface is up\n");
_LIT(KInterfaceDown,"Connection has closed and interface is down\n");
_LIT(KConnectionUpNote,"Connection is up....\n");
_LIT(KConnectionStartNote,"Start() was called on the connection\n");
_LIT(KConnectionClosingNote,"The connection is about to be closed\n");
_LIT(KWaitNote,"Waiting for the connection to close.....\n");
_LIT(KUpLinkData,"Data volume on uplink=%d bytes\n");
_LIT(KDownLinkData,"Data volume on downlink=%d bytes\n");
_LIT(KDataTransferNote,"Calling DataSendAndReceive()\n");
_LIT(KSocketErrNote,"Socket preparation failed\n");
_LIT(KSocketSetNote,"Socket successfully established\n");

_LIT(KRow1,"**********************************************************\n");
_LIT(KRow2,"	Section Demonstrating  :	          \n");
_LIT(KRow3,"**********************************************************\n");
_LIT(KTab,"\t");
_LIT(KNewLine,"\n");

#define KEchoPort 0xAB			//Port value for client socket.
#define KMaxBufferLength 2048   //Maximum length of buffer used to send and receive data.
#define KDelay 1000000			//Time interval in microseconds to wait before carrying out the next task.

/**
 Constructor.
 */		
CRConnection::CRConnection(CConsoleBase* aConsole)
	:iConsole(aConsole)	
	{
	//Set the destination address and port.
	//Before running the example user should start an Echo server 
	//and then specify the respective IP Address and port of the echo server 
	//for iDestAddr.
	//Until the Echo Server is established send and receive functionality won't work properly. 
	
	
	_LIT(KDestAddr,"127.0.0.1");  
	const TInt KSockPort= 7;
	
	iDestAddr.Input(KDestAddr);
	iDestAddr.SetPort(KSockPort);
	  
	StartESOCK();
	}


/**
 Destructor.
 */	
CRConnection::~CRConnection()
	{
	//Closing the monitor and socket server session.
	iMonitor.Close();
	iSocketServ.Close();
	}

/**
 Connects to the socket server and opens a monitor for receiving interface
 notification. The client is notified via monitor, whenever another connection 
 that uses a socket server (iSocketServ) goes up or down.
 */
void CRConnection::StartESOCK()	
	{
	
	//Connects to the socket server.
	TInt err=iSocketServ.Connect();
	if(err!=KErrNone)
		{
		_LIT(KConnectErr,"Connect failed\n");
		User::Panic(KConnectErr,err);		
		}	
	
	//Opens the monitor to receive notification when another connection
	//that uses the same socket server goes up or down. 
	err=iMonitor.Open(iSocketServ);
	if(err!=KErrNone)
		{
		iConsole->Printf(KOpenErr);
		return;	
		}	
	}
		
/**
 Starts an explicit connection using the default CommDb connection preferences.
 CommDb is the Symbian platform Communications Database.
 
 'Explicit' means the connection is established using an RConnection instance, rather than implicitly, where 
 the application opens and uses an RSocket or RHostResolver rather than RConnection.
 */
TInt CRConnection::ConnectWithoutDbOverrideL()
	{
	// Opens an RConnection and associates it with the same socket server session as iMonitor.
	// To use an RConnection instance first it needs to be opened on an existing socket server session 
	// and then Start() must be called to associate it with an underlying interface.
	TInt err=iConnection.Open(iSocketServ);	
	if(err!=KErrNone)
		{
		iConsole->Printf(KOpenErr);
		return err;	
		}
	
	//Starts the connection using default CommDb settings.
	err=iConnection.Start();
	if(err!=KErrNone)
		{
		iConsole->Printf(KStartErr);
		return err;	
		}
	else
		{
		iConsole->Printf(KConnectionStartNote);	
		return err;
		}				
	}
	
/**
 Starts an explicit connection, overriding the default Commdb connection settings.
 */	
TInt CRConnection::ConnectWithDbOverrideL()
	{
	// To use an RConnection instance first it needs to be opened on an existing socket server session 
	// and then Start() must be called to associate it with an underlying interface.
	TInt err=iConnection.Open(iSocketServ);	
	if(err!=KErrNone)
		{
		iConsole->Printf(KOpenErr);
		return err;
		}
	
	// Overrides some Commdb connection preferences.
	const TInt KIapId=9;
	TCommDbConnPref prefs;
	prefs.SetDialogPreference(ECommDbDialogPrefDoNotPrompt);
	prefs.SetIapId(KIapId);    //The commdb is set for Ethernet with Daemon Static IP. 
							   		
	//Starts an outgoing connection.
	err=iConnection.Start(prefs);
	
	if(err==KErrNone)
		{
		iConsole->Printf(KConnectionStartNote);	
		return err;
		}
	else
		{
		iConsole->Printf(KStartErr);
		return err;	
		}		
	}
/**
 Gets progress notification about the connection.
 RConnection::ProgressNotification() is used to monitor the state of the connection as 
 it undergoes various stages before it is fully established.
 */	
void CRConnection::GetProgressNotification()
	{
	TRequestStatus status;
	TNifProgressBuf buffer; //filled with progress/error information on completion 
	while((TUint)(buffer().iStage)!=KConnectionUp) 
		{
		iConnection.ProgressNotification(buffer,status);
		User::WaitForRequest(status);
		if(status==KErrNone&&(buffer().iError)==KErrNone)
			{
			DisplayProgressinfo(buffer());
			}
		User::After(KDelay);
		}
	iConsole->Printf(KConnectionUpNote); //Link layer is now open.	
		
			
	}
	
/**
 Gets progress information at each stage of the connection.
 @param aProgress Filled with progress/error information on completion.
 */	
void CRConnection::DisplayProgressinfo(const TNifProgress& aProgress)
	{
	switch(aProgress.iStage)
		{
		case KStartingSelection:
			{
			_LIT(KStartSelectionNote,"Starting Selection...........\n");
			iConsole->Printf(KStartSelectionNote);
			break;	
			}
					
		case KFinishedSelection:
			{
			_LIT(KFinishedSelectionNote,"Finished Selection...........\n");
			iConsole->Printf(KFinishedSelectionNote);
			break;	
			}
					
		case KConnectionOpen:
			{
			_LIT(KWlanAgtConnected,"Wlan agent connected....\n");
			iConsole->Printf(KWlanAgtConnected);
			break;	
			}
						
		//Generic progress notifications from the configuration daemon.				
		case KConfigDaemonLoading:
			{
			_LIT(KConfigDaemonLoadingNote,"Daemon loading....\n");
			iConsole->Printf(KConfigDaemonLoadingNote);
			break;	
			}
					
		case KConfigDaemonLoaded:
			{
			_LIT(KConfigDaemonLoadedNote,"Daemon loaded....\n");
			iConsole->Printf(KConfigDaemonLoadedNote);
			break;	
			}
						
		case KConfigDaemonStartingRegistration:
			{
			_LIT(KConfigDaemonStartingRegistrationNote,"Daemon starting registration....\n");
			iConsole->Printf(KConfigDaemonStartingRegistrationNote);
			break;	
			}
						
		case KConfigDaemonFinishedRegistration:
			{
			_LIT(KConfigDaemonFinishedRegistrationNote,"Daemon registration finished....\n");
			iConsole->Printf(KConfigDaemonFinishedRegistrationNote);
			break;	
			}	
	
		default:
			break;			
		}
	}

/**
 Prepares the socket for carrying out data transfer by:
 1) opening the socket and associating it with the connection.
 2) setting options on the socket and the local port for it.
 */		
TInt CRConnection::PrepareSocket()
	{
	TInt err = iSocket.Open(iSocketServ, KAfInet, KSockDatagram, KProtocolInetUdp, iConnection);
	if (err != KErrNone)
		{
		return err;	
		}
		
	//Sets a socket option. These options affect operations such as the routing of packets, 
	//out-of-band data transfer, and so on. Here the option level "KSolInetIp" is set for an IP socket,
	//and the option being set is "KSoReuseAddr". This means the socket is allowed to be bound to a local
	//address that is already in use.
	//The third parameter i.e 1 states that the option is enabled.A value of 0 means disabled.
	
	err = iSocket.SetOpt(KSoReuseAddr, KSolInetIp, 1); 
	if (err != KErrNone)
		{
		return err;	
		}
	
	//Sets a local port for the socket.
	err = iSocket.SetLocalPort(KEchoPort); 
	if (err != KErrNone)
		{
		return err;	
		}
	return err;		
	}
/**
  Sends UDP data (an ICMP packet) over the socket specified to the address specified.
  @param aPayloadSize The size of the packet to send.
  @return Error code
 */	 	
TInt CRConnection::SendUdpData(TUint aPayloadSize)
	{
	
	TBuf8<KMaxBufferLength> buffer;
	// construct an ICMP packet to send on the socket.
	buffer.SetMax();
	buffer.FillZ();
	buffer[0] = (TUint8) 0x08; // ICMP type = 8.
	buffer[1] = (TUint8) 0x00; // ICMP code = 0.
	buffer[2] = (TUint8) 0xF7; // ICMP checksum high byte.
	buffer[3] = (TUint8) 0xFF; // ICMP checksum low byte.
	
	// NB the rest of the buffer is zero
	// hence the checksum (0xFFFF - 0x800) since 0x8
	// is the only non-zero element of the buffer

	// set the length of the data to be sent to that specified (payload size!).
	buffer.SetLength(aPayloadSize);

	TRequestStatus status;

	// send the data out over the socket.
	iSocket.SendTo(buffer, iDestAddr, 0, status);
	User::WaitForRequest(status);

	return status.Int();
	}

/**
 Receives UDP packet from the network
 @param aPayloadSize The size of the packet to receive
 @return Error code
 */
TInt CRConnection::RecvUdpData(TUint aPayloadSize)
	{
	
	TInt timeoutInSecs = 30;
	RTimer timer;
	
	/* 
	Creates a timer so that we can handle error situation arising due to lost udp packet
	and dont completely rely on being errored by esock alone. This code waits for 30sec for
	udp data packet to arrive from the server. If the data is not available within this
	time the socket waiting for data is closed and the error value of KErrTimedOut is 
	returned. 
	*/
	TInt ret;
	if ((ret = timer.CreateLocal()) != KErrNone)
		{
		timer.Close(); // closing the timer.
		return ret;
		}
	TRequestStatus timerStatus;
	timer.After(timerStatus, timeoutInSecs * 1000000); //setting timer.
	
	TBuf8<KMaxBufferLength> buffer;
	buffer.Zero();
	buffer.SetLength(aPayloadSize);

	TRequestStatus status;
	iSocket.RecvFrom(buffer, iDestAddr, 0, status); //waiting on data from the server.
	User::WaitForRequest(status, timerStatus);

	// Timer expired, and if no data is received cancel the receive operation.
	if(timerStatus != KRequestPending)
		{
		iSocket.CancelAll();
		User::WaitForRequest(status);
		timer.Close();
		return KErrTimedOut;
		}
	// otherwise cancel the timer.
	timer.Cancel();
	User::WaitForRequest(timerStatus);
	timer.Close();

	//when data is successfully received, check against the previously sent buffer data,
	//to see what is received is what was sent.
	if (status != KErrNone)
		return status.Int();
	else
		{   
		if(buffer[0] == 0x08)
			{
			if(buffer[1] == 0x00)
				{
				if(buffer[2] == 0xF7)
					{
					if(buffer[3] == 0xFF)
						{
						;	
						}
					else 
						{
						_LIT(KFourthDataSet,"Fourth set of data not received\n");
						iConsole->Printf(KFourthDataSet);
						}		
					}
				else
					{
					_LIT(KThirdDataSet,"Third set of data not received\n");
					iConsole->Printf(KThirdDataSet);	
					}	

				}
			else
				{
				_LIT(KSecondDataSet,"Second set of data not received\n");
				iConsole->Printf(KSecondDataSet);	
				}
			}
		else
			{
			_LIT(KFirstDataSet,"First set of data not received\n");
			iConsole->Printf(KFirstDataSet);	
			}	
			
		}
	return KErrNone;
	}

/**
 Sends and receives data.
 */
void CRConnection::DataSendAndReceive(TUint aPayloadSize)
	{
	//Sends UDP Data.
	TInt err = SendUdpData(aPayloadSize);
	if (err != KErrNone)
		{
		_LIT(KDataSentErrNote,"DataSend Failed\n");
		iConsole->Printf(KDataSentErrNote);
		return ;	
		}
	else
		{
		_LIT(KDataSentNote,"DataSend Successful\n");
		iConsole->Printf(KDataSentNote);	
		}

	//Receives UDP Data.
	err = RecvUdpData(aPayloadSize);
	if(err!=KErrNone)
		{
		_LIT(KDataReceiveErrNote,"DataReceive Failed\n");
		iConsole->Printf(KDataReceiveErrNote);
		return ;	
		}
	else
		{
		_LIT(KDataReceiveNote,"DataReceive Successful\n");
		iConsole->Printf(KDataReceiveNote);
		}	
		
	}

/**
 Gets and prints the amount of data transmitted and received.
 */	
void CRConnection::DataTransferredRequest()
	{
	TPckg<TUint> uplinkVolumeDes(0); //holds the amount of data received.
	TPckg<TUint> downlinkVolumeDes(0); //holds the amount of data transmitted.
	
	TRequestStatus datastatus;

	iConnection.DataTransferredRequest(uplinkVolumeDes, downlinkVolumeDes, datastatus);
	User::WaitForRequest(datastatus);
	
	if(datastatus==KErrNone)
		{
		iConsole->Printf(KUpLinkData,uplinkVolumeDes());
		iConsole->Printf(KDownLinkData,downlinkVolumeDes());	
		}
			
	}
/**
 Requests notification when an amount of data equal to a threshold value has been 
 sent and received. The threshold used here is 100 bytes.
 */	
void CRConnection::DataTransferNotificationRequest()
	{
	const TUint KHundredBytes = 100; //The threshold value for notification.
	const TUint KThousandBytes = 1000; //The total amount of data to send/receive.
	TPckg< TUint > uplinkPkg(0); //Holds the amount of data sent.
	TPckg< TUint > downlinkPkg(0);//Holds the amount of data received.
	TRequestStatus dataSentStatus,dataReceivedStatus;
	
	//Registers for data sent notification. The process gets notified when 100 bytes of data are sent. 
	iConnection.DataSentNotificationRequest(KHundredBytes, uplinkPkg, dataSentStatus );
	
	//Registers for data received notification. The process gets notified when 100 bytes of data are received.
	iConnection.DataReceivedNotificationRequest(KHundredBytes, downlinkPkg, dataReceivedStatus );
	iConsole->Printf(KDataTransferNote);
	
	//Sends and receives 1000 bytes of data.
	DataSendAndReceive(KThousandBytes);
		
	/*
	Gets the total data sent/received = datavolume+UdpPacketOverhead
	here datavolume=1000 bytes,thus total data send/receive > 1000 as 
	some overhead gets added on top of the actual datavolume.
	*/
	
	User::WaitForRequest(dataSentStatus);
	User::WaitForRequest(dataReceivedStatus);
	if ((dataSentStatus.Int())== KErrNone)
		{
		_LIT(KSentSuccessNote,"DataSentNotificationRequest is successful and\n ");	
		iConsole->Printf(KSentSuccessNote);
		}
	else
		{
		_LIT(KSentFailureNote,"DataSentNotificationRequest has failed and \n");
		iConsole->Printf(KSentFailureNote);
		}
	iConsole->Printf(KUpLinkData,uplinkPkg());
					
	if (dataReceivedStatus.Int()==KErrNone)
		{
		_LIT(KReceivedSuccessNote,"DataReceivedNotificationRequest is successful and \n");	
		iConsole->Printf(KReceivedSuccessNote);
		}
	else 
		{
		_LIT(KReceivedFailureNote,"DataReceivedNotificationRequest has failed and \n");
		iConsole->Printf(KReceivedFailureNote);
		}		
	iConsole->Printf(KDownLinkData,downlinkPkg());	

	}
	
void CRConnection::DisplaySectionToDemo(const TDesC& aText)
	{
	TBuf<120> newtext;
	newtext.Append(KTab);
	newtext.Append(aText);
	newtext.Append(KTab);
	newtext.Append(KNewLine);
	iConsole->Printf(KRow1);
	iConsole->Printf(KRow2);
	iConsole->Printf(newtext);
	iConsole->Printf(KRow3);
	iConsole->Printf(KNewLine);
	}

/**
 Demonstrates the following RConnection functionality:
 1) requests connection/disconnection (interface up/down) notification
 2) starts an explicit connection without Commdb override
 3) obtains and prints connection progress information
 4) prepares socket for data transfer
 5) sends and receives data over the socket 
 6) prints the amount of data transferred
 7) closes the socket and connection
 */	
void CRConnection::DemoApiWithoutDbOverrideL()
	{
	_LIT(KDemoApiWithoutDbOverride,"RConnection API without CommDb override\n");
	iConsole->ClearScreen();
	DisplaySectionToDemo(KDemoApiWithoutDbOverride);
	
	TRequestStatus status;
	TInterfaceNotificationBuf info;
	//Makes a request for notification when the interface goes up i.e. the connection starts.
	iMonitor.AllInterfaceNotification(info,status);
		
	//Makes an explicit connection without CommDb override. 
	TInt err=ConnectWithoutDbOverrideL();
	if(err!=KErrNone)
		return;
	
	//Waits on TRequestStatus until the interface is up.
	User::WaitForRequest(status);
	if(info().iState==EInterfaceUp)
		{
		iConsole->Printf(KInterfaceUp);	
		}				
	
	//Gets and prints progress notification for the connection.
	GetProgressNotification();
	User::After(KDelay);  // waits for a while and then proceeds. This helps to get a sequential flow.
	
	err=PrepareSocket(); // Once the socket is ready for data transfer, send and receive operations can be carried out.
	if(err!=KErrNone)
		{
		iConsole->Printf(KSocketErrNote);
		return;	
		}
	else
		{
		iConsole->Printf(KSocketSetNote);	
		}	
	
	iConsole->Printf(KDataTransferNote);
	
	const TUint KPayLoadSize=512;
	//Sends and receives data through the socket.
	DataSendAndReceive(KPayLoadSize);
	
	DataTransferredRequest();// Gets and prints the amount of data that has been transferred.
		
	User::After(KDelay);//Wait for a while, before proceeding.	

	//Makes a request for notification when the interface goes down i.e. the connection closes.
	iMonitor.AllInterfaceNotification(info,status);	
		
	// All the operations related to the socket and connection are done, so close both.
	iSocket.Close();
	iConsole->Printf(KConnectionClosingNote);
	iConnection.Close();
	iConsole->Printf(KWaitNote); 
	
	//Waits for the 'interface down' notification
	User::WaitForRequest(status);
	if(info().iState==EInterfaceDown)
		{
		iConsole->Printf(KInterfaceDown);	
		}
		
	iConsole->Printf(KContinueNote);
	iConsole->Getch();

	}
	
/**
 Demonstrates the following RConnection functionality:
 1) requests connection/disconnection (interface up/down) notification
 2) starts an explicit connection with CommDb override
 3) obtains and prints connection progress information
 4) prepares socket for data transfer
 5) sends and receives data over the socket
 6) requests data send and receive notification
 7) closes the socket and connection
 */		
void CRConnection::DemoApiWithDbOverrideL()
	{
	
	_LIT(KDemoApiWithDbOverride,"RConnection API with CommDb override\n");
	iConsole->ClearScreen();
	DisplaySectionToDemo(KDemoApiWithDbOverride);
		
	TRequestStatus status;
	TInterfaceNotificationBuf info;
	
	// Requests notification when the connection is established.
	// Notification request is issued before calling Start().
	iMonitor.AllInterfaceNotification(info,status);
	
	//Start an explicit connection with CommDb override. 
	TInt err=ConnectWithDbOverrideL();
	if(err!=KErrNone)
		return;
	
	//Waits on TRequestStatus until the interface is up.
	User::WaitForRequest(status);
	if(info().iState==EInterfaceUp)
		{
		iConsole->Printf(KInterfaceUp);	
		}	

	//Gets and prints progress notification for the connection.
	GetProgressNotification();
	
	//Waits for a while before next operation.
	User::After(KDelay);
	
	err=PrepareSocket(); //Once the socket is ready for data transfer, send and receive operations can be carried out.
	if(err!=KErrNone)
		{
		iConsole->Printf(KSocketErrNote);
		return;	
		}
	else	
		{
		iConsole->Printf(KSocketSetNote);	
		}
	//Sends and receives data and requests notification when a threshold amount of data is transferred.
	DataTransferNotificationRequest(); 
	User::After(KDelay);

	//Makes a request for notification when the interface goes down i.e. the connection closes.
	iMonitor.AllInterfaceNotification(info,status);
	 	
	//Closes socket 
	iSocket.Close();
	iConsole->Printf(KConnectionClosingNote);
		
	//Closes the connection.
	iConnection.Close();
	iConsole->Printf(KWaitNote); 
	
	//Waits on the TRequestStatus until the interface is down 
	User::WaitForRequest(status);
	if(info().iState==EInterfaceDown)
		{
		iConsole->Printf(KInterfaceDown);	
		}	
	
	iConsole->Printf(KContinueNote);
	iConsole->Getch();
	
	}

/**
 Attaches a connection to an undelying interface.
 This demonstrates the following RConnection functionality:
 1) Opens 2 connections and starts one of them. The other is not yet associated with an underlying interface.
 2) Gets the connection info of the 1st connection.
 3) Attaches the 2nd connection to the existing interface,
 4) Gets the current state of the attached connection.
 5) Closes both connections.
 */
void CRConnection::AttachToExistingInterfaceL()
	{
	
	_LIT(KAttachToConnection,"Attaching to a connection\n");
	iConsole->ClearScreen();
	DisplaySectionToDemo(KAttachToConnection);
	
	// Attaching to a connection is done as follows:
	// 1)first open a connection interface.
	// 2)on this connection interface, obtain the no. of active connections and their connection info.
	// 3)a second connection is opened.
	// 4)this 2nd connection is attached to the existing active connection interface.
	 
	RConnection conn;
	
	//Opens the first RConnection.
	TInt err = conn.Open(iSocketServ);
	if(err==KErrNone)
		{
		CleanupClosePushL(conn);	
		}
	else
		{
		iConsole->Printf(KOpenErr);
		return;	
		}	
	
	//Opens the 2nd connection. 
	err =  iConnection.Open(iSocketServ);
	if(err!=KErrNone)
		{
		iConsole->Printf(KOpenErr);
		return;	
		}	
	
	//Starts the 1st connection.
	err=conn.Start();
	if(err!=KErrNone)
		{
		iConsole->Printf(KStartErr);
		return;	
		}
	
	// Enumerates the total number of connections on the 1st connection interface.
	// This call returns a count of 1.	
	TUint connectionCount;
	err = conn.EnumerateConnections(connectionCount);
	if((err != KErrNone) || (connectionCount < 1))
		{
		return;
		}
	
	
	// Gets connection info for the 1st connection (conn) which will be used by the 
	// 2nd connection to attach to. 
	TConnectionInfoBuf connectionInfo;
	err = conn.GetConnectionInfo(connectionCount, connectionInfo); // gets connection info about the indexed connection
																   // (the 1st connection).
	if(err != KErrNone)
		{
		return;
		}

	// Attaches iConnection to the existing underlying interface.
	// The connection is used for normal data transfer.
	// Both connections are now actively associated with an interface.
	err = iConnection.Attach(connectionInfo,RConnection::EAttachTypeNormal);
	if(err==KErrNone)
		{
		_LIT(KAttachNote,"Attached to interface\n");	
		iConsole->Printf(KAttachNote);			
		}
	else
		return;	
	
	// Gets progress info and prints it to the console. 
    TNifProgress progress;
    err = iConnection.Progress(progress);
    if(err==KErrNone&&progress.iError==KErrNone)
	    {
	    if((TUint)(progress.iStage)==KConnectionUp)
	    	iConsole->Printf(KConnectionUpNote);	
	    }
	
	User::After(KDelay);  //waits for a while before proceeding.
	
	iConsole->Printf(KConnectionClosingNote);
	 	
	//closes both connections.
	iConnection.Close();
	CleanupStack::PopAndDestroy(&conn); //conn.
	
	_LIT(KConnectionClosedNote,"Connection Closed\n");
	iConsole->Printf(KConnectionClosedNote);
	
	}

