// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains definition of functions defined in the CLongNumber class.
//



/**
 @file
*/

#include "longnumber.h"

/**
Creates an object of the CLongNumber class.
@param aConsole The user console.
@return A pointer to the object of the CLongNumber class.
*/
CLongNumber* CLongNumber::NewL(CConsoleBase* aConsole)
	{
	CLongNumber* self = new (ELeave) CLongNumber(aConsole);
	self->AddToScheduler();
	return self;
	}

/**
Constructor.
@param aConsole The user console.
*/
CLongNumber::CLongNumber(CConsoleBase* aConsole):
												// Constructor of the base class.
												CActive(CActive::EPriorityUserInput),
												// Create the linked list.
												iNumber(_FOFF(TDigit,iSLink)),
												iConsole(aConsole),
												// Initialize the iterator.
												iIterator(iNumber)
	{
	}


/**
Adds this active object to the active scheduler
*/
void CLongNumber::AddToScheduler()
	{
	CActiveScheduler::Add(this);
	}

/**
Handles the key-press events for this active object.
*/
void CLongNumber::RunL()
	{
	// Get the key code.
	TUint8 option = iConsole->KeyCode();
	// Print the selected option.
	_LIT(KTextFormat,"%c\n");
	iConsole->Printf(KTextFormat,option);

	TInt number = option - (TUint)'0';
	// The user has entered a number.
	if(number <10 && number > -1)
		{
		// Append to the list based on iOption.
		switch(iOption)
			{
			// Call the TSglQue::AddFirst() function.
			case EAddFirst:
				{
				// Create an object of the TDigit class.
				TDigit* digit = new (ELeave) TDigit;
				// Set TDigit::iDigit to the number entered by the user.
				digit->Set(number);
				// Add TDigit to the start of the list.
				iNumber.AddFirst(*digit);
				iOption = ETOptionNone;
				}
				// Print the number.
				PrintNumber();
				break;
			// Call the TSglQue::AddLast() function.
			case EAddLast:
				{
				// Create an object of the TDigit class.
				TDigit* digit = new (ELeave) TDigit;
				// Set TDigit::iDigit to the number entered by the user.
				digit->Set(number);
				// Add TDigit to the end of the list.
				iNumber.AddLast(*digit);
				iOption = ETOptionNone;
				}
				// Print the number.
				PrintNumber();
				break;
			case ETOptionNone:
				_LIT(KInvalidOption,"Invalid Option!\n");
				iConsole->Printf(KInvalidOption);
				break;
			default:
				iOption = ETOptionNone;
				break;
			}
		// Generate an asynchronous read request.
		ReadNumber();
		}
	// The user has not entered a number.
	// Hence, process this request as an option,
	// which represented by the TOption enumeration.
	else
		{
		switch(option)
			{
			case 'f':
				{
				// Add first.
				iOption = EAddFirst;
				_LIT(KTextEnterDigit,"\nEnter a digit\n");
				iConsole->Printf(KTextEnterDigit);
				// Read a digit.
				ReadFunc();
				}
				break;
			case 'l':
				{
				// Add last.
				iOption = EAddLast;
				_LIT(KTextEnterDigit,"\nEnter a digit\n");
				iConsole->Printf(KTextEnterDigit);
				// Read a digit.
				ReadFunc();
				}
				break;
			case 'r':
				{
				if(iNumber.IsEmpty())
					{
					_LIT(KTextEmpty,"Empty Queue !\n");
					iConsole->Printf(KTextEmpty);
					ReadNumber();
					break;
					}
				// Remove the last element in the list.
				TDigit* ptr = iNumber.Last();
				iNumber.Remove(*ptr);
				delete ptr;
				// Print the number.
				PrintNumber();
				// Read a digit.
				ReadNumber();
				}
				break;
			default:
				CActiveScheduler::Stop();
			}
		}
	}

/**
Issues an outstanding request to get a keystroke from the console.
*/
void CLongNumber::ReadFunc()
	{
	// Wait for a key press event.
	iConsole->Read(iStatus);
	SetActive();
	}

/**
Reads a number from the console.
*/
void CLongNumber::ReadNumber()
	{
	_LIT(KTextMenu,"\nMenu\n***\nAdd [f]irst digit to number\nAdd [l]ast digit to number\n[r]emove last digit from number\nAny other NON-NUMERIC key to stop\n***\n");
	iConsole->Printf(KTextMenu);
	ReadFunc();
	}

/**
Prints the number.
*/
void CLongNumber::PrintNumber()
	{
	if(iNumber.IsEmpty())
		{
		_LIT(KTextEmpty,"Empty Queue !\n");
		iConsole->Printf(KTextEmpty);							
		}
	_LIT(KTextNumber,"The number is: ");
	iConsole->Printf(KTextNumber);
	// Set the iterator to the first element of the list.
	iIterator.SetToFirst();

	// Iterate the list.
	while(iIterator != NULL)
		{
		// Get the digit.
		TDigit digit = *iIterator;
		// Print the digit.
		_LIT(KTextInt,"%d");
		iConsole->Printf(KTextInt,digit.Get());
		// Go to the next element of the list.
		iIterator++;
		}
	}

/**
Sets the iterator to the first element of the list.
*/
void CLongNumber::InitializeIter()
	{
	iIterator.SetToFirst();
	}

/**
Gets the integer number of the TDigit node, which is pointed to by the iterator.
@param aValue The reference to an integer. On return,
this stores the value of the digit.
@return EFalse if the iterator has reached the end of the list, else ETrue.
*/
TBool CLongNumber::GetNumber(TInt& aValue)
	{
	if(iIterator == NULL)
		{
		// The iterator has reached the end of the list.
		return EFalse;
		}
	TDigit digit = *iIterator;
	aValue = digit.Get();
	return ETrue;
	}

/**
Moves the iterator to the next node of the list.
@return EFalse, if the iterator has reached the end of the list, else ETrue.
*/
TBool CLongNumber::Slide()
	{
	// Set the iterator to point to the next element.
	iIterator++;
	if(iIterator == NULL)
		{
		// The iterator has reached the end of the list.
		return EFalse;
		}
	return ETrue;
	}

/**
Cancel any outstanding request.
*/
void CLongNumber::DoCancel()
	{
	if(IsActive())
		{
		// Cancel any outstanding read requests.
		iConsole->ReadCancel();
		}
	}

/**
Destructor.
The TDigit objects were allocated memory at runtime.
Iterate over the list to delete them.
*/
CLongNumber::~CLongNumber()
	{
	// Delete the elements from the list
	if(iIterator)
		{
		iIterator.SetToFirst();
		// Iterate the list and delete all TDigit objects
		TDigit *ptr = iIterator++;
		while (ptr != NULL)
			{
			delete ptr;
			ptr = iIterator++;
			}
		}
	}
