// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This encapsulates both issuing a request for getting pointer events to an asynchronous service provider
// and handling the completed requests.
//



/**
 @file
*/

#include "transparent.h"

/**
Allocates and constructs a CEventHandler object using two phase construction.
@param aWs Current window server session
*/
CEventHandler* CEventHandler::NewL(RWsSession& aWs)
	{
	CEventHandler* eventHandler = new(ELeave) CEventHandler(aWs);
	CleanupStack::PushL(eventHandler);
	eventHandler->ConstructL();
	CleanupStack::Pop(eventHandler);
	return eventHandler;
	}

/**
Parameterised constructor
@param aWs Current window server session
*/
CEventHandler::CEventHandler(RWsSession& aWs) : CActive(CActive::EPriorityStandard),iWs(aWs)
	{
	}

/**
Adds the active object to the scheduler.
*/
void CEventHandler ::ConstructL()
	{
	CActiveScheduler::Add(this);
	}

/**
Destructor
*/
CEventHandler ::~CEventHandler ()
	{
	Cancel();
	}

/**
Implements cancellation of an outstanding request.
This function is called as part of the active object's Cancel().
*/
void CEventHandler ::DoCancel()
	{
	iWs.EventReadyCancel();
	}

/**
Handles an active objects request completion event
and runs the desired task.
*/
void CEventHandler::RunL()
	{
	iWs.GetEvent(iWsEvent);
	TInt eventType = iWsEvent.Type();

	// Take action for the event that occurred
	switch (eventType)
		{
		// Events global within window group
		case EEventNull:
			break;
		case EEventPointer:
			{
			TPointerEvent& pointerEvent = *iWsEvent.Pointer();
			HandlePointerEvent(pointerEvent);
			}
			break;
		}
	// Maintain outstanding request
	IssueRequest();
	}

/**
Determines the type of pointer event and prints out
which button was pressed as an info message.
@param aPointerEvent Event determined by RunL()
*/
void CEventHandler::HandlePointerEvent(TPointerEvent& aPointerEvent)
	{
	switch (aPointerEvent.iType)
		{
		case TPointerEvent::EButton1Down:
			{
			InfoMessage(KEventButton1Pressed);
			break;
			}
			case TPointerEvent::EButton2Down:
			{
			InfoMessage(KEventButton2Pressed);
			break;
			}
			case TPointerEvent::EButton3Down:
			{
			InfoMessage(KEventButton3Pressed);
			break;
			}
			default:
			{
			break;
			}
		}
	}

/**
Indicates that the active object has issued a request
and that the request is now outstanding.
*/
void CEventHandler::IssueRequest()
	{
	if(!IsActive())
		{
		iWs.EventReady(&iStatus);
		SetActive();
		}
	}

/**
Displays a message in the system default corner of the screen.
@param TDesC& aMsg A descriptor message
*/
void CEventHandler::InfoMessage(const TDesC& aMsg)
	{
	CEikonEnv *obj = CEikonEnv::Static();
	obj->InfoMsg(aMsg);
	}

