// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "processserversession.h"


/*****************CProcessServerSession**********************/
/**
	Starts two phase construction
*/
CProcessServerSession* CProcessServerSession::NewL(CProcessServer& aServer)
	{
	CProcessServerSession* s = new(ELeave) CProcessServerSession(aServer);
	CleanupStack::PushL(s);
	s->ConstructL();
	CleanupStack::Pop();
	return s;	
	}
/**
	Constructor
*/	
CProcessServerSession::CProcessServerSession(CProcessServer& aServer)
:iServer(aServer)
	{
	
	}

void CProcessServerSession::ConstructL()
	{
	}
/**
	Destructor
*/		
CProcessServerSession::~CProcessServerSession()
	{
	delete iAsyncRequestHandler;		
	iServer.CloseLogicalChannel();
	iServer.DecrementRefCount();
	}
/**
	Completes construction of this server-side client session object when a client makes a 
	connection request and increases the reference count
*/
void CProcessServerSession::CreateL()
//
// 2nd phase constructor for sessions - called by the CServer framework
//
	{
	iServer.IncrementRefCount();		
	//load dummy ldd
	User::LeaveIfError(iServer.LoadDevice());
	iAsyncRequestHandler = CAsyncHandler::NewL(iServer); 	
	}
/**
	Handles the servicing of a client request that has been passed to the server	
*/
void CProcessServerSession::ServiceL(const RMessage2& aMessage)
	{
	TInt r = KErrNone;
	switch(aMessage.Function())
		{				
		case ELoadDeviceDriver:
			aMessage.Complete(iServer.LoadDevice());
			break;
		
		case EOpenDriver:
			aMessage.Complete(iServer.OpenLogicalChannel());
			break;
			
		case EDummyLDDSendData: //async
			r = iAsyncRequestHandler->ProcessRequest(aMessage);
			if (r!=KErrNone)
				aMessage.Complete(r);
			break;
			
		case EDummyLDDSendDataCancel: //cancel async
			iAsyncRequestHandler->Cancel(); 
			aMessage.Complete(KErrNone); 
			break;
			
		case EUnloadDeviceDriver:
			aMessage.Complete(iServer.UnloadDevice());
			break;
			
		default:
			User::Leave(KErrNotSupported);
			break;
		}

	}
		


/*****************CAsyncHandler**********************/
/**
	Active object class to handle asynchronous requests
*/	
CProcessServerSession::CAsyncHandler* CProcessServerSession::CAsyncHandler::NewL(CProcessServer& aServer)
	{
	CAsyncHandler* self = new(ELeave) CAsyncHandler(aServer);
	CleanupStack::PushL(self);
	self ->ConstructL();
	CleanupStack::Pop();
	return self;
	}
/**
	Destructor
*/
CProcessServerSession::CAsyncHandler::~CAsyncHandler()
	{
	Cancel();
	iMessageArray.Close();
	}
/**
	Handle the event
*/
void CProcessServerSession::CAsyncHandler::RunL()
	{
	// complete the request
	Complete(iStatus.Int());
	
	//continue to execute next request if there is any
	ExecuteFirstRequestInArray();
	}
/**
	Cancel request	
*/
void CProcessServerSession::CAsyncHandler::DoCancel()
	{
	iServer.CancelSendData();
	// complete the request
	Complete(iStatus.Int());
	}
/**
	Constructor
*/
CProcessServerSession::CAsyncHandler::CAsyncHandler(CProcessServer& aServer)
	:CActive(EPriorityStandard), iServer(aServer)
	{
	CActiveScheduler::Add(this);
	}

void CProcessServerSession::CAsyncHandler::ConstructL()
	{
	
	}
/**
	Implement simple message queue
	@param aMessage Message to be stored in the message queue
	@return KErrNone or standard error code. 
*/	
TInt CProcessServerSession::CAsyncHandler::ProcessRequest(const RMessage2& aMessage)
	{
	TInt ret;
	//store message
	TMessage newMessage;
	newMessage.Message() = aMessage;
	ret= iMessageArray.Append(newMessage);
	if (ret != KErrNone)
		return ret;
	
	// kick off the first request in the array when there is only one in the array 
	if (iMessageArray.Count() == 1) 
		ret= ExecuteFirstRequestInArray();

	return ret;
	}
/** 
	Read the first message in the message queue
	@return KErrNone or standard error code.
*/
TInt CProcessServerSession::CAsyncHandler::ExecuteFirstRequestInArray()
	{
	TInt r = KErrNone;
	if (iMessageArray.Count() != 0)
		{
		const RMessage2& message = iMessageArray[0].Message();
		switch (message.Function())
			{
			case EDummyLDDSendData:
				{
				TBuf8<KMaxMessageLen> sendObject;
				r= message.Read(0, sendObject);
				if (r == KErrNone)
					{
					r = iServer.SendDataToDevice(iStatus, sendObject);
					if (r==KErrNone)
						SetActive();
					}
				}
				break;
			
			default:
				break;
			}
		
		}
	return r; 
	}
/**
	Complete issued request
*/	
void CProcessServerSession::CAsyncHandler::Complete(TInt aResult)
	{
	iMessageArray[0].Message().Complete(aResult);
	iMessageArray.Remove(0);
	iServer.UpdateDriverState(CProcessServer::ELogicalChannelOpened);
	}

/***************TMessage****************/
/**
	Constructor
*/
const RMessage2& CProcessServerSession::CAsyncHandler::TMessage::Message() const
	{
	return iMessage;
	}
	
RMessage2& CProcessServerSession::CAsyncHandler::TMessage::Message()
	{
	return iMessage; 
	}

//EOF
