// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This example program demonstrates the TChar, TCharF, TCharLC and TCharUC character classes.
//



/**
 @file 
*/

#include "tcharexample.h"

static CConsoleBase* console;

/**
Constructor
*/
CTCharExample::CTCharExample()
	{
	}

void CTCharExample::ConstructL()
	{
	_LIT(KTitle, "\r\nTChar Example" );
	console = Console::NewL(KTitle, TSize(KConsFullScreen, KConsFullScreen));
	
	_LIT(KWelcome, "\r\nWelcome to the TChar example application");
	console->Printf(KWelcome);
	}

/**
Destructor
*/
CTCharExample::~CTCharExample()
	{
	delete console;
	}

/**
Allocates and constructs a CTCharExample object.
@return A CTCharExample object
*/
CTCharExample* CTCharExample::NewL()
	{
	CTCharExample* self = new(ELeave)CTCharExample();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/** 
Extracts and prints information about a character using the TChar API.
@param aCharValue The character of interest.
*/
void PrintIsXXXFunction(TChar aCharValue)
	{
	if(aCharValue.IsLower())
		{
		_LIT(KLower," The character is lowercase\r\n");
		console->Printf(KLower);
		}
	if(aCharValue.IsUpper())
		{
		_LIT(KUpper," The character is uppercase\r\n");
		console->Printf(KUpper);
		}
	if(aCharValue.IsAlpha())
		{
		_LIT(KAlpha," The character is alphabetic\r\n"); 
		console->Printf(KAlpha);
		}
	if(aCharValue.IsDigit())
		{
		_LIT(KDigit," The character is a digit\r\n");
		console->Printf(KDigit);
		}
	if(aCharValue.IsHexDigit())
		{
		_LIT(KHexDigit," The character is a hexadecimal digit\r\n");
		console->Printf(KHexDigit);
		}
	if(aCharValue.IsSpace())
		{
		_LIT(KSpace," The character is a space\r\n");
		console->Printf(KSpace);
		}
	if(aCharValue.IsPunctuation())
		{
		_LIT(KPunctuation," The character is punctuation\r\n");
		console->Printf(KPunctuation);
		}
	else if(aCharValue.IsPrint())
		{
		_LIT(KPrint," The character is printable\r\n");
		console->Printf(KPrint);
		}
	}
    
/** 
Demonstrates the use of the TChar::IsXXX() functions
*/	
void TCharIsXXXFunctions()
	{
	_LIT(KEnterAKey,"\r\nEnter a character to process with the TChar::IsXXX() functions: ");
	console->Printf(KEnterAKey);
	
	TChar charValue = console->Getch();
	_LIT(KPrintChar,"%C\r\n");
	console->Printf(KPrintChar, (TUint)charValue);
	
	// Get and print the character type using the IsXXX() functions.
	PrintIsXXXFunction(charValue);
	}
	
/**
Demonstrates the use of the TChar::GetXXX() functions and member enumerations.
*/
void TCharGetXXXFunctions()
	{
	_LIT(KEnterAKey,"\r\nEnter a character to process with the TChar:GetXXX() functions: ");
	console->Printf(KEnterAKey);
	_LIT(KPrintChar,"%C\r\n");
	TChar charValue = console->Getch();
	
	console->Printf(KPrintChar, (TUint)charValue);

	// Gets the combining class of the character.
	TInt getCombiningClass = charValue.GetCombiningClass();
	_LIT(KCharCombNotClassMsg," The character does not have a combining class\r\n");
	_LIT(KCharCombClassMsg," The character has %d combining classes\r\n");
	if(getCombiningClass == 0)
		console->Printf(KCharCombNotClassMsg);
	else
		console->Printf(KCharCombClassMsg, getCombiningClass);
	
	_LIT(KENeutalWidthMsg," ENeutralWidth: This character type includes 'ambiguous width' as defined in Unicode Technical Report 11: East Asian Width\r\n");
	_LIT(KEHaldWidthMsg," EHalfWidth: This character occupies a single cell\r\n");
	_LIT(KEFullWidthMsg," EFullWidth: This character occupies 2 cells\r\n");
	_LIT(KENarrowMsg," ENarrow: This character is narrow and has explicit full-width counterparts\r\n");
	_LIT(KEWideMsg," EWide: This character is wide\r\n");
	_LIT(KNoWidthMsg," This character has no width defined\r\n");
	// Gets and prints the Chinese, Japanese, Korean (CJK) notional width of the character.
	TChar::TCjkWidth cjkWidth = charValue.GetCjkWidth();
	switch(cjkWidth)
		{
		case TChar::ENeutralWidth:
			console->Printf(KENeutalWidthMsg);
			break;
		case TChar::EHalfWidth:
			console->Printf(KEHaldWidthMsg);
			break;
		case TChar::EFullWidth:
			console->Printf(KEFullWidthMsg);
			break;	
		case TChar::ENarrow:
			console->Printf(KENarrowMsg);
			break;
		case TChar::EWide:
			console->Printf(KEWideMsg);
			break;
		default:
			console->Printf(KNoWidthMsg);
		}

	// Prints the different cases if the character is alphabetic.
	if(charValue.IsAlpha())
		{
		// Gets the character after conversion to uppercase (or returns the character itself if no uppercase form exists).
		TUint getCaseValue = charValue.GetUpperCase();
		_LIT(KAferConv2UpMsg," The character after conversion to upper case is %C:\r\n");
		console->Printf(KAferConv2UpMsg, getCaseValue);
	
		// Gets the character after conversion to lowercase (or the character itself if no lowercase form exists).
		getCaseValue = charValue.GetLowerCase();
		_LIT(KAferConv2LowerMsg," The character after conversion to lower case is %C:\r\n"); 
		console->Printf(KAferConv2LowerMsg, getCaseValue);

		// Gets the character after conversion to title-case (or the character itself if no title-case form exists).
		getCaseValue = charValue.GetTitleCase();
		_LIT(KAferConv2TitleMsg," The character after conversion to title case is %C:\r\n");
		console->Printf(KAferConv2TitleMsg, getCaseValue);
		}
	}
	

/**
Extracts information about a character using TCharInfo 
*/ 
void TCharInfoFunctions()
	{
	_LIT(KEnterAKey,"\r\nTCharInfo holds all character information\r\nEnter a character to process with TCharInfo: ");
	console->Printf(KEnterAKey);
	
	TChar enteredCharacter = console->Getch();
	_LIT(KPrintChar,"%C\r\n");
	console->Printf(KPrintChar, (TUint)enteredCharacter);
	
	// A structure to hold information about the character.
	TChar::TCharInfo charInfo;
	
	enteredCharacter.GetInfo(charInfo);
	
	// Gets the combining class of the entered character.
	// This determines the relationship between characters in a string
	TInt numberOfCombiningClass = charInfo.iCombiningClass;
	_LIT(KCharCombNotClassMsg," The character does not have a combining class\r\n");
	_LIT(KCharCombClassMsg," The character has %d combining classes\r\n");
	if(numberOfCombiningClass <= 0)
		console->Printf(KCharCombNotClassMsg);
	else
		console->Printf(KCharCombClassMsg, numberOfCombiningClass);

	// Returns True if the character is mirrored.
	// Mirrored characters can change direction according to the directionality of the surrounding characters
	TBool booleanValue = charInfo.iMirrored;
	_LIT(KCharMirroredMsg," The character is mirrored\r\n");
	_LIT(KCharNotMirroredMsg," The character is not mirrored\r\n");
	if(booleanValue)	
		console->Printf(KCharMirroredMsg);
	else
		console->Printf(KCharNotMirroredMsg);
	
	// Gets the numeric value of the character (-1 if none, -2 if a fraction).
	TInt getNumericValue = charInfo.iNumericValue;
	_LIT(KCharNoNumericMsg," The character has no numeric value\r\n");
	_LIT(KCharNumericFractionMsg, " The character is a fraction\r\n");
	if(getNumericValue == -1)	
		console->Printf(KCharNoNumericMsg);
	else if(getNumericValue == -2)	
		console->Printf(KCharNumericFractionMsg);
	
	// Print the different cases if the character is alphabetic.
	if(enteredCharacter.IsAlpha())
		{
		// Gets and prints the title-case form of the character.
		TUint caseValue = charInfo.iTitleCase;
		_LIT(KAferConv2TitleMsg," The character after conversion to title case is %C\r\n");
		console->Printf(KAferConv2TitleMsg, caseValue);
	
		// Gets and prints the upper case form of the character.
		caseValue = charInfo.iUpperCase;
		_LIT(KAferConv2UpMsg," The character after conversion to upper case is %C\r\n");
		console->Printf(KAferConv2UpMsg, caseValue);	
	
		// Gets and prints the lower case form of the character.
		caseValue = charInfo.iLowerCase;
		_LIT(KAferConv2LowerMsg," The character after conversion to lower case is %C\r\n"); 
		console->Printf(KAferConv2LowerMsg, caseValue);	
		} 
	}


/**
The TChar class main menu. 
*/
void CTCharExample::UseTCharFunctions()
	{
	//Split the TChar functions into logical groups. 
	TInt userChoice = 0;
		do
		{
		_LIT(KLoopMessage, "\r\nTChar function menu.\r\n 1 to use the IsXXX() functions,\r\n 2 to use the Getxxx() functions,\r\n 3 to use the TCharInfo class,\r\n or 0 to return to the main menu.\r\n");
		console->Printf(KLoopMessage);
		
		userChoice = console->Getch();
		
		switch(userChoice) 
			{
			case '1':
				// Use the TChar::IsXXX() functions
				TCharIsXXXFunctions();
				break;	
	
			case '2':
				// Use the TChar::GetXXX(); functions
		   		TCharGetXXXFunctions();
		   		break;
		
			case '3':
				// Use TCharInfo functions
				TCharInfoFunctions();	
				break;
					
			case '0':
				// Exit
				break;
				
			default:
				{
				_LIT(KEnterValidMsg, "\r\nEnter a valid character.\r\n");
				console->Printf(KEnterValidMsg);
				}
		        
			}
			
		} while (userChoice != '0');
	}

/**
Demonstrates the use of the TCharF class. 	
*/
void CTCharExample::UseTCharFFunctions()
	{	
	_LIT(KEnterALowerCaseChar,"\r\nTCharF ignores character differences like case and accent.\r\n Enter a character: ");
	console->Printf(KEnterALowerCaseChar);

	// Gets a character from the console.
	TChar enteredLowerChar = console->Getch();
	_LIT(KPrintChar,"%C\r\n");
	console->Printf(KPrintChar, (TUint)enteredLowerChar);
	
	_LIT(KEnterAnUpperCaseChar," Enter the same character but with a different case or accent.\r\n (use the 'Alt Gr' key to produce an accent): ");
	console->Printf(KEnterAnUpperCaseChar);

	// Gets a character from the console.
	TChar enteredUpperChar = console->Getch();
	console->Printf(KPrintChar, (TUint)enteredUpperChar);
	
	// TCharF folds a specified character so that character differences like case and accents are ignored.
	TCharF lowerCaseValue(enteredLowerChar);
	TCharF upperCaseValue(enteredUpperChar);
	
	// Test both lower and upper case value using TCharF.
	if (lowerCaseValue == upperCaseValue)
		{
		_LIT(KCharEquiMsg," The characters are equivalent\r\n");
		console->Printf(KCharEquiMsg);
		}
	else
		{
		_LIT(KCharNotEquiMsg, " The characters are not equivalent\r\n");
		console->Printf(KCharNotEquiMsg);
		}		
	}

/**
Demonstrates the use of TCharLC class. 
*/	
void CTCharExample::UseTCharLCFunctions()
	{
	
	_LIT(KEnterACharForTCharLC,"\r\nTCharLC converts a character to lower case\r\n Enter an upper case character to process with TCharLC: ");
	console->Printf(KEnterACharForTCharLC);

	// Gets a character from the console.
	TChar charValue = console->Getch();
	_LIT(KPrintChar,"%C\r\n");
	console->Printf(KPrintChar, (TUint)charValue);
	
	// Convert and print the character.
	TCharLC convertedCharToLowerCase(charValue);
		
	_LIT(KConvertedTCharLC, "\r\n After conversion the character is: ");
	console->Printf(KConvertedTCharLC);
	console->Printf(KPrintChar,(TUint)convertedCharToLowerCase);
	}
	
/**
Demonstrates the use of the TCharUC class. 
*/	
void CTCharExample::UseTCharUCFunctions()
	{	
	_LIT(KEnterACharForTCharUC,"\r\nTCharUC converts a character to upper case\r\n Enter a lower case character to process with TCharLC: ");
	console->Printf(KEnterACharForTCharUC);
	_LIT(KPrintChar,"%C\r\n");
	// Gets a character from the console.
	TChar charValue = console->Getch();
	console->Printf(KPrintChar, (TUint)charValue);
	
	// Convert and print the character.
	TCharUC convertedCharUC(charValue);
		
	_LIT(KConvertedTCharUC, "\r\n After conversion the character is: ");
	console->Printf(KConvertedTCharUC);
	console->Printf(KPrintChar,(TUint)convertedCharUC);
	}	
	

void MainL()
	{
	CTCharExample* app = CTCharExample::NewL();
	CleanupStack::PushL(app);

	TInt userChoice = 0;

	do
		{
		_LIT(KLoopMessage, "\r\nPress 1 to use the TChar functions,\r\n 2 to use the TCharF function,\r\n 3 to use the TCharLC function,\r\n 4 to use the TCharUC function,\r\n or 0 to exit the application.\r\n");
		console->Printf(KLoopMessage);
		
		userChoice = console->Getch();
		
		switch(userChoice) 
			{
			case '1':
				// Use functions belonging to the TChar character class.
				app->UseTCharFunctions();
				break;	
	
			case '2':
				// Use functions belonging to the TCharF character class.
		   		app->UseTCharFFunctions();
		   		break;
		
			case '3':
				// Use functions belonging to the TCharLC character class.
				app->UseTCharLCFunctions();	
				break;
	
			case '4':
				// Use functions belonging to the TCharUC character class.
				app->UseTCharUCFunctions();
				break;
			
			case '0':
				// Exit
				break;
			
			default:
				_LIT(KEnterValidMsg, "\r\nEnter a valid character.\r\n");			
				console->Printf(KEnterValidMsg);
			}
			
		} while (userChoice != '0');
		
		CleanupStack::PopAndDestroy(app);
	} 

extern TInt E32Main()
	{
	__UHEAP_MARK;

	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	TRAPD(err, MainL());
	if(err !=KErrNone)
		{
		_LIT(KFailed, "\nFailed to complete");
		User::Panic(KFailed, err);
		}	
	delete cleanup;

	__UHEAP_MARKEND;
	return KErrNone;
	}
  
  
  
  
 
 
 
