/*
 * Copyright  2008 Nokia Corporation.
 */

// INCLUDE FILES
#include <aknutils.h>       // Fonts
#include <w32std.h>         // key codes and events
#include "AudioStreamView.h"

#include <TXTRICH.H>


const TUint KNumberOfLines = 0;
const TUint KTextLimit = 0;


CAudioStreamView* CAudioStreamView::NewL(const TRect & aRect, 
                                         CAudioStreamEngine* aEngine)
    {
    CAudioStreamView* self = NewLC(aRect, aEngine);
    CleanupStack::Pop(self);
    return self;
    }

CAudioStreamView* CAudioStreamView::NewLC(const TRect & aRect, 
                                          CAudioStreamEngine* aEngine)
    {
    CAudioStreamView* self = new (ELeave) CAudioStreamView();
    CleanupStack::PushL(self);
    self->ConstructL(aRect, aEngine);
    return self;
    }

// ----------------------------------------------------------------------------
// CAudioStreamView::ConstructL(const TRect& aRect, 
//                                CAudioStreamEngine* aEngine)
//
// Standard EPOC 2nd phase constructor
// ----------------------------------------------------------------------------
void CAudioStreamView::ConstructL(const TRect& aRect, 
                                  CAudioStreamEngine* aEngine)
    {
    iEngine = aEngine;
    // create new window,
    CreateWindowL();

    // Create a rich text editor into which information can be logged
    iLogComponent = new(ELeave) CEikRichTextEditor;
    iLogComponent->ConstructL(this, KNumberOfLines, KTextLimit,
            CEikEdwin::EReadOnly | CEikEdwin::EAvkonDisableCursor);
    iLogComponent->CreateScrollBarFrameL()->SetScrollBarVisibilityL(
            CEikScrollBarFrame::EOff, CEikScrollBarFrame::EOn);

    // Set font
    #ifdef __WINS__
    _LIT(KFontArial,"Arial");
    TFontSpec MyeFontSpec (KFontArial, 8*10);
    MyeFontSpec.iTypeface.SetIsProportional(ETrue);
    // Set font height
    TCharFormatMask charFormatMask;
    charFormatMask.SetAttrib(EAttFontHeight);
    TCharFormat charFormat(MyeFontSpec.iTypeface.iName, MyeFontSpec.iHeight);
    charFormat.iFontPresentation.iTextColor=KRgbBlack;
    charFormat.iFontSpec.iHeight=100;
    // apply formatting    
    CRichText* text = iLogComponent->RichText();
    text->ApplyCharFormatL(charFormat,charFormatMask,0,1);
    #endif
    
    SetRect(aRect);
    _LIT(KInfo, "Audio streaming example");
    ShowMessageL(KInfo);
    
    // activate window
    ActivateL();    

    // SetRect() is called again after ActivateL() to that ScrollBarFrame widht
    // can be readed in SizeChanged()
    SetRect(aRect);
    }

// ----------------------------------------------------------------------------
// CAudioStreamView::CAudioStreamView()
//
// constructor
// ----------------------------------------------------------------------------
CAudioStreamView::CAudioStreamView()
    {
    }

// ----------------------------------------------------------------------------
// CAudioStreamView::~CAudioStreamView()
//
// destructor
// ----------------------------------------------------------------------------
CAudioStreamView::~CAudioStreamView()
    {
    delete iLogComponent;
    }

// ----------------------------------------------------------------------------
// CAudioStreamView::SizeChanged()
//
// called by framework when the view size is changed
// ----------------------------------------------------------------------------
void CAudioStreamView::SizeChanged()
    {
    TRect rect = Rect();
    
    TInt scrollbarWidth = iLogComponent->ScrollBarFrame()->ScrollBarBreadth(CEikScrollBar::EVertical);
    // If scrollbars are not drawn yet, the width remains zero. In this
    // case, an intentionally magical number is returned.
    if (scrollbarWidth == 0)
        {
        scrollbarWidth = 8;
        }
    iLogComponent->SetExtent(TPoint(0, 0),
            TSize(rect.Width() - scrollbarWidth, rect.Height()));
    DrawNow();
    }

// ---------------------------------------------------------
// CAudioStreamView::HandleResourceChange()
// Called by framework when resource is changed.
// ---------------------------------------------------------
//
void CAudioStreamView::HandleResourceChange(TInt aType)
    {
    CCoeControl::HandleResourceChange(aType);
    if ( aType==KEikDynamicLayoutVariantSwitch )
        {
        TRect rect;
        AknLayoutUtils::LayoutMetricsRect(AknLayoutUtils::EMainPane, rect);
        SetRect(rect);
        }
    }


// ----------------------------------------------------------------------------
// CAudioStreamView::CountComponentControls() const
// ----------------------------------------------------------------------------
TInt CAudioStreamView::CountComponentControls() const
    {
    return 1;   // return number of controls inside this container
    }

// ----------------------------------------------------------------------------
// CAudioStreamView::ComponentControl(TInt aIndex) const
// ----------------------------------------------------------------------------
CCoeControl* CAudioStreamView::ComponentControl(TInt aIndex) const
    {
    switch ( aIndex )
        {
        case 0:
            return iLogComponent;
        default:
            return NULL;
        }
    }


// ----------------------------------------------------------------------------
// CAudioStreamView::Draw(const TRect& aRect) const
// ----------------------------------------------------------------------------
void CAudioStreamView::Draw( const TRect& aRect ) const
    {
    CWindowGc& gc = SystemGc();   
    gc.SetPenStyle( CGraphicsContext::ENullPen );
    gc.SetBrushColor( KRgbWhite );
    gc.SetBrushStyle( CGraphicsContext::ESolidBrush );
    gc.DrawRect( aRect );
    }

// ----------------------------------------------------------------------------
// CAudioStreamView::ShowMessageL(const TDesC& aMsg)
//
// displays status messages to user
// ----------------------------------------------------------------------------
void CAudioStreamView::ShowMessageL(const TDesC& aMsg)
    {
    CPlainText* text = iLogComponent->Text();
    text->InsertL(text->DocumentLength(), aMsg);
    text->InsertL(text->DocumentLength(), CEditableText::ELineBreak);
    iLogComponent->HandleTextChangedL();
    // Keep displaying the bottom of the screen
    iLogComponent->MoveCursorL(TCursorPosition::EFPageDown, EFalse);
    DrawNow();
    }


// End of File  


