// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description: A dummy disk status system wide property is provided to demonstrate
// the System State Manager framework.
//

/**
Contains the implementation of a System Wide Property Policy file.

It consists of two parts:
- TransitionAllowed implementation, that defines the transitions that are supported by the current 
policy file.
- Command list preparation.

For this policy file (ssm.swp.policy.dummy.diskstatus.dll), the command list preparation is 
done using a resource file.

Note: If you are looking to construct the command list without using a resource file,
use the Command factory (SsmCommandFactory) class.

This example starts taking a back up of all the contacts, messages and other files when the disk status is reported
low. A dummy backup application is invoked by the disk status policy when the available memory of disk is low.
*/


#include <ssm/ssmswppolicy.h>
#include <ssm/ssmcommandlist.h>
#include <ssm/ssmcommandlistresourcereader.h>
#include "dummy_diskstatus_swppolicy.h"
#include "dummy_diskstatus_swppolicy.hrh"

/**
 Dummy Disk Status swp policy resource file path.
*/ 
_LIT(KCommandListPath, "z:\\private\\2000d75b\\ssmaeg\\swp_dummy_diskstatus\\");

/**
Panic used by Dummy Disk Status policy if the resource reader is invalid.
*/
_LIT(KExamplePolicyPanic,"DummyDiskStatus");

/**
Creates an instance of MSsmSwpPolicy-based class.

@return A pointer to an instance of MSsmSwpPolicy
*/
EXPORT_C MSsmSwpPolicy* CSsmDummyDiskStatusSwpPolicy::NewL()
	{
	CSsmDummyDiskStatusSwpPolicy* self = new (ELeave) CSsmDummyDiskStatusSwpPolicy;
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
Default Constructor.
*/
CSsmDummyDiskStatusSwpPolicy::CSsmDummyDiskStatusSwpPolicy()
	{
	}

/**
Destructor.
*/
CSsmDummyDiskStatusSwpPolicy::~CSsmDummyDiskStatusSwpPolicy()
	{
	iFs.Close();
	delete iCommandListResourceReader;
	}

/**
Instantiates all the objects required.
*/
void CSsmDummyDiskStatusSwpPolicy::ConstructL()
	{
	User::LeaveIfError(iFs.Connect());
	// create resource reader
	iCommandListResourceReader = CSsmCommandListResourceReader::NewL(iFs, KCommandListPath(), *this);
	}

/**
Initializes the objects.
*/
void CSsmDummyDiskStatusSwpPolicy::Initialize(TRequestStatus& aStatus)
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	// initialise command list resource reader.
	iCommandListResourceReader->Initialise(aStatus);
	}

/**
Cancels the Initialize request.
*/
void CSsmDummyDiskStatusSwpPolicy::InitializeCancel()
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	iCommandListResourceReader->InitialiseCancel();
	}

/** 
Deletes all resources and frees itself.

@see MSsmSwpPolicy::Release
*/
void CSsmDummyDiskStatusSwpPolicy::Release()
	{
	delete this;
	}

/** 
Determines if an incoming transition request should be accepted or rejected.
For this Swp Policy, transitions are allowed for all values.
@param aSwp Contains the value and property of the Swp.
@param aMessage Message sent by SSM server.
@return one of the TResponse value
@see MSsmSwpPolicy::TransitionAllowed
*/
MSsmSwpPolicy::TResponse CSsmDummyDiskStatusSwpPolicy::TransitionAllowed(const TSsmSwp& /*aSwp*/, const RMessagePtr2& /*aMessage*/)
	{
	TResponse response = EAllowed;
	return response;
	}

/** 
Creates the command list associated with a Swp transition.
@param aSwp Contains the Value and Key of a Swp.
@param aStatus Status associated with the request.
@see MSsmSwpPolicy::PrepareCommandList
*/ 
void CSsmDummyDiskStatusSwpPolicy::PrepareCommandList(const TSsmSwp& aSwp, TRequestStatus& aStatus)
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	// Build the command list for this value from a resource 
	iCommandListResourceReader->PrepareCommandList(aSwp.Value(), aSwp, aStatus);
	}

/**
Cancels an asynchronous PrepareCommandList operation.
@see MSsmSwpPolicy::PrepareCommandListCancel
*/
void CSsmDummyDiskStatusSwpPolicy::PrepareCommandListCancel()
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	iCommandListResourceReader->PrepareCommandListCancel();
	}

/**
Returns the command list once the PrepareCommandList has completed.
Ownership of the returned command list is transferred to the caller.
@see MSsmSwpPolicy::CommandList
@return The command list created during the preceding PrepareCommandList step
*/
CSsmCommandList* CSsmDummyDiskStatusSwpPolicy::CommandList()
	{	
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	return iCommandListResourceReader->GetCommandList();
	}

/**
Handles the return value after the command list is executed.
@param aSwp Contains the value and property of the Swp.
@param aError Contains the return value of the previous executed command.
@param aSeverity Contains the severity of the previous executed command.
@param aStatus to complete when the operation has finished
@see MSsmSwpPolicy::HandleCleReturnValue
*/
void CSsmDummyDiskStatusSwpPolicy::HandleCleReturnValue(const TSsmSwp& /*aSwp*/, TInt /*aError*/, TInt /*aSeverity*/, TRequestStatus& aStatus)
	{
	/*
	This function is going to handle the return value. Even an reported error is treated as 
	a successful API request. The request always needs to be completed with KErrNone.
	*/
	TRequestStatus* rs = &aStatus;
	User::RequestComplete(rs, KErrNone);
	}

/**
Cancels the HandleCleReturnValue request
@see MSsmSwpPolicy::HandleCleReturnValueCancel
*/
void CSsmDummyDiskStatusSwpPolicy::HandleCleReturnValueCancel()
	{
	// Nothing to do as HandleCleReturnValue always completes immediately
	}

/**
Callback used by CSsmCommandListResourceReader when a decision needs to be made
if to include a command in a command list or not.

@param aResourceFile Instance of CResourceFile
@param aResourceId Resource id of SSM_SYMBIAN_CONDITIONAL_INFORMATION struct for the command
@return ETrue if the command needs to be included in the command list, else EFalse.

@see MSsmConditionalCallback::ConditionalCommandAllowedL
*/
TBool CSsmDummyDiskStatusSwpPolicy::ConditionalCommandAllowedL(CResourceFile& /*aResourceFile*/, TInt /*aResourceId*/)
	{
	return EFalse; // command list does not contain commands that have conditions, so return EFalse
	}
