//
// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description: A dummy back up state policy is provided to demonstrate the utility
// of System State Manager framework.
//

/**
Contains the implementation of a Dummy Backup State Policy file.

It demonstrates:
- Checking for the state transitions that are supported.
- Construction of a command list to execute when there is a state change.

For this policy file (i.e. ssm.state.policy.0004.dll), the command list preparation is 
done using a resource file.

Note: If you want to construct the command list without using a resource file, 
use the Command factory (SsmCommandFactory) class.
*/

#include <ssm/ssmcommandlist.h>
#include <ssm/ssmcommandlistresourcereader.h>
#include <ssm/ssmsubstates.hrh>
#include "dummy_backup_statepolicy.h"

/**
 Dummy Back-up state policy resource file path. 
*/ 
_LIT(KCommandListPath, "z:\\private\\2000d75b\\ssmaeg\\state_dummy_backup\\");

/**
Panic used by Dummy Back-up policy plug-in if the resource reader is invalid.
*/
_LIT(KExamplePolicyPanic,"Dummy Back up State Policy");

/**
Creates an instance of MSsmStatePolicy-based class.

@return A pointer to an instance of MSsmStatePolicy
*/
EXPORT_C MSsmStatePolicy* CSsmDummyBackUpStatePolicy::NewL()
	{
	CSsmDummyBackUpStatePolicy* self = new (ELeave) CSsmDummyBackUpStatePolicy;
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
Default constructor.
*/
CSsmDummyBackUpStatePolicy::CSsmDummyBackUpStatePolicy()
	{
	}

/**
Destructor.
*/
CSsmDummyBackUpStatePolicy::~CSsmDummyBackUpStatePolicy()
	{
	delete iCommandListResourceReader;
	iFs.Close();
	iCurrentlySupportedTransitions.Close();
	}

/**
Makes a RFs connection and maintains a list of supported transition states from Backup state.

@leave One of the error value returned by RFs::Connect() or RArray::AppendL()
*/
void CSsmDummyBackUpStatePolicy::ConstructL()
	{
	User::LeaveIfError(iFs.Connect());
	
	// Add supported transitions from Backup 'ESsmBackupSubState'
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmShutdown, KSsmAnySubState));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmFail, KSsmAnySubState));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmShutdown, ESsmShutdownSubStateCritical));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmNormal, KSsmAnySubState));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmRestore, ESsmRestoreSubState));
	// create resource reader
	iCommandListResourceReader = CSsmCommandListResourceReader::NewL(iFs, KCommandListPath(), *this);
	}

/**
Initializes a command list resource reader.

@param aStatus to complete when the initialization operation has finished
@panic EInvalidResourceReader if the command list resource reader is invalid

@see MSsmStatePolicy::Initialize
*/
void CSsmDummyBackUpStatePolicy::Initialize(TRequestStatus& aStatus)
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	// initialise command list resource reader.
	iCommandListResourceReader->Initialise(aStatus);
	}

/**
Cancels an asynchronous Initialize operation.

@see MSsmStatePolicy::InitializeCancel

A time delay can be put in the above initialize function to test cancellation.
*/
void CSsmDummyBackUpStatePolicy::InitializeCancel()
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	iCommandListResourceReader->InitialiseCancel();
	}

/** 
Deletes all resources and frees itself.

@see MSsmStatePolicy::Release
*/
void CSsmDummyBackUpStatePolicy::Release()
	{
	delete this;
	}

/** 
Determines whether to allow the requested state transition or not.

A transition to a ESsmFail, ESsmNormal, ESsmRestore or ESsmShutdown state is allowed 
from the ESsmBackup state. All other requests are rejected.

Clients calling this API must possess the ECapabilityPowerMgmt capability, 
or the API will return ENotAllowed.

@param aRequest Contains information about the new requested state 
@param aCurrent Contains NULL or the first accepted but not yet completed transition request
@param aQueued Contains NULL or a second accepted but not yet started transition request
@param aMessage Client message, used to check if the client has PowerMgmt capability 

@return one of the TResponse value
@see MSsmStatePolicy::TransitionAllowed
@see MSsmStatePolicy::TResponse
*/
MSsmStatePolicy::TResponse CSsmDummyBackUpStatePolicy::TransitionAllowed(const TSsmStateTransition& aRequest, TSsmStateTransition const* aCurrent, TSsmStateTransition const* aQueued, const RMessagePtr2& aMessage)
	{
	TResponse response = ENotAllowed;
	if (!aMessage.HasCapability(ECapabilityPowerMgmt))
		{
		return response;
		}

	//Check if the requested transition is supported from current state
	if (TransitionSupported(aRequest.State()))
		{
		if((NULL == aCurrent) && (NULL == aQueued))
			{
			// SsmServer is idle
			response = EDefinitelyAllowed;
			}
		else if(aRequest.State().MainState() == ESsmFail || aRequest.State().MainState() == ESsmShutdown)
			{
			// Going into failed state will override anything currently ongoing or queued
			response = EReplaceCurrentClearQueue;
			}
		}
	return response;
	}
/** 
Creates the command list associated with a substate transition.

@param aState Contains the state and substate that identifies the command list to create
@param aReason Contains the reason as given by the request
@param aStatus to complete when the operation has finished
@panic EInvalidResourceReader if the command list resource reader is invalid
@see MSsmStatePolicy::PrepareCommandList
*/ 
void CSsmDummyBackUpStatePolicy::PrepareCommandList(TSsmState aState, TInt /*aReason*/, TRequestStatus& aStatus)
	{	
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	//Let's start from the beginning if no specific sub state is selected
	const TUint16 substate = aState.SubState();
	//Build the command list from a resource for this substate
	iCommandListResourceReader->PrepareCommandList(substate, aState, aStatus);
	} 

/**
Cancels an asynchronous PrepareCommandList operation.

@see MSsmStatePolicy::PrepareCommandListCancel
*/
void CSsmDummyBackUpStatePolicy::PrepareCommandListCancel()
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	iCommandListResourceReader->PrepareCommandListCancel();
	}

/**
Return the command list once the PrepareCommandList has completed.
Ownership of the returned command list is transferred to the caller.
@panic EInvalidResourceReader if the command list resource reader is invalid
@return The command list created during the preceding PrepareCommandList step
*/
CSsmCommandList* CSsmDummyBackUpStatePolicy::CommandList()
	{
	__ASSERT_ALWAYS(iCommandListResourceReader, User::Panic(KExamplePolicyPanic, EInvalidResourceReader));
	return iCommandListResourceReader->GetCommandList();
	}

/**
Determines the next substate transition.
@param aCurrentTransition Contains the last executed state
@param aReason Contains the reason as given by the request
@param aError Contains the completion code from the last executed sub-state transition
@param aSeverity Contains the severity of the failed command in case the substate transition ended with an error
@param aNextState The next System State to head for, if there is one
@return ETrue if aNextState contains another System State to head for, or 
		EFalse if there are no further transitions to do.
@see MSsmStatePolicy::GetNextState
*/
TBool CSsmDummyBackUpStatePolicy::GetNextState(TSsmState /*aCurrentTransition*/, TInt /*aReason*/, TInt /*aError*/, TInt /*aSeverity*/, TSsmState& /*aNextState*/)
	{
	return EFalse;
	} 

/**
Callback used by CSsmCommandListResourceReader when a decision needs to be made
on whether to include a command in a command list or not.

@param aResourceFile Instance of CResourceFile
@param aResourceId Resource id of SSM_SYMBIAN_CONDITIONAL_INFORMATION struct for the command
@return ETrue in case the command needs to be included in a command list, else EFalse.

@see MSsmConditionalCallback::ConditionalCommandAllowedL
*/
TBool CSsmDummyBackUpStatePolicy::ConditionalCommandAllowedL(CResourceFile& /*aResourceFile*/, TInt /*aResourceId*/)
	{
	return EFalse; //command list doesn't contains commands which have conditions and hence returning EFalse 
	}

/*
Helper function to check whether a requested transition is supported or not.
@param aRequestedState Requested transition
@return ETrue if transition is supported
		EFalse if transition is not supported
*/
TBool CSsmDummyBackUpStatePolicy::TransitionSupported(const TSsmState& aRequestedState) const
	{
	return (iCurrentlySupportedTransitions.Find(aRequestedState) > KErrNotFound);
	}
