//
// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description: A dummy restore state policy is provided with the system state manager
// example code which demonstrates the various utilities of SSM framework.
//

/**
Contains the implementation of a System State Policy file.

It demonstrates:
- Addition of the new state transitions that are supported.
- Construction of a command list to be executed when there is a state change.

For this policy file (i.e ssm.state.policy.0005.dll), command list preparation is 
done without using a command list factory.

Note: If you want to construct the command list using a resource file,
use the CSsmCommandListResourceReader class.
*/

#include <ssm/ssmcommandlist.h>
#include <ssm/ssmpublishsystemstateinfo.h>
#include <ssm/ssmcommandfactory.h>
#include <ssm/ssmstartupproperties.h>
#include "dummy_restore_statepolicy.h"


/**
Used to create an instance of a MSsmStatePolicy-based class.

@return A pointer to an instance of MSsmStatePolicy
*/
EXPORT_C MSsmStatePolicy* CSsmDummyRestoreStatePolicy::NewL()
	{
	CSsmDummyRestoreStatePolicy* self = new (ELeave) CSsmDummyRestoreStatePolicy;
	CleanupStack::PushL(self);
	CleanupStack::Pop(self);
	return self;
	}
 
/**
Default constructor.
*/
CSsmDummyRestoreStatePolicy::CSsmDummyRestoreStatePolicy()
	{
	}

/**
Destructor.
*/
CSsmDummyRestoreStatePolicy::~CSsmDummyRestoreStatePolicy()
	{
 	delete iCommandList;
	}

/**
Initialize the objects.
*/
void CSsmDummyRestoreStatePolicy::Initialize(TRequestStatus& aStatus)
	{
	TRequestStatus* rs = &aStatus;
	User::RequestComplete(rs, KErrNone);
	}

/**
Cancels the Initialize request.
*/
void CSsmDummyRestoreStatePolicy::InitializeCancel()
	{
	}

/** 
Deletes all resources and frees itself.

@see MSsmStatePolicy::Release
*/
void CSsmDummyRestoreStatePolicy::Release()
	{
	delete this;
	}

/** 
Determines whether to allow the requested state transition or not.

Transition to all the states is allowed, without any verification.

@param aRequest Contains information about the new request
@param aCurrent Contains NULL, or the first accepted but not yet completed transition request
@param aQueued Contains NULL, or a second accepted but not yet started transition request
@param aMessage aMessage client message.
@return one of the TResponse value
@see MSsmStatePolicy::TransitionAllowed
@see MSsmStatePolicy::TResponse
*/
MSsmStatePolicy::TResponse CSsmDummyRestoreStatePolicy::TransitionAllowed(const TSsmStateTransition& /*aRequest*/, TSsmStateTransition const* /*aCurrent*/, TSsmStateTransition const* /*aQueued*/, const RMessagePtr2& /*aMessage*/)
	{
	return EDefinitelyAllowed;
	}

/** 
Creates the command list associated with a substate transition.

@param aState Contains the state and substate that identifies the command list to create
@param aReason Contains the reason as given by the request
@param aStatus to complete when the operation has finished
@see MSsmStatePolicy::PrepareCommandList
*/ 
void CSsmDummyRestoreStatePolicy::PrepareCommandList(TSsmState aState, TInt /*aReason*/, TRequestStatus& aStatus)
	{	
	TRAPD(err, doPrepareCommandListL(aState));
	TRequestStatus* rs = &aStatus;
	User::RequestComplete(rs, err);
	} 

void CSsmDummyRestoreStatePolicy::doPrepareCommandListL(TSsmState aState)
	{
	// Reset the command list
	delete iCommandList;
	iCommandList = NULL;
	iCommandList = CSsmCommandList::NewL();
	_LIT(KProcessName, "dummybackup.exe");
	_LIT(KNullDesc, "");
	// Start a process from your command list.
	CSsmStartupProperties* startupProperties = CSsmStartupProperties::NewL(KProcessName,KNullDesc);
	CleanupStack::PushL(startupProperties);
	TSsmPublishSystemStateInfo info;
	TInt16 retries =0;//Number of retries for this System State is given as '0'
	info.Set(aState, ETraverseParentsFirst, retries); 
	// Create a command to publish Restore System State.
	MSsmCommand* cmd1 = SsmCommandFactory::ConstructPublishSystemStateCommandLC(ECmdCriticalSeverity, ESsmWaitForSignal, info);
	// Construct a command which will start a process
	MSsmCommand* cmd2 = SsmCommandFactory::ConstructStartProcessCommandLC(ECmdCriticalSeverity,startupProperties);
	iCommandList->AppendL(cmd1);
	iCommandList->AppendL(cmd2);
	CleanupStack::Pop(2); // Pops the above cmd1 and cmd2 from the stack.
	CleanupStack::PopAndDestroy(startupProperties); 
	}

/**
Cancels an asynchronous PrepareCommandList operation.

@see MSsmStatePolicy::PrepareCommandListCancel
*/
void CSsmDummyRestoreStatePolicy::PrepareCommandListCancel()
	{
 	delete iCommandList;
 	iCommandList = NULL;
	}

/**
Returns the command list once the PrepareCommandList has completed.
Ownership of the returned command list is transferred to the caller.
@return The command list created during the preceding PrepareCommandList step
*/
CSsmCommandList* CSsmDummyRestoreStatePolicy::CommandList()
	{
	CSsmCommandList* list = iCommandList;
	iCommandList = NULL;
	return list;
	}

/**
Determines the next substate transition.
@param aCurrentTransition Contains the last executed state
@param aReason Contains the reason as given by the request
@param aError Contains the completion code from the last executed sub-state transition
@param aSeverity Contains the severity of the failed command in case the sub-state transition ended with an error
@param aNextState The next System State to head for, if there is one
@return 	ETrue if aNextState contains another System State to head for, or 
		    EFalse if there is no further transitions to do.
@see MSsmStatePolicy::GetNextState
*/
TBool CSsmDummyRestoreStatePolicy::GetNextState(TSsmState /*aCurrentTransition*/, TInt /*aReason*/, TInt /*aError*/, TInt /*aSeverity*/, TSsmState& /*aNextState*/)
	{
	return EFalse;
	} 
