// gsastatepolicynormal.cpp
//
// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Extends the GSA (Generic Startup Architecture) normal policy to  
// to support the dummy backup and restore states added as part of this example. 
// The default Normal state policy supports only the states start up, shut down, and fail.
// The example uses this extended normal state, which replaces the existing normal policy.
//  
// This example Normal State policy creates and registers a SwP (ssm.swp.policy.dummy.diskstatus.dll)
// with key KExampleSwPKey and KExampleSwPValue.

#include <barsread2.h>
#include <e32debug.h>
#include <ssm/ssmcmd.hrh>
#include <ssm/ssmsubstates.hrh>

#include <ssm/ssmstatetransition.h>
#include <ssm/ssmcommandlistresourcereader.h>

#include "gsastatepolicynormal.h"

/**
Panic used by Normal policy plug-in when resource reader is invalid.
*/
_LIT(KPanicGsaNormalState, "NormalStatePolicy");

/**
Create an instance of a MSsmStatePolicy-based class.

@return A pointer to an instance of MSsmStatePolicy
*/
EXPORT_C MSsmStatePolicy* CEgGsaStatePolicyNormal::NewL()
	{
	CEgGsaStatePolicyNormal* self = new (ELeave) CEgGsaStatePolicyNormal;
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}
 
 /**
Default constructor.
*/
CEgGsaStatePolicyNormal::CEgGsaStatePolicyNormal()
	{
	}

/**
Destructor.
*/
CEgGsaStatePolicyNormal::~CEgGsaStatePolicyNormal()
	{
	delete iCommandListResourceReader;
	iCurrentlySupportedTransitions.Close();
	iFs.Close();	
	}

/**
Makes a RFs connection and maintains a list of the supported transition states from the normal state.

@leave One of the error values returned by RFs::Connect() or RArray::AppendL()
*/
void CEgGsaStatePolicyNormal::ConstructL()
	{
	User::LeaveIfError(iFs.Connect());
	
	// Add supported transitions from Normal 'ESsmNormal'
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmShutdown, KSsmAnySubState));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmFail, KSsmAnySubState));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmShutdown, ESsmShutdownSubStateCritical));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmBackup, ESsmBackupSubState));
	iCurrentlySupportedTransitions.AppendL(TSsmState(ESsmRestore, ESsmRestoreSubState));
	
	/*
    Normal state policy resource file path ("z:\private\<SID of SSM>\normal\").
    This resource file is already being generated by the production code, hence its not re-generated.
	*/
	_LIT(KCommandListPath, "z:\\private\\2000D75B\\normal\\");
	
	// Create a resource reader
	iCommandListResourceReader = CSsmCommandListResourceReader::NewL(iFs, KCommandListPath(), *this);
	}

/**
Initializes the command list resource reader.

@param aStatus to complete when the initialization operation has finished
@panic EInvalidResourceReader if the command list resource reader is invalid

@see MSsmStatePolicy::Initialize
*/
void CEgGsaStatePolicyNormal::Initialize(TRequestStatus& aStatus)
	{
	__ASSERT_DEBUG(iCommandListResourceReader, User::Panic(KPanicGsaNormalState, EInvalidResourceReader));

	// initialise command list resource reader.
	iCommandListResourceReader->Initialise(aStatus);
	}

/**
Cancels an asynchronous Initialize operation.

@see MSsmStatePolicy::InitializeCancel
*/
void CEgGsaStatePolicyNormal::InitializeCancel()
	{
	__ASSERT_DEBUG(iCommandListResourceReader, User::Panic(KPanicGsaNormalState, EInvalidResourceReader));
	iCommandListResourceReader->InitialiseCancel();
	}

/** 
Deletes all resources and frees itself.

@see MSsmStatePolicy::Release
*/
void CEgGsaStatePolicyNormal::Release()
	{
	delete this;
	}

/** 
Determines if an incoming normal state transition request should be accepted or rejected.
Transition to ESsmFail,ESsmShutdown,ESsmBackup and ESsmRestore state is allowed from ESsmNormal state. 
All other requests are rejected.

Clients calling this API should possess the ECapabilityPowerMgmt capability, 
or the API will return ENotAllowed.

@param aRequest Contains information about the new request
@param aCurrent Contains NULL, or the first accepted but not yet completed transition request
@param aQueued Contains NULL, or a second accepted but not yet started transition request
@param aMessage Message sent by SSM server, used to check if the client has 'ECapabilityPowerMgmt'

@return A TResponse value
@see MSsmStatePolicy::TransitionAllowed
@see MSsmStatePolicy::TResponse
*/
MSsmStatePolicy::TResponse CEgGsaStatePolicyNormal::TransitionAllowed(const TSsmStateTransition& aRequest, TSsmStateTransition const* aCurrent, 
															TSsmStateTransition const* aQueued, const RMessagePtr2& aMessage)
	{
	TResponse response = ENotAllowed;
	if (!aMessage.HasCapability(ECapabilityPowerMgmt))
		{
		return response;
		}

	// Check if the requested transition is supported from the current state
	if (TransitionSupported(aRequest.State()))
		{
		if((NULL == aCurrent) && (NULL == aQueued))
			{
			// SsmServer is idle
			response = EDefinitelyAllowed;
			}
		else if(aRequest.State().MainState() == ESsmFail)
			{
			// Going into failed state will override anything currently ongoing or queued
			response = EReplaceCurrentClearQueue;
			}
		}
	TSsmStateName name = aRequest.State().Name();
	return response;		
	}

/** 
Create the command list for a state transition from any other state(like Backup,Restore etc) to Normal state.
A transition from any other state to Normal is allowed only if the Normal state is listed in their allowed transitions list.
Once the verification of allowed transition is successful,then the command list preparation is followed.

@param aState Contains the state and substate that identifies the command list to create
@param aReason Contains the reason given by the request
@param aStatus to complete when the operation has finished
@panic EInvalidResourceReader if the command list resource reader is invalid
@see MSsmStatePolicy::PrepareCommandList
*/ 
void CEgGsaStatePolicyNormal::PrepareCommandList(TSsmState aState, TInt /*aReason*/, TRequestStatus& aStatus)
	{
	__ASSERT_DEBUG( iCommandListResourceReader , User::Panic(KPanicGsaNormalState, EInvalidResourceReader));

	// Start from the beginning if no specific substate is selected
	const TUint16 substate = aState.SubState();
	const TSsmState state(aState.MainState(), (substate==KSsmAnySubState) ? ESsmNormalSubState : substate);
	const TInt commandListId = state.SubState();

	// Build the command list from the resource
	iCommandListResourceReader->PrepareCommandList(commandListId, state, aStatus);
	} 

/**
Cancels an asynchronous PrepareCommandList operation.

@see MSsmStatePolicy::PrepareCommandListCancel
*/
void CEgGsaStatePolicyNormal::PrepareCommandListCancel()
	{
	iCommandListResourceReader->PrepareCommandListCancel();
	}

/**
Returns the command list once the PrepareCommandList has completed.
Ownership of the returned command list is transferred to the caller.
@panic EInvalidResourceReader if the command list resource reader is invalid
@return The command list created during the preceding PrepareCommandList step
*/
CSsmCommandList* CEgGsaStatePolicyNormal::CommandList()
	{
	__ASSERT_DEBUG( iCommandListResourceReader , User::Panic(KPanicGsaNormalState, EInvalidResourceReader));

	return iCommandListResourceReader->GetCommandList();
	}

/**
Determines the next substate transition.
@param aCurrentTransition Contains the last executed state
@param aReason Contains the reason as given by the request
@param aError Contains the completion code from the last executed sub-state transition
@param aSeverity Contains the severity of the failed command in case the sub-state transition ended with an error
@param aNextState The next System State to head for, if there is one
@panic EInvalidNormalState if the current state is not Normal
@return 	ETrue if aNextState contains another System State to head for, or 
		EFalse if there is no further transitions to do.
@see MSsmStatePolicy::GetNextState
*/
TBool CEgGsaStatePolicyNormal::GetNextState(TSsmState aCurrentTransition, TInt /*aReason*/, TInt aError, TInt /*aSeverity*/, TSsmState& aNextState)
	{
	__ASSERT_ALWAYS(aCurrentTransition.MainState() == ESsmNormal, User::Panic(KPanicGsaNormalState, EInvalidNormalState));
	if (KErrNone != aError)
		{
		aNextState = TSsmState(ESsmFail, KSsmAnySubState);
		TSsmStateName name = aNextState.Name();
		return ETrue;
		}

	return EFalse;	// no more substates to execute
	} 

/**
Callback used by CSsmCommandListResourceReader when a decision needs to be made
to include a command in a command list or not.

@param aResourceFile Instance of CResourceFile
@param aResourceId Resource id of the SSM_SYMBIAN_CONDITIONAL_INFORMATION struct for the command
@return ETrue if the command needs to be included in command list, else EFalse.

@see MSsmConditionalCallback::ConditionalCommandAllowedL
*/
TBool CEgGsaStatePolicyNormal::ConditionalCommandAllowedL(CResourceFile& /*aResourceFile*/, TInt /*aResourceId*/)
	{
	// no commands use 'conditional_information' in Normal state command list.
	User::Panic(KPanicGsaNormalState, EConditionalInfoNotImplemented);
	return EFalse;	// avoid a compiler warning.
	}

/*
Helper function to check whether requested transition is supported or not.
@param aRequestedState Requested transition
@return 	ETrue if transition is supported
		EFalse if transition is not supported
*/
TBool CEgGsaStatePolicyNormal::TransitionSupported(const TSsmState& aRequestedState) const
	{
	return (iCurrentlySupportedTransitions.Find(aRequestedState) > KErrNotFound);
	}
