//
// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description: This example is used to demonstrate the utilities of System state 
// Manager framework.
//	

/**
Implements the E32Main() function that is called by the operating system to start the program.

It presents the main menu to the user, and calls various System State Manager Utilities,
according to the user's choices.

An example SwP "ssm.swp.policy.dummy.diskstatus.dll" is created and registered using the 
Normal State Policy provided by the example.
*/

#include "ssmanager.h"
#include "dummy_diskstatus_swppolicy.hrh"
#include <ssm/ssmsubstates.hrh>
#include <ssm/ssmstate.h>
#include <ssm/ssmstateawaresession.h>
#include <ssm/ssmdomaindefs.h>

_LIT(KTxtExampleCode,"System State Manager Example");
_LIT(KTxtError,"error is %d\n");
_LIT(KTxtFormat,"*****************************\n");
_LIT(KTxtFormat1,".............\n");

_LIT(KTxtSwpInfo,"\nKey of the SwP is %d \n \nValue of the SwP is %d \n\n");
_LIT(KTxtSasResult,"\nState aware session connected with [%d]");

_LIT(KTxtState,"\nValue of new state :");
_LIT(KTxtReqNotificationChgofState,"\nRequest Notification for any change of state\n");
_LIT(KTxtStateValues,"\n Main state: %04x \n SubState: %04x \n");

_LIT(KTxtStateChangeResult,"\nState transition completed with : %d\n");
_LIT(KTxtSwpChangeResult,"\nSwP transition completed with : %d\n");

const TUint KFiveSecDelay  = 500000;

static TInt StopScheduler(TAny* /*aAny*/)
 	{
 	CActiveScheduler::Stop();
 	return KErrNone;
 	}
	
CSysStateManagerExample* CSysStateManagerExample::NewLC()
	{
	CSysStateManagerExample* self=new(ELeave)CSysStateManagerExample();
	CleanupStack::PushL(self);
	return self;
	}

CSysStateManagerExample::CSysStateManagerExample()
	{
	
	}

/**
Destructor.
*/
CSysStateManagerExample::~CSysStateManagerExample()
	{
	iSsmStateManager.Close();
	delete iConsole; // delete iConsole.
	}

/**
This API is called to notify the program about a change in a SwP value.
*/
void CSysStateManagerExample::SwpChanged(TSsmSwp /*aSwp*/)
	{
	_LIT(KTxtUserNotifiedSwPChange,"Notification received of a change in the SwP value.\n");
	iConsole->Printf(KTxtUserNotifiedSwPChange);
	}

/**
Makes a System Wide Property change request. 
*/
void CSysStateManagerExample::RequestSwpChange()
	{
	/**
	Example SwP is created and registered using the Normal state policy
	resource file (gsanormalcmdlist.rss) provided with the example.
	*/
	TSsmSwp swp(EExampleSwPKey, ESwpValueJustPublishSwP); 
	// Request a Swp change
	TRequestStatus status;
	iSsmStateManager.RequestSwpChange(swp, status);
	User::WaitForRequest(status);
	iConsole->Printf(KTxtSwpChangeResult,status.Int());
	iConsole->Printf(KTxtSwpInfo,swp.Key(),swp.Value());
	}
	
/**
Makes a System State Change request.
*/
void CSysStateManagerExample::RequestSsChange()
	{
	_LIT(KTxtNormalToRestore,"\nRequesting a System state change from Normal to Restore\n");
	TInt reason =0; //Various reasons for state transition can be stated, the value is taken as '0'
	TSsmStateTransition state(ESsmRestore, ESsmRestoreSubState, reason); 
	iConsole->Printf(KTxtNormalToRestore);
    TRequestStatus status;
	iSsmStateManager.RequestStateTransition(state,status);
	User::WaitForRequest(status);
	User::After(KFiveSecDelay); //Wait for 0.5sec to allow transitions to fully complete
	iConsole->Printf(KTxtStateChangeResult,status.Int());
	//This session helps you to get the details of the current state of the device
	RSsmStateAwareSession stateAwareSession;
	/*
	'KSM2UiServicesDomain3' is the domain id for the third domain in the UI 
	Service level of the Startup Domain Hierarchy.
	*/
	TInt ret = stateAwareSession.Connect(KSM2UiServicesDomain3);
	if(ret!=KErrNone)
		{
		iConsole->Printf(KTxtSasResult,ret);
		}
	else
		{
		TSsmState current = stateAwareSession.State(); 
		TSsmStateName name = current.Name();
		iConsole->Printf(KTxtState);
		iConsole->Printf(name);
		iConsole->Printf(KTxtStateValues,current.MainState(),current.SubState());
		stateAwareSession.Close();
		}
	}

/**
Makes a System Wide Property change request, and subscribes to receive
notifications of when the change is made.
*/
 void CSysStateManagerExample::RequestSwpChangeWithNotificationL()
	{
	CActiveScheduler* sched = new(ELeave) CActiveScheduler;
	CleanupStack::PushL(sched);
	CActiveScheduler::Install(sched);	
	/**
	CSsmSystemWideProperty is used to monitor for Swp changes. It maintains a list of
	subscribers for the Swp. Each of the subscribers will be notified about the change.
	*/
	CSsmSystemWideProperty* cProp = CSsmSystemWideProperty::NewLC(EExampleSwPKey);
	
	// Subscribe to changes in dummy disk status Swp
	cProp->AddSubscriberL(*this);
	
	// Request Swp changes
	TSsmSwp swp(EExampleSwPKey,ESwpValueJustPublishSwP);
	TRequestStatus status;
	iSsmStateManager.RequestSwpChange(swp, status);
	User::WaitForRequest(status);
	iConsole->Printf(KTxtSwpChangeResult,status.Int());	
	iConsole->Printf(KTxtSwpInfo,swp.Key(),swp.Value());
	
	// Request Swp changes
	swp.Set(EExampleSwPKey,ESwpValueForBackup);
	iSsmStateManager.RequestSwpChange(swp, status);
	User::WaitForRequest(status);
	iConsole->Printf(KTxtSwpChangeResult,status.Int());	
	iConsole->Printf(KTxtSwpInfo,swp.Key(),swp.Value());
	CAsyncCallBack* stopper = new(ELeave) CAsyncCallBack(CActive::EPriorityIdle);
	CleanupStack::PushL(stopper);
	TCallBack stopSchedulerCallback(StopScheduler, this);
	stopper->Set(stopSchedulerCallback);
	
	// This callback will not run until the Swp change is finished, because it has idle priority. 
	stopper->CallBack();
	// Causes CSsmSystemWideProperty::RunL to run, as cProp is subscribing on 'Dummy Disk Status' changes
	sched->Start();
	CleanupStack::PopAndDestroy(3, sched); // Pop and Destroy sched, cprop and stopper.
	}

 /**
 Makes a System State change request, and subscribes to receive
 notifications of when the change is made.
 */
 void CSysStateManagerExample::RequestSsChangeWithNotification()
	{
	_LIT(KTxtStateChangeNotificationSuccess,"\nNotification of the state transition received\n");
	_LIT(KTxtStateChangeNotificationFailure,"\nNotification of the state transition failed\n");
	_LIT(KTxtNormalToBackup,"\nRequest a System State change from Normal to Back up\n");
	_LIT(KTxtAlreadyInThisState,"\nSystem state is already in Backup\n");
	
	// This session helps you to get the details of the current state of the device
	RSsmStateAwareSession stateAwareSession;
	
	/*
	'KSM2UiServicesDomain3' is the domain id for the third domain in the UI 
	Service level of the Startup Domain Hierarchy.
	*/
	TInt err = stateAwareSession.Connect(KSM2UiServicesDomain3);
	iConsole->Printf(KTxtSasResult,err);
	TSsmState current = stateAwareSession.State();
	TSsmStateName name = current.Name();
	
	if(current.MainState()!= ESsmBackup)
		{
		TSsmStateTransition state1(ESsmBackup,ESsmBackupSubState, 0);
		iConsole->Printf(KTxtNormalToBackup);
		TRequestStatus status1,status2;
		iConsole->Printf(KTxtReqNotificationChgofState);
		stateAwareSession.RequestStateNotification(status1);
		iSsmStateManager.RequestStateTransition(state1,status2);
		User::WaitForRequest(status2);
		User::After(KFiveSecDelay); // Wait for 0.5 sec to allow transitions to fully complete
		if(status2.Int() == KErrNone)
			{
			iConsole->Printf(KTxtStateChangeResult,status2.Int());
			current = stateAwareSession.State();
			name = current.Name();
			}
		else
			{
			iConsole->Printf(KTxtStateChangeResult,status2.Int());	
			}
		iConsole->Printf(KTxtState);
		iConsole->Printf(name);
		iConsole->Printf(KTxtStateValues,current.MainState(),current.SubState());
		User::WaitForRequest(status1); //status1 of the Notification request waits for status2 to complete
		if(status1.Int() == KErrNone)
			{
			iConsole->Printf(KTxtStateChangeNotificationSuccess);
			}
		else
			{
			iConsole->Printf(KTxtStateChangeNotificationFailure);	
			}
		}
	else
		{
		iConsole->Printf(KTxtAlreadyInThisState);
		}
	stateAwareSession.Close();
	}   

 void CSysStateManagerExample::SSManagerL()
	{
	_LIT(KTextPressAnyKey," [press any key]");
	_LIT(KTextPressAnyKeyToExit," [press any key to exit]");
	
	_LIT(KTxtSwp,"Swp means SystemWide Property.\nDisk status, Battery information are few examples of System wide properties.\n");
	_LIT(KTxtSs,"\n SS means System State.\n Ex:Backup,Restore, Start-up, Shutdown are few examples of System states.\n");
	_LIT(KTxtExampleDemo,"\n The example helps the user in understanding, how to define new System states and System wide properties.It also helps him in exercising the various features of SSM framework.\n");
	_LIT(KTxtSSM, "Exercise the features of System State Manager\n");			
	_LIT(KTxtConnect,"Connecting to System State Manager.........\n");
	_LIT(KTxtSuccess,"Connection is successful.........\n");
		
	_LIT(KTxtReqChangeInSwp,"1)	request a change to a Swp, without notification\n");
	_LIT(KTxtReqChangeInSs,"2) request a change to a system state, without notification\n");
	_LIT(KTxtNotifyChangeInSwp,"3) request a change to a Swp, with notification\n");
	_LIT(KTxtNotifyChangeInSs,"4) request a change to a system state, with notification\n");
	_LIT(KTxtOptionNotSupported,"this option is not supported\n");
		
	_LIT(KTxtPressESC,"Press ESC to stop using the System State Information utility\n");
	_LIT(KTxtContinue,"Choose any option 1 - 4 to continue\n");	

	iConsole= Console::NewL(KTxtExampleCode,TSize(KConsFullScreen,KConsFullScreen));
 	iConsole->Printf(KTxtFormat);
 	iConsole->Printf(KTxtSwp);
 	iConsole->Printf(KTxtSs);
 	iConsole->Printf(KTxtExampleDemo);
 	iConsole->Printf(KTxtFormat);
 	iConsole->Printf(KTxtSSM);			
 	iConsole->Printf(KTxtConnect);
 	iConsole->Printf(KTxtFormat1);
 	TInt connect = iSsmStateManager.Connect();	
 	if(connect==KErrNone)
 		{
 		iConsole->Printf(KTxtSuccess);
 		iConsole->Printf(KTxtFormat);
 		iConsole->Printf(KTextPressAnyKey);
 		iConsole->Getch(); // Get and ignore character.
 		iConsole->ClearScreen();
 		iConsole->Printf(KTxtReqChangeInSwp);
 		iConsole->Printf(KTxtReqChangeInSs);
 		iConsole->Printf(KTxtNotifyChangeInSwp);
 		iConsole->Printf(KTxtNotifyChangeInSs);
 		iConsole->Printf(KTxtContinue);
 		TChar tchar = iConsole->Getch();
 		//make a choice
 		while (tchar != EKeyEscape)
 			{
 			switch(tchar)
 				{
 				case '1':
 					{
 					RequestSwpChange();
 					break;
 					}
 				case '2':
 					{
 					RequestSsChange();
 					break;
 					}
 				case '3':
 					{
 					RequestSwpChangeWithNotificationL();
 					break;
 					}
 				case '4':
 					{
 					RequestSsChangeWithNotification();
 					break;
 					}
 				default:
 					{
 					iConsole->Printf(KTxtOptionNotSupported);
 					break;
 					}
 				}
 			iConsole->Printf(KTextPressAnyKey);
 			iConsole->Getch(); // Get and ignore character.
 			iConsole->ClearScreen();
 			iConsole->Printf(KTxtPressESC);
 			iConsole->Printf(KTxtContinue);
 	 		iConsole->Printf(KTxtReqChangeInSwp);
 	 		iConsole->Printf(KTxtReqChangeInSs);
 	 		iConsole->Printf(KTxtNotifyChangeInSwp);
 	 		iConsole->Printf(KTxtNotifyChangeInSs);
 			tchar = iConsole->Getch(); 
 			}
 		}
 	else
 		{
 		iConsole->Printf(KTxtError,connect);	
 		}
	iConsole->Printf(KTextPressAnyKeyToExit);
	iConsole->Getch(); // Get and ignore character.
	}
 
static void doExampleL()
	{
	CSysStateManagerExample* ssManager = CSysStateManagerExample::NewLC();
	ssManager->SSManagerL();
 	CleanupStack::PopAndDestroy(ssManager);
	}
	
extern TInt E32Main()
    {
	__UHEAP_MARK;
	CTrapCleanup* cleanup=CTrapCleanup::New(); // Get clean-up stack.
	if(cleanup!=NULL)
		{
		TRAPD (error,doExampleL());
		__ASSERT_ALWAYS(!error,User::Panic(KTxtExampleCode,error));
		}
	delete cleanup; // Delete clean-up stack.
	__UHEAP_MARKEND;
	return KErrNone; // And return.
    }
