// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the implementation of functions described in the RSecureSubSession class.
//



/**
 @file
*/

#include "secureclientandserver.h"
#include "secureclient.h"

/**
Creates a sub session and returns a handle to it.
@param aSession The session handle.
@return KErrNone if successful, otherwise one of the system-wide error codes.
*/
TInt RSecureSubSession::Open(RSecureSession &aSession)
	{
	return CreateSubSession(aSession,ESecureServerCreateSubSession);
	}

/**
Closes the sub session.
*/
void RSecureSubSession::Close()
	{
	RSubSessionBase::CloseSubSession(ESecureServerCloseSubSession);
	}

/**
Sets the counter value from a string.
@param aString The descriptor containing the counter value.
@return KErrNone if successful, otherwise one of the system-wide error codes.
*/
TInt RSecureSubSession::SetFromString(const TDesC& aString)
	{
	TIpcArgs args(&aString);
	return SendReceive(ESecureServerInitSubSession, args);
	}

/**
Increases the counter value by one.
*/
void RSecureSubSession::Increase()
	{
	if (SubSessionHandle())
		{
		SendReceive(ESecureServerIncrease);
		}
	}

/**
Increases the counter value by the supplied value.
@param aInt The increase in the counter value.
*/
void RSecureSubSession::IncreaseByL(TInt aInt)
	{
	if (SubSessionHandle())
		{
		TIpcArgs args(aInt);
		User::LeaveIfError(SendReceive(ESecureServerIncreaseBy, args));
		}
	}

/**
Decreases the counter value by one.
*/
void RSecureSubSession::Decrease()
	{
	if (SubSessionHandle())
		{
		SendReceive(ESecureServerDecrease);
		}
	}

/**
Decreases the counter value by the supplied value.
@param aInt The decrease in the counter value.
*/
void RSecureSubSession::DecreaseByL(TInt aInt)
	{
	if (SubSessionHandle())
		{
		TIpcArgs args(aInt);
		User::LeaveIfError(SendReceive(ESecureServerDecreaseBy, args));
		}
	}

/**
Resets the counter to zero.
*/
void RSecureSubSession::Reset()
	{
	if (SubSessionHandle())
		{
		SendReceive(ESecureServerReset);
		}
	}

/**
Gets the value of the counter.
@return The counter value.
*/
TInt RSecureSubSession::CounterValueL()
	{
	TInt res = KErrNotFound;
	TPckgBuf<TInt> pckg;

	if (SubSessionHandle())
		{
		TIpcArgs args(&pckg);
		User::LeaveIfError(SendReceive(ESecureServerValue, args));
		res = pckg();
		}
	return res;
	}

/**
Saves the counter value to the file counter.dat.
@see CSecureServerSubSession::SaveCounterValueL()
*/
void RSecureSubSession::SaveCounterL()
	{
	if (SubSessionHandle())
		{
		User::LeaveIfError(SendReceive(ESecureServerSaveCounter));
		}
	}

/**
Sets the counter to the value read from the file counter.dat.
@see CSecureServerSubSession::SetCounterValueFromFileL()
*/
void RSecureSubSession::SetCounterFromFileL()
	{
	if (SubSessionHandle())
		{
		User::LeaveIfError(SendReceive(ESecureServerSetCounterFromFile));
		}
	}
