// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the implementation of functions described in the CSecureServer class.
//



/**
 @file
*/
#include "secureclientandserver.h"
#include "secureserver.h"

/**
Panics the server.
@param aPanic The reason for the panic.
*/
void CSecureServer::PanicServer(TSecureServerPanic aPanic)
	{
	_LIT(KTxTSecureServer,"SecureServer");
	User::Panic(KTxTSecureServer,aPanic);
	}

/**
Creates a new server object and returns the handle to it.
@param aActiveObjectPriority The priority of the secure server. The secure server is an active object. It is derived from CActive and the priority value is the priority of the active object.
@see CActive
*/
CSecureServer* CSecureServer::NewL(CActive::TPriority aActiveObjectPriority)
	{
	CSecureServer* self = new (ELeave) CSecureServer(aActiveObjectPriority);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop();
	return self;
	}

/**
Creates a new object container when creating sub sessions.
@return An object container.
*/
CObjectCon* CSecureServer::NewContainerL()
	{
	return iContainerIndex->CreateL();
	}

/**
Removes an object from the object container.
@param aCon A pointer to the object to be removed from the container.
*/
void CSecureServer::RemoveContainer(CObjectCon *aCon)
	{
	iContainerIndex->Remove(aCon);
	}

/**
The secure server constructor.
The constructor calls the base class constructor CPolicyServer::CPolicyServer() and passes:
- the priority value
- the security policy defined by the constant KSecureServerPolicy in secureserver.h
- the type of session the server can create.
@param aActiveObjectPriority The priority of the secure server. The secure server is an active object. This priority value is passed down through the class hierarchy to the CActive base class.
*/
CSecureServer::CSecureServer(CActive::TPriority aActiveObjectPriority)
	: CPolicyServer(aActiveObjectPriority, KSecureServerPolicy, ESharableSessions)
	{
	// KSecurityPolicy is the TPolicy object that defines the security policy of the server.
	}

/**
Constructs a new object container index.
Subsession objects are reference counting objects, and are derived from CObject.
Reference counting objects are held in object containers. An object container is a CObjectCon type, and one of these is created for each session.
An object container index is a container for object containers, and one of these is created when the server starts.
@see CObject
@see CObjectCon
@see CObjectConIx
*/
void CSecureServer::ConstructL()
	{
	iContainerIndex = CObjectConIx::NewL();
	}

/**
Destructor.
*/
CSecureServer::~CSecureServer()
	{
	delete iContainerIndex;
	}

/**
Creates a server-side session object.
@param aVersion The version information supplied by the client.
@param aMessage Represents the details of the client request that is requesting the creation of the session.
*/
CSession2* CSecureServer::NewSessionL(const TVersion &aVersion,const RMessage2& /*aMessage*/) const
	{
	TVersion v(KSecureServMajorVersionNumber,KSecureServMinorVersionNumber,KSecureServBuildVersionNumber);
	if (!User::QueryVersionSupported(v,aVersion))
		{
		User::Leave(KErrNotSupported);
		}
	return new (ELeave) CSecureServerSession;
	}

/**
Performs a custom security check.

The policy server framework calls this function when a client sends requests to the server that are identified by the five enum values TSecureServerRequest::ESecureServerIncrease to TSecureServerRequest::ESecureServerReset.
The request numbers match the client-side requests identified by the functions:
- RSecureSubSession::Increase();
- RSecureSubSession::Decrease();
- RSecureSubSession::IncreaseBy(TInt anInt);
- RSecureSubSession::DecreaseBy(TInt anInt);
- RSecureSubSession::Reset();

See the const arrays defined in secureserver.h.
See the use of CPolicyServer::ECustomCheck in the third element in the array KSecureServerElementsIndex in secureserver.h.
@param aMsg The message to check.
@param aAction A reference to the action to take if the security check fails.
@param aMissing A reference to the list of security attributes missing from the checked process.
@return A value from TCustomResult.
*/
CPolicyServer::TCustomResult CSecureServer:: CustomSecurityCheckL(const RMessage2 &aMsg, TInt &aAction, TSecurityInfo &aMissing)
	{
	TInt sid;
	RMessagePtr2 msg = aMsg;
	sid = msg.SecureId();
	switch(aMsg.Function())
		{
		case ESecureServerIncreaseBy:
		case ESecureServerDecreaseBy:
			{
			TInt val = aMsg.Int0();
			if(val > 10)
				{
				if(sid > 0x70fffff5)
					{
					return EFail;
					}
				}
			}
			break;
		case ESecureServerReset:
		case ESecureServerDecrease:
		case ESecureServerIncrease:
			{
			}
			break;
		default:
			{
			aAction = EPanicClient;
			CheckFailedL(aMsg,aAction,aMissing);
			return EFail;
			}
		}
	return EPass;
	}
