// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the implementation of functions described in the CSecureServerSession class.
//



/**
 @file
*/
#include "secureclientandserver.h"
#include "secureserver.h"

/**
Constructor of the CSecureServerSession class.
*/
CSecureServerSession::CSecureServerSession()
	{
	}

/**
Creates the object index and object container for the session.
*/
void CSecureServerSession::CreateL()
	{
	iCountersObjectIndex = CObjectIx::NewL();
	iContainer=((CSecureServer*)Server())->NewContainerL();
	}

/**
Closes the session.
This function deletes the object index and object container.
*/
void CSecureServerSession::CloseSession()
	{
	delete iCountersObjectIndex;
	iCountersObjectIndex = NULL;
	((CSecureServer*)Server())->RemoveContainer(iContainer);
	iContainer = NULL;
	}

/**
Returns the appropriate CCountSubSession object given a client's sub session handle.
@param aMessage The request message.
@param aHandle Client's sub session handle.
@return The CCountSubSession object.
*/
CSecureServerSubSession* CSecureServerSession::CounterFromHandle(const RMessage2& aMessage,TInt aHandle)
	{
	CSecureServerSubSession* counter = (CSecureServerSubSession*)iCountersObjectIndex->At(aHandle);
	if (counter == NULL)
		{
		PanicClient(aMessage, EBadSubsessionHandle);
		}
	return counter;
	}

/**
Handles the servicing of a client request that has been passed to the server.
This function dispatches requests to the appropriate handler.
Some messages are handled by the session itself, and are implemented as CSecureServerSession member functions.
Other messages are handled by the subsession, and are implemented as CSecureServerSubSession member functions.
Note that the server calls this function after a client request has passed the security checks.
@param aMessage The message containing the details of the client request.
*/
void CSecureServerSession::ServiceL(const RMessage2& aMessage)
	{
	// Check if the message is intended to the parent session.
	switch (aMessage.Function())
		{
	case ESecureServerCreateSubSession:
		NewCounterL(aMessage);
		aMessage.Complete(KErrNone);
		return;
	case ESecureServerCloseSession:
		CloseSession();
		aMessage.Complete(KErrNone);
		return;
	case ESecureServerResourceCount:
		NumResourcesL(aMessage);
		aMessage.Complete(KErrNone);
		return;
		}
	// Check if the message is intended to a sub session.
	CSecureServerSubSession* counter=CounterFromHandle(aMessage,aMessage.Int3());
	switch (aMessage.Function())
		{
		case ESecureServerInitSubSession:
			counter->SetFromStringL(aMessage);
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerCloseSubSession:
			DeleteCounter(aMessage.Int3());
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerIncrease:
			counter->Increase();
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerIncreaseBy:
			counter->IncreaseBy(aMessage);
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerDecrease:
			counter->Decrease();
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerDecreaseBy:
			counter->DecreaseBy(aMessage);
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerReset:
			counter->Reset();
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerValue:
			counter->CounterValueL(aMessage);
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerSaveCounter:
			counter->SaveCounterValueL();
			aMessage.Complete(KErrNone);
			return;
		case ESecureServerSetCounterFromFile:
			counter->SetCounterValueFromFileL();
			aMessage.Complete(KErrNone);
			return;
		default:
			PanicClient(aMessage,EBadRequest);
			return;
		}
	}

/**
Creates a new sub session for the current session.
@param aMessage The message containing the details of the client request.
*/
void CSecureServerSession::NewCounterL(const RMessage2& aMessage)
	{
	// Create a sub session object and add it to the container index.
	CSecureServerSubSession* counter = new (ELeave) CSecureServerSubSession(this);
	iContainer->AddL(counter);

	TInt handle=iCountersObjectIndex->AddL(counter);
	TPckgBuf<TInt> handlePckg(handle);
	TRAPD(res,aMessage.WriteL(3,handlePckg));
	if (res!=KErrNone)
		{
		iCountersObjectIndex->Remove(handle);
		PanicClient(aMessage,EDescriptorNonNumeric);
		return;
		}
	// For every new sub session created, increase the resource count.
	iResourceCount++;
	}

/**
Deletes the sub session object.
@param aHandle The handle to the sub session object.
*/
void CSecureServerSession::DeleteCounter(TInt aHandle)
	{
	iCountersObjectIndex->Remove(aHandle);
	// For every new sub session deleted, decrease the resource count.
	iResourceCount --;
	}

/**
Gets the number of resources held by the session.
@return The number of resources held by the session.
*/
TInt CSecureServerSession::CountResources()
	{
	return iResourceCount;
	}

/**
Returns the resource count to the client by writing the resource count value into the client message.
@param aMessage The message containing the details of the client request.
*/
void CSecureServerSession::NumResourcesL(const RMessage2& aMessage)
	{
	TPckgBuf<TInt> resourcesPckg(iResourceCount);
	aMessage.WriteL(0,resourcesPckg);
	}

/**
Panics the client.
@param aMessage The message containing the details of the client request.
@param aPanic The reason for the panic.
*/
void CSecureServerSession::PanicClient(const RMessage2& aMessage,TInt aPanic) const
	{
	_LIT(KTxtServer,"SecureServer");
	aMessage.Panic(KTxtServer,aPanic);
	}
