// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the implementation of functions described in the CSecureServerSubSession class.
//



/**
 @file
*/
#include "secureclientandserver.h"
#include "secureserver.h"

/**
Constructor of the CSecureServerSubSession class.
@param aSession The session to which this subsession is to be attached.
@see CSecureServerSession
*/
CSecureServerSubSession::CSecureServerSubSession(CSecureServerSession* aSession) : iSession(aSession)
 {
 }

/**
Sets the counter value from a string.
@param aMessage Message that encapsulates the string sent by the client.
*/
void CSecureServerSubSession::SetFromStringL(const RMessage2& aMessage)
	{
	TInt deslen = aMessage.GetDesLength(0);
	RBuf buffer;

	buffer.CreateL(deslen);
	buffer.CleanupClosePushL();
	// The descriptor is encapsulated in the 0th offset of aMessage.
	// Hence, read the 0th offset of the message to the buffer.
	aMessage.ReadL(0,buffer,0);
	// Check for empty string.
	if (buffer.Length() == 0)
		{
		User::Leave(ENonNumericString);
		}
	TLex16 lexer;
	lexer.Assign(buffer);
	while (!lexer.Eos())
		{
		TChar thechar;
		thechar = lexer.Peek();
		// Check for non numeric character.
		if (!thechar.IsDigit())
			{
			User::Leave(ENonNumericString);
			}
		lexer.Inc();
		}
	lexer.Assign(buffer);
	if (lexer.Val(iCount))
		{
		User::Leave(ENonNumericString);
		}
	CleanupStack::PopAndDestroy();
	}

/**
Increases the counter value by one.
*/
void CSecureServerSubSession::Increase()
	{
	iCount ++;
	}

/**
Increases the counter value by the value passed in the message.
@param aMessage The message containing the value sent by the client.
@see RSecureSubSession::Increase()
*/
void CSecureServerSubSession::IncreaseBy(const RMessage2& aMessage)
	{
	iCount += aMessage.Int0();
	}

/**
Decreases the counter value by one.
*/
void CSecureServerSubSession::Decrease()
	{
	iCount --;
	}

/**
Decreases the counter value by the value passed in the message.
@param aMessage The message containing the value sent by the client.
@see RSecureSubSession::Decrease()
*/
void CSecureServerSubSession::DecreaseBy(const RMessage2& aMessage)
	{
	iCount -= aMessage.Int0();
	}

/**
Resets the counter to 0.
@see RSecureSubSession::Reset()
*/
void CSecureServerSubSession::Reset()
	{
	iCount = 0;
	}

/**
Sends the counter value to the client.
@param aMessage The message that encapsulates the counter value sent to the client by the server.
*/
void CSecureServerSubSession::CounterValueL(const RMessage2& aMessage)
	{
	// Write the value of counter to the 0th offset of aMessage by converting it to a package.
	TPckgBuf<TInt> p(iCount);
	aMessage.WriteL(0,p);
	}

/**
Saves the counter value in the file counter.dat.

The example creates the file in the private directory of the executable process. The private directory is created on a devices system drive. This is the drive that a device marks as the standard place for application data.

In this example, this is the location:
C:\private\70fffff0\counter.dat

0x70fffff0 is the Secure ID (SID) of the process; this is defined in secureclient.mmp.

C: is the system drive. When built and run on the emulator, see \TechView\epoc32\...\C:\...
*/
void CSecureServerSubSession::SaveCounterValueL()
	{
	RFs fs;
	User::LeaveIfError(fs.Connect());
	_LIT(KFileName,"counter.dat");
	User::LeaveIfError(fs.CreatePrivatePath(RFs::GetSystemDrive()));
	User::LeaveIfError(fs.SetSessionToPrivate(RFs::GetSystemDrive()));
	TFileName thePath;
	User::LeaveIfError(fs.PrivatePath(thePath));
	TInt err=fs.MkDir(thePath);
	if (err!=KErrAlreadyExists)
		{
		User::LeaveIfError(err);
		}
	thePath.Append(KFileName);
	RFile file;

	TBuf8<20> counterVal;
	counterVal.FillZ();
	// Append the counter value to a buffer and write to the file.
	counterVal.AppendNum(iCount);
	User::LeaveIfError(file.Replace(fs,thePath,EFileWrite));
	User::LeaveIfError(file.Write(counterVal));
	User::LeaveIfError(file.Flush());

	file.Close();
	fs.Close();
	}

/**
Gets the counter value from the file counter.dat.
@see CSecureServerSubSession::SaveCounterValueL()
*/
void CSecureServerSubSession::SetCounterValueFromFileL()
	{
	RFs fs;
	User::LeaveIfError(fs.Connect());
	_LIT(KFileName,"counter.dat");
	User::LeaveIfError(fs.CreatePrivatePath(RFs::GetSystemDrive()));
	User::LeaveIfError(fs.SetSessionToPrivate(RFs::GetSystemDrive()));
	TFileName thePath;
	User::LeaveIfError(fs.PrivatePath(thePath));
	TInt err=fs.MkDir(thePath);
	if (err!=KErrAlreadyExists)
		{
		User::LeaveIfError(err);
		}
	thePath.Append(KFileName);
	RFile file;

	TBuf8<20> readVal;
	User::LeaveIfError(file.Open(fs,thePath,EFileRead));
	// Read counter value to the readVal buffer.
	User::LeaveIfError(file.Read(readVal));
	file.Close();
	// Convert value in readVal to TInt.
	TLex8 lex(readVal);
	lex.Val(iCount);

	fs.Close();
	}
