/*
 * Copyright  2008 Nokia Corporation.
 */


// INCLUDES
#include <e32base.h> 
#include "aknnotewrappers.h"
#include "..\inc\BluetoothRefreshTimer.h"
#include "btdiscoverer.h"
#include "DeviceListContainer.h"

// a second
const TInt KThreadSecond = 1000 * 1000;

_LIT( refreshDevices, "refreshing devices..." );

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::CBluetoothRefreshTimer()
//
// Constructor.
// ----------------------------------------------------------------------------
CBluetoothRefreshTimer::CBluetoothRefreshTimer(
         CSharedIntermediator* aSMediator,
         CActiveSchedulerWait* aWait)
	     : CTimer(CActive::EPriorityStandard),
	       iWait(aWait), iSMediator(aSMediator), 
           iBTDiscoverer(NULL), iRefreshTime(0), iTime(0)  
	{
	iSMediator->SetBTRefreshTimerPtr(this);
	}

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::~CBluetoothRefreshTimer()
//
// Destructor.
// ----------------------------------------------------------------------------
CBluetoothRefreshTimer::~CBluetoothRefreshTimer()
	{
	Cancel();
	delete iBTDiscoverer;
	}


CBluetoothRefreshTimer* CBluetoothRefreshTimer::NewL(
     CSharedIntermediator* aSMediator, CActiveSchedulerWait* aWait)
	{
	CBluetoothRefreshTimer* self = CBluetoothRefreshTimer::NewLC(
	                                                    aSMediator, aWait );
	CleanupStack::Pop(self);
	return self;
	}

CBluetoothRefreshTimer* CBluetoothRefreshTimer::NewLC(
    CSharedIntermediator* aSMediator, CActiveSchedulerWait* aWait)
	{
	CBluetoothRefreshTimer* self = new (ELeave) CBluetoothRefreshTimer(
	                                                     aSMediator, aWait );
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
	}

// Standard Symbian OS 2nd phase constructor
void CBluetoothRefreshTimer::ConstructL()
	{
	CTimer::ConstructL();
   
	iBTDiscoverer = CBTDiscoverer::NewL(iSMediator);
	}

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::StartL()
//
// Start discovering.
// ----------------------------------------------------------------------------
void CBluetoothRefreshTimer::StartL()
	{
	iRefreshTime = KThreadSecond*iSMediator->RefreshTimerInitlVal();

	iBTDiscoverer->StartDiscoveringDevicesL();
	After( KThreadSecond );
	}

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::StopWaitLoop()
//
// Cancel active scheduler wait loop and cancel bluetooth discoverer.
// ----------------------------------------------------------------------------
void CBluetoothRefreshTimer::StopWaitLoop()
	{
	iBTDiscoverer->DoCancel();
	Cancel();
	iWait->AsyncStop();	
	}	

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::RunL()
//
// ----------------------------------------------------------------------------
void CBluetoothRefreshTimer::RunL()
	{
	// Is it time to refresh yet
	if ( iTime >= iRefreshTime ) 
		{
		CDeviceListContainer* container = iSMediator->DeviceListContainer();
	
		container->ClearListBox();
		container->AddItemL( refreshDevices );

		// Clear also shared intermediator
		iSMediator->ResetArray();

		// Find bluetooth devices again
		iBTDiscoverer->RefreshDevices();
		
		iTime = 0;
		}

	// Stop seaching for bt devices
	if (iSMediator->StopSearching())
		{
		// Stop Active Scheduler wait loop
		StopWaitLoop();
		}

	// Wait for a while and run the same function again 
	iTime+=KThreadSecond;
	After( KThreadSecond );
	}

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::DoCancel()
//
// Cancel and stop the timer. 
// ----------------------------------------------------------------------------
void CBluetoothRefreshTimer::DoCancel()	
	{
	CTimer::DoCancel();
	}

// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::SetRefreshTime(TInt aRefreshTime)
//
// Set a new refresh time. Convert time to microseconds.
// ----------------------------------------------------------------------------
void CBluetoothRefreshTimer::SetRefreshTime(TInt aRefreshTime)
	{
	
	if ( aRefreshTime < KRefreshTimeMin )
		{
		aRefreshTime = KRefreshTimeDefault;
		}

	// Convert to microseconds
	iRefreshTime = aRefreshTime * KThreadSecond;
	}
// ----------------------------------------------------------------------------
// CBluetoothRefreshTimer::RunError(TInt)
//
// Handles a leave occurring in the request completion event handler RunL().
// ----------------------------------------------------------------------------
TInt CBluetoothRefreshTimer::RunError(TInt)
	{
	return KErrNone;
	}

