/*
 * Copyright  2008 Nokia Corporation.
 */


// INCLUDES
#include "..\inc\btdiscoverer.h"
#include "SharedIntermediator.h"
#include "BluetoothInfo.h"

#include <btmanclient.h>
#include <btextnotifiers.h>
#include <aknlists.h>
#include <btsdp.h>

#include <bt_sock.h> 
#include <BTDevice.h>

_LIT(KBTProtocol, "BTLinkManager");
_LIT( KLine, "-" );

const TInt KMinBTAddressLength = 5;
const TInt KBufSize = 32;
const TInt KAddressBufSize = 1024;
const TInt KConversionChars = 2;

CBTDiscoverer* CBTDiscoverer::NewL(CSharedIntermediator* aSMediator)
    {
    CBTDiscoverer* self = CBTDiscoverer::NewLC(aSMediator);
    CleanupStack::Pop(self);
    return self;
    }

CBTDiscoverer* CBTDiscoverer::NewLC( CSharedIntermediator* aSMediator )
    {
    CBTDiscoverer* self = new (ELeave) CBTDiscoverer( aSMediator );
    CleanupStack::PushL( self );
    self->ConstructL();
    return self;
    }

// ----------------------------------------------------------------------------
// CBTDiscoverer::CBTDiscoverer()
//
// Constructor.
// ----------------------------------------------------------------------------
CBTDiscoverer::CBTDiscoverer( CSharedIntermediator* aSMediator )
	: CActive( EPriorityStandard ), iSMediator ( aSMediator )
	{
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::CBTDiscoverer()
//
// Destructor.
// ----------------------------------------------------------------------------
CBTDiscoverer::~CBTDiscoverer()
	{
	Cancel();

	iDeviceNames->ResetAndDestroy();
	delete iDeviceNames;

	iDeviceAddress->ResetAndDestroy();
	delete iDeviceAddress;
	
	iInqSockAddrArray.Close();
	iHostResolver.Close();
	iSocketServer.Close();
	}

// Standard Symbian OS 2nd phase constructor
void CBTDiscoverer::ConstructL()
    {
	iDeviceNames =  new (ELeave) CArrayPtrFlat<HBufC>(1);
	iDeviceAddress = new (ELeave) CArrayPtrFlat<HBufC>(1);
    
	CActiveScheduler::Add(this);
    }
// ----------------------------------------------------------------------------
// CBTDiscoverer::GetNames()
//
// Return a pointer to list of names.
// ----------------------------------------------------------------------------
CArrayPtrFlat< HBufC >* CBTDiscoverer::GetNames()
	{
	return iDeviceNames;
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::GetDeviceAddress()
//
// Return a pointer to list of addresses.
// ----------------------------------------------------------------------------
CArrayPtrFlat< HBufC >* CBTDiscoverer::GetDeviceAddress()
	{
	return iDeviceAddress;
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::StartDiscoveringDevicesL()
//
// Find devices.
// ----------------------------------------------------------------------------
 void CBTDiscoverer::StartDiscoveringDevicesL()
	{
	User::LeaveIfError(iSocketServer.Connect() ); 
	
	TProtocolDesc protocolInfo;

	// Gets the description of a protocol by name.
	User::LeaveIfError(iSocketServer.FindProtocol(
                                    KBTProtocol(), protocolInfo) ); 

	// Initialises a name resolution service provided by a particular protocol. 
	// It must be called before other object functions are used.
	User::LeaveIfError( iHostResolver.Open( iSocketServer, 
                    protocolInfo.iAddrFamily, protocolInfo.iProtocol ) );

	// Interested in all bluetooth devices that have BT turned on.
	iSockAddr.SetIAC(KGIAC); 

	// Resolve name of the device and device address. Do not use cache.
	iSockAddr.SetAction(KHostResInquiry | KHostResName | KHostResIgnoreCache); 
	
	// Start discovering devices asynchronously. RunL is called.
	iHostResolver.GetByAddress(iSockAddr,iNameEntry,iStatus); 

	// Set this active object active.
	SetActive();

	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::RunL()
//
// ----------------------------------------------------------------------------
 void CBTDiscoverer::RunL()
	{
	
	// Found a new device, iStatus would be KErrHostResNoMoreResults
	// if no more new devices are found.
	if ( iStatus == KErrNone ) 
		{	
		PushListL(iNameEntry); 
		// Find the next bluetooth device
		iHostResolver.Next(iNameEntry,iStatus);

		SetActive();
		}	
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::TransformSockAddressL(TInquirySockAddr& aInquirySockAddr)
//
// Transform inquiryaddress to format XX-XX-XX-XX-XX (BT address)
// ----------------------------------------------------------------------------
void CBTDiscoverer::TransformSockAddressL(TInquirySockAddr& aInquirySockAddr) 
	{
	TBTDevAddr localAddr= aInquirySockAddr.BTAddr();
	TBuf<KAddressBufSize> btAddr; 

	for(TInt i=0; i < localAddr.Des().Length() ;i++) 
		{ 
		btAddr.AppendNumFixedWidthUC( localAddr[i], EHex, KConversionChars );
		if ( i != localAddr.Des().Length() -1 ) 
			{
			btAddr.Append( KLine );
			}
		}

	HBufC* element = HBufC::New(KBTDeviceAddress);
	element = btAddr.Alloc();
	iDeviceAddress->AppendL( element );
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::RunError(TInt aError)
// ----------------------------------------------------------------------------
TInt CBTDiscoverer::RunError(TInt)
	{
	return KErrNone;
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::DoCancel()
// ----------------------------------------------------------------------------
 void CBTDiscoverer::DoCancel()
	{
	iHostResolver.Cancel();
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::PushListL(TNameEntry& aNameEntry) 
//
// Separete both name/address from the aNameEntry and insert them into separe 
// lists.
// ----------------------------------------------------------------------------
void CBTDiscoverer::PushListL(TNameEntry& aNameEntry) 
	{ 

	TInquirySockAddr &addr = TInquirySockAddr::Cast(aNameEntry().iAddr); 

	// Add only different devices. 
	for( TInt i = 0; i < iInqSockAddrArray.Count(); i++ )
		{ 
		if( iInqSockAddrArray[i].BTAddr() == addr.BTAddr( ) || 
			 addr.BTAddr().Des().Length() < KMinBTAddressLength )
			{ 
			return; 
			}	 
		}	 

	HBufC* element = NULL;
	
	// Check the length of the bluetooth device name
	if ( aNameEntry().iName.Length() >= KBTDeviceLength )
		{
		element = aNameEntry().iName.Left( KBTDeviceLength -1 ).AllocLC();
		}
	// Get the name 
	else {
		element = aNameEntry().iName.AllocLC();
		}
	// Add the name onto the list. Devices must have some kind of name
	// otherwise it won't be added.
	if ( element->Length() != 0 ) 
		{
		iDeviceNames->AppendL( element );
		CleanupStack::Pop( element ); 		
		}
	
	TInquirySockAddr *addr_ptr = new (ELeave)TInquirySockAddr( addr ); 
	
	CleanupStack::PushL( addr_ptr ); 

	// Append the BT address onto the list
	User::LeaveIfError( iInqSockAddrArray.Append(*addr_ptr) ); 
	CleanupStack::Pop( addr_ptr ); 

	// Make address printable.
	TransformSockAddressL( *addr_ptr );

	// Get name of the device and BT address as hex. Add name&address
	// into shared intermediator.
	TPtr16 deviceName = (*iDeviceNames)[ iDeviceNames->Count() - 1 ]->Des();
	TPtr16 deviceBTAddr = 
                    (*iDeviceAddress)[ iDeviceAddress->Count() - 1 ]->Des();
	TBluetoothInfo btInfo = TBluetoothInfo (deviceName, deviceBTAddr);

	// Give data to the shared intermediator. Shared intermediator gives
	// device names to the listbox after this.
	iSMediator->AddBluetoothInfoL(btInfo);
	}

// ----------------------------------------------------------------------------
// CBTDiscoverer::RefreshDevices()
//
// Cancels the wait for completion of an outstanding bluetooth request. 
// Cleans lists and starts the search again by calling GetByAddress.
// ----------------------------------------------------------------------------
void CBTDiscoverer::RefreshDevices()
	{
	// In case bt discover is already active.
	Cancel();

	 //Clear all lists 
	iDeviceNames->Reset();
	iInqSockAddrArray.Reset();	
	iDeviceAddress->Reset();
	
	iSockAddr.SetAction(KHostResInquiry | KHostResName | KHostResIgnoreCache); 

	// Start discovering devices asynchronously. RunL is called afterwards.
	iHostResolver.GetByAddress(iSockAddr,iNameEntry,iStatus); 
	SetActive();
	}
