/*
 * Copyright  2008 Nokia Corporation.
 */


// INCLUDES
#include "SharedIntermediator.h"
#include "BluetoothInfo.h"
#include "Devicelistcontainer.h"
#include "BluetoothRefreshTimer.h"
#include "ThreadAOAppUi.h"
#include "btdiscoverer.h"

//CONSTANTS
const TInt KRefreshTime = 20;

// ----------------------------------------------------------------------------
// CSharedIntermediator::CSharedIntermediator(CDeviceListContainer*
//                                                 aDeviceListContainer)
//
// Constructor.
// ----------------------------------------------------------------------------
CSharedIntermediator::CSharedIntermediator(
			CDeviceListContainer* aDeviceListContainer)
			: iDeviceListContainer(aDeviceListContainer),
			iStopSearching(false),
			iBTRefreshTimer(NULL),
		    iInititialRefreshTime( KRefreshTime )
	{
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::~CSharedIntermediator()
//
// Destructor.
// ----------------------------------------------------------------------------
CSharedIntermediator::~CSharedIntermediator()
	{
	iMutex.Close();
	if(	iBluetoothInfoArray )
		{
		iBluetoothInfoArray->Reset();
		}
	delete iBluetoothInfoArray;
	}

CSharedIntermediator* CSharedIntermediator::NewL(CDeviceListContainer* aView)
	{
	CSharedIntermediator* self = CSharedIntermediator::NewLC(aView);
	CleanupStack::Pop();
	return self;
	}

CSharedIntermediator* CSharedIntermediator::NewLC(CDeviceListContainer* aView)
	{
	CSharedIntermediator* self = new (ELeave) CSharedIntermediator(aView);
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
	}

// Standard Symbian OS 2nd phase constructor
void CSharedIntermediator::ConstructL()
	{
	iBluetoothInfoArray = new (ELeave) CArrayFixFlat <TBluetoothInfo>(1);	
	iMutex.CreateLocal();	
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::AddBluetoothInfoL(TBluetoothInfo& btInfoElement)
//
// Add bluetooth info into bluetooth info array and add's new bluetooth
// device's name into UI's list box.
// ----------------------------------------------------------------------------
void CSharedIntermediator::AddBluetoothInfoL(TBluetoothInfo& aBtInfoElement)
	{
	iMutex.Wait();

    // Add new device data into array
	iBluetoothInfoArray->AppendL(aBtInfoElement);	

	TBuf <KBTDeviceLength> deviceName;
    aBtInfoElement.GetDeviceName(deviceName);
    // Show new device name in the list box.
	iDeviceListContainer->AddItemL(deviceName);

	iMutex.Signal();
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::ResetArray()
//
// Clears bluetooth info array.
// ----------------------------------------------------------------------------
void CSharedIntermediator::ResetArray()
	{
	iMutex.Wait();
	iBluetoothInfoArray->Reset();
	iMutex.Signal();
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::SetRefreshTimerInitlVal(TInt aRefreshTime)
//
// Set thread one's CBluetoothrefreshtimer initial value.
// ----------------------------------------------------------------------------
void CSharedIntermediator::SetRefreshTimerInitlVal(TInt aRefreshTime)
	{
	if ( aRefreshTime < KRefreshTimeMin )
		{
		aRefreshTime = KRefreshTimeDefault;
		}

	iInititialRefreshTime = aRefreshTime;
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::RefreshTimerInitlVal()
//
// Get thread one's CBluetoothrefreshtimer initial value.
// ----------------------------------------------------------------------------
TInt CSharedIntermediator::RefreshTimerInitlVal()
	{
	return iInititialRefreshTime;
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::GetAddress(TDes& aAddress, TInt aIndex)
//
// Get bluetooth device address from the bluetooth info array.
// ----------------------------------------------------------------------------
void CSharedIntermediator::GetAddress(TDes& aAddress, TInt aIndex)
	{ 
	iMutex.Wait();
	if ( aIndex < iBluetoothInfoArray->Count() )
		{
		TBluetoothInfo& tempBTInfo = iBluetoothInfoArray->At(aIndex);
		tempBTInfo.GetAddressHex(aAddress);
		}
	iMutex.Signal();
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::SetStopSearching(bool aStopSearhing)
//
//  Set variable (iStopSearching), which tells to thread
//  that Activescheduler's wait loop should stop. After that thread cleans up
//  itself, the program is terminated "naturally".
// ----------------------------------------------------------------------------
void CSharedIntermediator::SetStopSearching(bool aStopSearhing)
	{
	iStopSearching = aStopSearhing;
	}


// ----------------------------------------------------------------------------
// TBool CSharedIntermediator::StopSearching()
// 
// ----------------------------------------------------------------------------	
TBool CSharedIntermediator::StopSearching()
	{
	return iStopSearching;
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::BTRefreshTimerPtr()
//
//  Get pointer to thread's bluetooth refresh timer. New refresh time is set
//  in main program. Refresh time is time that has passed after bluetooth
//  discovery is started again.
// ----------------------------------------------------------------------------
CBluetoothRefreshTimer* CSharedIntermediator::BTRefreshTimerPtr()
	{
	return iBTRefreshTimer;
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::SetBTRefreshTimerPtr(CBluetoothRefreshTimer* timer)
//
// Set pointer to thread's bluetooth refresh timer. New refresh time is set
// in the main program by the user.
// ----------------------------------------------------------------------------
void CSharedIntermediator::SetBTRefreshTimerPtr(CBluetoothRefreshTimer* aTimer)
	{
	iBTRefreshTimer = aTimer;
	}

// ----------------------------------------------------------------------------
// CSharedIntermediator::DeviceListContainer()
//
// Get pointer to listbox container. Function is used
// when CBTRefreshTimer clears the listbox.
// ----------------------------------------------------------------------------
CDeviceListContainer* CSharedIntermediator::DeviceListContainer()
	{
	return iDeviceListContainer;
	}

