/*
 * Copyright  2008 Nokia Corporation.
 */

// INCLUDE FILES
#include "ContactsModelAppUi.h"
#include "ContactsModelContainer.h"
#include "ContactsModelDocument.h"
#include <ContactsModel.rsg>
#include "ContactsModel.hrh"

#include <AknCommonDialogs.h>
#include <CAknFileSelectionDialog.h>
#include <CAknFileNamePromptDialog.h>
#include <aknnotewrappers.h>
#include <eikmenup.h>
#include <f32file.h>
#include <s32file.h>
#include <avkon.hrh>
#include <maknfileselectionobserver.h>

_LIT(KDefaultFileName, "vcard");
_LIT(KFailedToOpen, "Failed to open file for import");
_LIT(KFailedToImport, "Failed to import");
_LIT(KFailedToCreateFile, "Failed to create file for export");

const TInt KBufferSize = 256;

// ================= MEMBER FUNCTIONS =======================
//
// ----------------------------------------------------------
// CContactsModelAppUi::ConstructL()
// ----------------------------------------------------------
//
void CContactsModelAppUi::ConstructL()
    {
    BaseConstructL(EAknEnableSkin);

    iAppContainer = new (ELeave) CContactsModelContainer;
    iAppContainer->SetMopParent( this );
    iAppContainer->ConstructL( ClientRect(), (CContactsModelDocument*)(iDocument) );
    AddToStackL( iAppContainer );
    }

// ----------------------------------------------------
// CContactsModelAppUi::~CContactsModelAppUi()
// Destructor
// Frees reserved resources
// ----------------------------------------------------
//
CContactsModelAppUi::~CContactsModelAppUi()
    {
    if (iAppContainer)
        {
        RemoveFromStack( iAppContainer );
        delete iAppContainer;
        }
   }

// ------------------------------------------------------------------------------
// CContactsModelAppUi::DynInitMenuPaneL(TInt aResourceId,CEikMenuPane* aMenuPane)
//  This function is called by the EIKON framework just before it displays
//  a menu pane. Its default implementation is empty, and by overriding it,
//  the application can set the state of menu items dynamically according
//  to the state of application data.
// ------------------------------------------------------------------------------
//
void CContactsModelAppUi::DynInitMenuPaneL(
    TInt aResourceId,CEikMenuPane* aMenuPane)
    {
	if (aResourceId == R_CONTACTSMODEL_MENU)
    	{
    	if(iAppContainer->ItemCount() < 1)
    		{
    		aMenuPane->SetItemDimmed(EContactsModelCmdAppExport, ETrue);
    		}
    	}
    }

// ----------------------------------------------------
// CContactsModelAppUi::HandleKeyEventL(
//     const TKeyEvent& aKeyEvent,TEventCode /*aType*/)
// takes care of key event handling
// ----------------------------------------------------
//
TKeyResponse CContactsModelAppUi::HandleKeyEventL(
    const TKeyEvent& /*aKeyEvent*/,TEventCode /*aType*/)
    {
    return EKeyWasNotConsumed;
    }

// ----------------------------------------------------
// CContactsModelAppUi::HandleCommandL(TInt aCommand)
// takes care of command handling
// ----------------------------------------------------
//
void CContactsModelAppUi::HandleCommandL(TInt aCommand)
    {
    switch ( aCommand )
        {
        case EAknSoftkeyExit:
        case EEikCmdExit:
            {
            Exit();
            break;
            }
        case EContactsModelCmdAppImport:
            {

			TBuf<KBufferSize> fileName;
			//select file to import from
			if(AknCommonDialogs::RunSelectDlgLD(fileName, R_FILE_SELECTION_DIALOG))
			{
				//import to the default contact database
				ImportL(fileName);
				//update listbox to display imported contacts
				HandleModelChangeL();
			}

            break;
            }
        case EContactsModelCmdAppExport:
            {
			//get file to export to
			TFileName defaultFileName(KDefaultFileName);
			if(AknCommonDialogs::RunSaveDlgLD(defaultFileName, R_MEMORY_SELECTION_DIALOG))
			{
				//and export selected item
				ExportL(defaultFileName, iAppContainer->GetSelectedItem());
			}

            break;
            }

        default:
            break;
        }
    }

// ----------------------------------------------------
// CContactsModelAppUi::ImportL(TDesC& aFileName)
// Import new contact information (VCard)
// ----------------------------------------------------
//
void CContactsModelAppUi::ImportL(const TDesC& aFileName)
{
	// Create a file to read contacts
	RFs fileSession;
	User::LeaveIfError(fileSession.Connect());
	CleanupClosePushL(fileSession);

	RFile file;
	if (file.Open(fileSession, aFileName, EFileRead) != KErrNone)
	{
		CAknInformationNote* informationNote = new (ELeave)CAknInformationNote;
		informationNote->ExecuteLD(KFailedToOpen);
		CleanupStack::PopAndDestroy(); // Close fileSession
		return;
	}
	CleanupClosePushL(file);

	//open file
    RFileReadStream inputFileStream(file);
    CleanupClosePushL(inputFileStream);

	TInt result = ((CContactsModelDocument*)iDocument)->ImportL(inputFileStream);
	if(result!=KErrNone)
	{
		CAknInformationNote* informationNote = new (ELeave)CAknInformationNote;
		informationNote->ExecuteLD(KFailedToImport);
	}

    // Clean up
    CleanupStack::PopAndDestroy(); // Close inputFileStream

    CleanupStack::PopAndDestroy(); // Close file
    CleanupStack::PopAndDestroy(); // Close fileSession

	// read from the default database to display new items
	((CContactsModelDocument*)iDocument)->UpdateContactsL();
}

// ----------------------------------------------------
// CContactsModelAppUi::ExportL(TDesC& aFileName, TInt aItemIndex)
// Export contact information
// ----------------------------------------------------
//
void CContactsModelAppUi::ExportL(const TDesC& aFileName, TInt aItemIndex)
{
	// Create a file to write contacts
	RFs fileSession;
	User::LeaveIfError(fileSession.Connect());
	CleanupClosePushL(fileSession);

	RFile file;
	if (file.Replace(fileSession, aFileName, EFileWrite) != KErrNone)
	{
		CAknInformationNote* informationNote = new (ELeave)CAknInformationNote;
		informationNote->ExecuteLD(KFailedToCreateFile);
		CleanupStack::PopAndDestroy(); // Close fileSession
		return;
	}
	CleanupClosePushL(file);

	//open file
    RFileWriteStream outputFileStream(file);
    CleanupClosePushL(outputFileStream);

	((CContactsModelDocument*)iDocument)->ExportL(outputFileStream, aItemIndex);

    // Clean up
    CleanupStack::PopAndDestroy(); // Close outputFileStream

    CleanupStack::PopAndDestroy(); // Close file
    CleanupStack::PopAndDestroy(); // Close fileSession
}

void CContactsModelAppUi::HandleModelChangeL()
    {
	//read from the document to the listbox model
    iAppContainer->UpdateL();
	//and redraw
	iAppContainer->DrawNow();
    }

// ---------------------------------------------------------
// CDBMSAppUi::HandleResourceChangeL()
// Called by framework when resource is changed.
// ---------------------------------------------------------
//
void CContactsModelAppUi::HandleResourceChangeL(TInt aType)
	{
	CAknAppUi::HandleResourceChangeL(aType); //call to upper class

    // ADDED FOR SCALABLE UI SUPPORT
    // *****************************
	//if ( aType == KEikDynamicLayoutVariantSwitch )
	//hard coded constant so it can be compiled with first edition
	if ( aType == 0x101F8121 )
		{
        if(iAppContainer)
        	iAppContainer->SetRect( ClientRect() );
	    }
	}

// End of File

