/*
 * Copyright  2008 Nokia Corporation.
 */

// INCLUDE FILES
#include "ContactsModelDocument.h"
#include "ContactsModelAppUi.h"

#include <cntfilt.h>
#include <cntitem.h>

const TInt KBufferSize = 256;

// ================= MEMBER FUNCTIONS =======================

// constructor
CContactsModelDocument::CContactsModelDocument(CEikApplication& aApp)
: CAknDocument(aApp)
    {
    }

// destructor
CContactsModelDocument::~CContactsModelDocument()
    {
	delete iContacts;
	delete iTextDef;
    }

// EPOC default constructor can leave.
void CContactsModelDocument::ConstructL()
    {
	iContacts = NULL;
	UpdateContactsL();
	//create custom text definition to display contact item
	iTextDef = CContactTextDef::NewL();
	//add given name field and family name field
	iTextDef->AppendL(TContactTextDefItem(KUidContactFieldGivenName, _L(" ")));
	iTextDef->AppendL(TContactTextDefItem(KUidContactFieldFamilyName));
    }

// Two-phased constructor.
CContactsModelDocument* CContactsModelDocument::NewL(
        CEikApplication& aApp)     // CContactsModelApp reference
    {
    CContactsModelDocument* self = new (ELeave) CContactsModelDocument( aApp );
    CleanupStack::PushL( self );
    self->ConstructL();
    CleanupStack::Pop();

    return self;
    }

// Get the contact information
HBufC * CContactsModelDocument::GetItemL(TInt anIndex)
	{
	// Open the default contact database
	CContactDatabase* contactsDb = CContactDatabase::OpenL();
	CleanupStack::PushL(contactsDb);

	TBuf<KBufferSize> buf;
	//read contact item with the default text definition
	if((*iContacts).Count() >= anIndex)
		contactsDb->ReadContactTextDefL((*iContacts)[anIndex], buf, iTextDef);

	CleanupStack::PopAndDestroy(); // contactsDb
	return buf.AllocL();
	}

// return contacts items number
TInt CContactsModelDocument::ItemCount()
	{
	return iContacts->Count();
	}

void CContactsModelDocument::UpdateContactsL()
	{
	//open database
	// Open the default contact database
	CContactDatabase* contactsDb = CContactDatabase::OpenL();
	CleanupStack::PushL(contactsDb);

	CCntFilter *filter = CCntFilter::NewLC();

	//get all contact items (no groups, templates...)
	filter->SetContactFilterTypeALL(EFalse);
	filter->SetContactFilterTypeCard(ETrue);

	contactsDb->FilterDatabaseL(*filter);

	if(iContacts)
	{
		delete iContacts;
		iContacts = NULL;
	}
	iContacts = CContactIdArray::NewL(filter->iIds);

	CleanupStack::PopAndDestroy(2); //filter, contactsDb
	}

TInt CContactsModelDocument::ImportL(RReadStream& aReadStream)
	{
	// Open the default contact database
	CContactDatabase* contactsDb = CContactDatabase::OpenL();
	CleanupStack::PushL(contactsDb);

	TBool result = EFalse;
	TUid p1;
	p1.iUid = KVersitEntityUidVCard;
	CArrayPtr<CContactItem>* imported = contactsDb->ImportContactsL(p1,
								aReadStream,
								result,
								CContactDatabase::ETTFormat);
	imported->ResetAndDestroy();
	delete imported;

	CleanupStack::PopAndDestroy(); // contactsDb

	return result?KErrNone:KErrGeneral;
	}

void CContactsModelDocument::ExportL(RWriteStream& aWriteStream, TInt aExportItem)
	{
	// Open the default contact database
	CContactDatabase* contactsDb = CContactDatabase::OpenL();
	CleanupStack::PushL(contactsDb);

	CContactIdArray* exportContacts = CContactIdArray::NewL();
	CleanupStack::PushL(exportContacts);

	if((*iContacts).Count() >= aExportItem)
		exportContacts->AddL( (*iContacts)[aExportItem] );

	TUid p1;
	p1.iUid = KVersitEntityUidVCard;
	contactsDb->ExportSelectedContactsL(p1,
								*exportContacts,
								aWriteStream,
								CContactDatabase::EExcludeUid);

	CleanupStack::PopAndDestroy(2); // exportContacts, contactsDb
	}

// ----------------------------------------------------
// CContactsModelDocument::CreateAppUiL()
// constructs CContactsModelAppUi
// ----------------------------------------------------
//
CEikAppUi* CContactsModelDocument::CreateAppUiL()
    {
    return new (ELeave) CContactsModelAppUi;
    }

// End of File
