// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the definition of member functions of the CProducer class.
//



/**
 @file 
*/

#include "producer.h"

/**
Performs the two-phase construction of an object of the CProducer class.
@param aConsole The console object.
@param aTokens A pointer to the CQueue object containing the tokens' queue.
@return A CProducer object.
*/
CProducer* CProducer::NewL(CConsoleBase* aConsole,CQueue* aTokens)
	{
	CProducer* self = new (ELeave)CProducer;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole,aTokens);
	CleanupStack::Pop(self);
	return self;
	}

/**
The second phase constructor of the CProducer class.
It creates the following member objects of the class:
- iPeriodicProd
- iProdThread
The call back function of the iPeriodicProd object - InsertFunction() invokes the iProdThread every second.
@param aConsole The console object.
@param aTokens A pointer to the CQueue object containing the tokens' queue.
@see CPeriodic::Start().
@see RThread::Create().
*/
void CProducer::ConstructL(CConsoleBase* aConsole,CQueue* aTokens)
	{
	iConsole = aConsole;
	iQueue = aTokens;

	// Create an object of the CPeriodic class.
	iPeriodicProd = CPeriodic::NewL(CActive::EPriorityUserInput);

	// Create the producer thread.
	_LIT(KProducer,"ProducerThread");
	User::LeaveIfError(iProdThread.Create(KProducer,ProdThreadFunc,KDefaultStackSize,KMinHeapSize,256*KMinHeapSize,iQueue,EOwnerThread));
	iProdThread.SetPriority(EPriorityMore);

	// The producer thread invokes the InsertFunction() function every two seconds.
	// Associate InsertFunction() as the call back function for the iPeriodicCons object.
	iPeriodicProd->Start(0,3000000,TCallBack(InsertFunction,this));
	}

/**
Constructor.
*/
CProducer::CProducer()
	{
	}

/**
Displays the contents of the queue.
*/
void CProducer::Display()
	{
	RArray<TInt> array;
	// Initialize the array.
	// The array reflects the contents of the CQueue queue.
	iQueue->GetTokens(array);
	// Print the array.
	iConsole->Printf(KNewLine);
	_LIT(KTextQueue,"Queue Contents:\n[  ");
	iConsole->Printf(KTextQueue);

	_LIT(KTextIntFormat,"%d  ");

	for(TInt ix = 0 ; ix < array.Count() ; ix++)
		{
		iConsole->Printf(KTextIntFormat,array[ix]);
		}

	_LIT(KTextQueueEnd,"]");
	iConsole->Printf(KTextQueueEnd);
	iConsole->Printf(KNewLine);
	iConsole->Printf(KNewLine);
	array.Close();
	}

/**
The producer thread function.
It inserts a token into the queue.
@param aPtr A pointer to the arguments passed to the thread function.
*/
TInt CProducer::ProdThreadFunc(TAny* aPtr)
	{
	// The while loop mechanism ensures that the thread performs the CQueue::Insert() operation each time RThread::Resume() is invoked on the thread handle.
	while(true)
		{
		CQueue* ptr = (CQueue*)aPtr;
		__ASSERT_ALWAYS(ptr,User::Panic(KTxtPanic,-1));
		// Insert a token into the queue.
		ptr->Insert();
		// Suspend the thread.
		RThread().Suspend();
		}
	return KErrNone;
	}

/**
Resumes the execution of the producer thread.
*/
void CProducer::ResumeProd()
	{
	// Call the RThread::Resume() function on the producer thread.
	// Control goes to the CProducer::ProdThreadFunc() function after this statement.
	iProdThread.Resume();
	}

/**
Produce a token.
*/
void CProducer::Produce()
	{
	// Call the CQueue::Insert() function.
	iQueue->Insert();
	}

/**
Destructor.
*/
CProducer::~CProducer()
	{
	iProdThread.Kill(KErrCancel);
	delete iPeriodicProd;
	}


/**
The call back function associated with the CPeriodic object of the CProducer class.
@see CProducer::iPeriodicProd.
@see CPeriodic.
*/
TInt CProducer::InsertFunction(TAny* aPtr)
	{
	CProducer* ptr = static_cast<CProducer*> (aPtr);
	__ASSERT_ALWAYS(ptr,User::Panic(KTxtPanic,-1));
	// Invoke the RThread::Resume() function on the producer thread repeatedly.
	ptr->ResumeProd();
	return KErrNone;
	}
