// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains the definition of member functions of the CQueue class.
//



/**
 @file 
*/

#include "queue.h"

/**
Performs the two-phase construction of an object of the CQueue class.
@return A CQueue object.
*/
CQueue* CQueue::NewL()
	{
	CQueue* self = new (ELeave)CQueue;
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

/**
The second phase constructor of the CQueue class.
@see RCondVar::CreateLocal().
@see RMutex::CreateLocal().
*/
void CQueue::ConstructL()
	{
	// Create the local RMutex variable.
	User::LeaveIfError(iMutex.CreateLocal());
	// Create the local RCondVar variable.
	User::LeaveIfError(iCondVar.CreateLocal());
	}

/**
Constructor.
*/
CQueue::CQueue()
	{
	}

/**
Inserts a token into the queue.
@see RCondVar::Signal().
*/
void CQueue::Insert()
	{
	// Wait for the mutex.
	// The mutex is required to ensure that only one thread updates iArray at a time.
	iMutex.Wait();
	// Insertion operation.
	// Insert some random number as a token into the queue.
	iArray.Append(Math::Random()%10);
	// Signal the local condition variable to indicate that queue has new data.
	iCondVar.Signal();
	// release the mutex so that the other thread has access to iArray.
	iMutex.Signal();
	// RCondVar::Broadcast() can be used as an alternative to the RCondVar::Signal() function.
	// It is more relevant to use Broadcast() when more than two threads are blocking on single condition variable.
	// The following code block illustrates the use of the Broadcast() function.
	//
	//if(!IsEmpty())
	//	{
	//	iCondVar.Broadcast();
	//	iMutex.Signal();
	//	}
	//
	}

/**
Removes a token from the queue.
@see RCondVar::Wait().
*/
void CQueue::Remove()
	{
	// Ensures that mutex is held by the current thread.
	iMutex.Wait();
	// Test to see whether there is anything in the queue before attempting to remove an item.
	// If the queue is empty wait for a signal on the Condition Variable before proceeding
	while(!IsEmpty())
		{
		// The following call blocks this thread and releases the mutex so that the other thread can update the shared data.
		// When the other thread signals iCondVar and releases the mutex this thread continues.  
		iCondVar.Wait(iMutex);
		// If there were more than one consumer waiting for the same signal
		// it would be essential to test IsEmpty() again because another thread, or threads,
		// might have got the mutex before this one and emptied the array.
		// For this reason if is good practise to use while( IsEmpty() ) instead of if( IsEmpty() ).
		}
	// Deletion operation.
	iArray.Remove(0);
	// Release the mutex.
	iMutex.Signal();
	}

/**
Checks if the queue has at least one element.
@return ETrue If the number of elements in the queue is greater than one, else EFalse.
*/
TBool CQueue::IsEmpty()
	{
	if(iArray.Count() > 0)
		{
		return ETrue;
		}
	return EFalse;
	}

/**
Constructs an array of elements present in the CQueue::iArray queue.
@param aArray A reference to the array to be constructed.
*/
void CQueue::GetTokens(RArray<TInt>& aArray)
	{
	for(TInt ix = 0; ix < iArray.Count(); ix++)
		{
		aArray.Append(iArray[ix]);
		}
	}

/**
Destructor.
*/
CQueue::~CQueue()
	{
	iMutex.Close();
	iCondVar.Close();
	}
