// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

#include "FeatMngrExample.h"
#include <e32base.h>
#include <e32std.h>
#include <e32def.h>
#include <e32cmn.h>

//String literals
_LIT(KNewline, "\n");

CFeatMngrExample* CFeatMngrExample::NewL(TInt aPriority)
	{
	CFeatMngrExample* self=new(ELeave)CFeatMngrExample(aPriority);
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}

void CFeatMngrExample::ConstructL()
	{
	//Create console
	_LIT(KTextConsoleTitle, "FeatureManager");
	iConsole = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen));
	_LIT(KWelcome, "Welcome to the Feature Management Example\n");
	iConsole->Printf(KWelcome);
	
	//Create another process to receive notification. 
	_LIT(KProcessName, "FeatureChecker.exe");
	User::LeaveIfError(iProcess.Create(KProcessName, KNullDesC));
	iProcess.Resume();
	
	User::LeaveIfError(iFeatControl.Open());
	}

CFeatMngrExample::CFeatMngrExample(TInt aPriority):CActive( aPriority )
	{	
	CActiveScheduler::Add(this);
	}

/**
Destructor
*/
CFeatMngrExample::~CFeatMngrExample()
	{
	Cancel();
	delete iConsole;
	iFeatControl.Close();
	iProcess.Close();
	}

void CFeatMngrExample::RunL()
	{
	TRAPD(err,ProcessKeyPressL(TChar(iConsole->KeyCode())));
	if(err != KErrNone)
		{
		iConsole->Printf(_L("Failed. Error: %d\r\n"),err);
		RequestCharacter();
		}	
	}

/**
 * This function is called after the user has made a selection from the menu options.
 * Depending on the user's choice, the appropriate function is called.
 */
void CFeatMngrExample::ProcessKeyPressL(TChar aChar)
    {
	switch(aChar)
		{
		case '1':
			{
			ListFeaturesL();
			break;
			}
		case '2':
			{
			AddL();
			break;
			}
		case '3':
			{
			UpdateL();
			break;
			}
		case '4':
			{
			EnableFeatureL();
			break;
			}
		case '5':
			{
			DeleteL();
			break;
			}
		case '6':
			{
			CheckFeatureSupportL();
			break;
			}
		default:
			CActiveScheduler::Stop();
			break;
		}
    }

/**
Gets the UID of a feature added by the user. A limited range of UIDs is subscribed 
to by the FeatureChecker process. The UID should be in that range.
*/
void CFeatMngrExample::AddL()
	{
	_LIT(KFeat, "Enter the UID of the feature to be added.\n");
	_LIT(KSelectUIDL1, "Select any UID, but in order to receive notification from featurechecker.exe,\n");
	_LIT(KSelectUIDL2,"choose a UID from the range 12000000 to 1200001A.\n");
	_LIT(KSelectUIDL3, "Notification takes the form of an info message displayed at the top right\n");
	_LIT(KSelectUIDL4, "corner of the screen.\n");
	iConsole->Printf(KFeat);
	iConsole->Printf(KSelectUIDL1);
	iConsole->Printf(KSelectUIDL2);
	iConsole->Printf(KSelectUIDL3);
	iConsole->Printf(KSelectUIDL4);
	TUid uidFeature;
	uidFeature.iUid = GetUserInput(EFalse);
		
	TBitFlags32 flags(0);
	//Set the feature to be upgradable and modifiable.
    flags.Set( EFeatureSupported );
    flags.Set( EFeatureModifiable );
	
	_LIT(KSetFeaturedata, "\nEnter an integer value to set the feature data\n");
	iConsole->Printf(KSetFeaturedata);
	TUint32 fdata = GetUserInput(ETrue);
	iConsole->Printf(KNewline);
	TFeatureEntry featureEntry(uidFeature ,flags, fdata); 
	TInt featadded = User::LeaveIfError(iFeatControl.AddFeature(featureEntry));
	if(featadded == KErrNone)
		{
		_LIT(KFeatureAdded, "A new feature is added.\n");
		iConsole->Printf(KFeatureAdded);
		}
	RequestCharacter();
	}

/**
Gets the UID of the feature which needs to be deleted and deletes it. This causes the corresponding 
notification to be received by the featurechecker process. 
*/
void CFeatMngrExample::DeleteL()
	{
	_LIT(KUIDtoDelete, "Enter the UID of the feature to delete\n");
	iConsole->Printf(KUIDtoDelete);
	TUid uidFeature;
	uidFeature.iUid = GetUserInput(EFalse);
	iConsole->Printf(KNewline);
	TInt featdeleted = User::LeaveIfError(iFeatControl.DeleteFeature(uidFeature));
	if(featdeleted == KErrNone)
		{
		_LIT(KFeatDeleted, "The feature is deleted.\n");
		iConsole->Printf(KFeatDeleted);
		}
	RequestCharacter();
	}

/**
Gets the UID of the feature whose feature flag needs to be updated. This causes the corresponding 
notification to be received by the featurechecker process. 
*/
void CFeatMngrExample::EnableFeatureL()
	{
	_LIT(KUIDtoEnable, "Enter the UID of the feature to enable/disable\n");
	iConsole->Printf(KUIDtoEnable);
	TUid uidFeature;
	uidFeature.iUid = GetUserInput(EFalse);
	
	_LIT(KEnable, "\nPress e/d to enable/disable it\n");
	iConsole->Printf(KEnable);
	TChar ch = iConsole->Getch();
	if(ch == 'e')
		{
		User::LeaveIfError(iFeatControl.EnableFeature(uidFeature));
		}
	else if(ch == 'd')
		{
		User::LeaveIfError(iFeatControl.DisableFeature(uidFeature));
		}
	RequestCharacter();
	}

/**
Checks if the given feature is supported in the current environment. If it is not supported 
it prints an error message, depending on the value returned by RFeatureControl::FeatureSupported(). 
*/
void CFeatMngrExample::CheckFeatureSupportL()
	{
	_LIT(KPrintUIDtoCheck, "Enter the UID to check if it is supported\n");
	iConsole->Printf(KPrintUIDtoCheck);
	
	TUid uidFeature;
	uidFeature.iUid = GetUserInput(EFalse);
	iConsole->Printf(KNewline);
	User::LeaveIfError(iFeatControl.FeatureSupported(uidFeature));
	
	TInt err = iFeatControl.FeatureSupported(uidFeature);
	if(err == KFeatureSupported)
		{
		_LIT(KFeatSupported, "\nThis feature is supported\n");
		iConsole->Printf(KFeatSupported);
		}
	else if(err == KFeatureUnsupported)
		{
		_LIT(KFeatNotsupported, "\nThis feature is not supported\n");
		iConsole->Printf(KFeatNotsupported);
		}
	else if(err == KErrNotFound)
		{
		_LIT(KFeatNotFound, "\nThis feature is not found\n");
		iConsole->Printf(KFeatNotFound);
		}
	else
		{
		_LIT(KErr, "Failed : %d");
		iConsole->Printf(KErr, err);
		}
	
	RequestCharacter();
	}

/**
Gets the list of all the current features.
*/
void CFeatMngrExample::ListFeaturesL()
	{
	RFeatureUidArray supportedFeatures;
	CleanupClosePushL(supportedFeatures);
	
	iFeatControl.ListSupportedFeatures(supportedFeatures);
	TInt featureCount = supportedFeatures.Count();
	_LIT(KFeatureCount, "The number of features present is : %d \n");
	iConsole->Printf(KFeatureCount, featureCount);
	for(int j=0; j< featureCount; j++)
		{
		_LIT(KFeatureUID, "Feature UID%d is %x \n");
		iConsole->Printf(KFeatureUID, (j+1), supportedFeatures[j]);
		//list 12 entries at a time, to ensure visibility in the console.
		if((j+1)%12 == 0)
			{
			_LIT(KPress, "Press any key to continue\n");
			iConsole->Printf(KPress);
			iConsole->Getch();
			}
		}
	CleanupStack::PopAndDestroy(&supportedFeatures);
	RequestCharacter();	
	}

/**
Takes input from the console and returns a TInt64 value. It resembles scanf. It takes a boolean 
value as a parameter which identifies whether the data to be scanned is decimal or hexadecimal. 

@param aDecimal When the user inputs a decimal number this is set to true and when the input is 
hexadecimal, it is false. 
*/
TInt64 CFeatMngrExample::GetUserInput(TBool aDecimal)
	{
	TBufC<50> buf;
	TPtr ptr(buf.Des());
	TKeyCode ch = iConsole->Getch();
	while(ch != EKeyEnter)
		{
		_LIT(KChar, "%c");
		iConsole->Printf(KChar,ch);
		if(ch!= EKeyBackspace)
			{
			ptr.Append(ch);
			}
		ch=iConsole->Getch();   			
		}
	TInt64 value;
	TInt err;
	if(aDecimal)
		{
		err = TLex(ptr).Val(value);
		}
	else
		{
		err = TLex(ptr).Val(value, EHex);
		}
	if (err == KErrNone)
		{
		return value;
		}
	else 
		{
		return KErrArgument; // Error condition - could not parse input.
		}
	}

/**
Updates the feature data.
*/
void CFeatMngrExample::UpdateL()
	{
	_LIT(KFeatUpdate, "Enter the UID of the feature to be updated\n");
	iConsole->Printf(KFeatUpdate);
	TUid uidFeature;
	uidFeature.iUid = GetUserInput(EFalse);
			
	_LIT(KSetFeaturedata, "\nEnter a number to set as the feature data\n");
	iConsole->Printf(KSetFeaturedata);
	TUint32 fdata = GetUserInput(ETrue);
	iConsole->Printf(KNewline);
	//Updates any existing feature	
	User::LeaveIfError(iFeatControl.SetFeature(uidFeature , fdata));
	RequestCharacter();
	}


void CFeatMngrExample::DoCancel()
	{
	iConsole->ReadCancel();
	}


void CFeatMngrExample::RequestCharacter()
	{
	_LIT(KMenuL1,"\nSelect one option from the following:\n");
	_LIT(KMenuL2,"1.List the features currently present\n");
	_LIT(KMenuL3,"2.Add a feature\n");
	_LIT(KMenuL4,"3.Update a feature\n");
	_LIT(KMenuL5,"4.Enable/Disable a feature\n");
	_LIT(KMenuL6,"5.Delete a feature\n");
	_LIT(KMenuL7,"6.Check feature support\n");
	_LIT(KMenuL8,"Press any other key to exit\n");
	
	iConsole->Printf(KMenuL1);
	iConsole->Printf(KMenuL2);
	iConsole->Printf(KMenuL3);
	iConsole->Printf(KMenuL4);
	iConsole->Printf(KMenuL5);
	iConsole->Printf(KMenuL6);
	iConsole->Printf(KMenuL7);
	iConsole->Printf(KMenuL8);
	iConsole->Printf(KNewline);
	// Read the key press. RunL() will be called when iStatus completes.
    iConsole->Read(iStatus);
    if(!IsActive())
    	{
    	SetActive();
    	}
	}

static void MainL()
	{
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);
	
	CFeatMngrExample* featMngr = CFeatMngrExample::NewL();
	CleanupStack::PushL(featMngr);
	featMngr->RequestCharacter();
	CActiveScheduler::Start();
	
	CleanupStack::PopAndDestroy(featMngr);
	CleanupStack::PopAndDestroy(scheduler);
	}

extern TInt E32Main()
	{
	// Create cleanup stack
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	// Run application code inside a TRAP harness.
	TRAPD(mainError, MainL());
	if(mainError != KErrNone)
		{
		_LIT(KUserPanic,"Failed to complete");	
		User::Panic(KUserPanic, mainError);
		}
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
