// Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//


#include "processclient.h"
#include "processclientserver.h"
#include<e32debug.h>

#define KMaxServerNameLength 256

static const TUid KUidProcessServer = {0x10283037}; // Process Server UID
const TInt KCreateSessionRetryCount = 2; 					//CreateSession retry count
const TInt KServerDefaultMessageSlots = 2; 			//server async Message Slots

/**
	Starts the server process. Simultaneous launching
	of two such processes should be detected when the second one attempts to
	create the server object with its globally unique name.
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/	
static TInt StartServer()
	{
	const TUidType serverUid(KNullUid,KNullUid,KUidProcessServer);

	RProcess server;
	TInt r=server.Create(KProcessServerServerImg,KNullDesC,serverUid);
	if (r!=KErrNone)
		return r;
	TRequestStatus stat;
	server.Rendezvous(stat);

	if (stat!=KRequestPending)
		server.Kill(KErrCouldNotConnect);		// abort startup
	else
		server.Resume();	// logon OK - start the server
	
	User::WaitForRequest(stat);		// wait for start or death
	// The server exit type may be a panic value and a panic value can be zero, which is the same 
	// value as KErrNone. So, the exit type is tested for a panic before being returned.
	if (server.ExitType()==EExitPanic)
		{
		r = KErrServerTerminated;
		}
	else
		{
		r = stat.Int();
		}
	server.Close();
	return r;
	}
/**
	Connects to the server, attempting to start it if necessary.
	@return The error value.
*/	
EXPORT_C TInt RProcessClient::Connect()
	{
	TInt retry=KCreateSessionRetryCount; //number of CreateSession retries
	FOREVER
		{
		// try to create a session with the server which has KServerDefaultMessageSlots async message slots.
		TInt r=CreateSession(KProcessServerName,
							 TVersion(KProcessServerVersion,
									  KProcessServerMinorVersionNumber,
									  KProcessServerBuildVersionNumber),
							 KServerDefaultMessageSlots);							 
		if (r!=KErrNotFound && r!=KErrServerTerminated)
			return r;
		if (--retry==0)
			return r;
		r=StartServer();	
		if (r!=KErrNone && r!=KErrAlreadyExists)
			return r;
		// 1. r=KErrNone means server starts up successfully.
		// 2. r=KErrAlreadyExists means a duplicate server was trying to start up.
		// 		Note: if there are other functions called before CServer2::StartL() during construction,
		//       	  other errors might happen when the duplicate server is trying to start.
		//            Therefore, we recommend CServer2::StartL() is the first function in server construction
		//            @see CProcessServer::ConstructL()
		//
		// NOTE: If users would like to retry start up server after other scenarios happened, e.g. panic,then go 
		//		 through the following steps:
		// 		  1. Increase the value of KCreateSessionRetryCount (only start up server once in this example)
		//		  2. Need another variable, e.g. TExitType, together with error code returned in StartServer()
		//			to distinguish these scenarios
		//		  3. Modify the third if statement to stop exit when the required scenarios happens.
		//           
		}
	}
/**
	Opens the device driver's logical channel.
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/		
EXPORT_C TInt RProcessClient::OpenDriver()
	{
	return SendReceive(EOpenDriver);
	}
/**
	Sends data to device (asynchronous request).
	@param aBuf A descriptor which contains the data to be sent.
	@param aStatus A TRequestStatus reference.
*/	
EXPORT_C void RProcessClient::Send(const TDesC8& aBuf, TRequestStatus& aStatus)
	{
	SendReceive(EDummyLDDSendData, TIpcArgs(&aBuf), aStatus);
	}
/**
	Cancels sending request.
*/	
EXPORT_C void RProcessClient::SendCancel()
	{
	SendReceive(EDummyLDDSendDataCancel);
	}
/**
	Unloads the device driver.
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/	
EXPORT_C TInt RProcessClient::UnloadDeviceDriver()
	{
	return SendReceive(EUnloadDeviceDriver);
	}
/**
	Loads the device driver.
	@return KErrNone if successful, otherwise one of the other system wide error codes.
*/	
EXPORT_C TInt RProcessClient::LoadDeviceDriver()
	{
	return SendReceive(ELoadDeviceDriver);
	}

//EOF
