// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Demonstrates the speculative publishing pattern of a byte-array property
//



/**
 @file
*/

#include "publish.h"

LOCAL_D CConsoleBase* console;

/**
Publishes the byte-array property
@param aProperty The property to be published
@param aBuf The descriptor containing the byte-array values
*/
void PublishL(RProperty& aProperty,TDes16& aBuf)
	{
	TInt bufLength = aBuf.Length();
	console->Printf(KTxtPublish);

	// Publish the property and handle the value returned appropriately
	// Ignore the KErrNotFound error
	TInt err = aProperty.Set(aBuf);
	switch(err)
		{
		// No error, property is defined
		case KErrNone:
			{
			// Print array elements
			console->Printf(KTxtArray);
			for(TInt ix = 0; ix < bufLength; ix++)
				{
				console->Printf(KTxtArrayElement,aBuf[ix]);
				}
			console->Printf(KTxtNewLine);
			}
			break;
		// Property is not defined, ignore this error
		case KErrNotFound:
			{
			console->Printf(KTxtNotFound);
			}
			break;
		// Leave if any other errors are returned
		default:
			{
			User::LeaveIfError(err);
			}
		}
	}

void DoExampleL()
	{
	console->Printf(KTxtSpecPublish);

	RProperty myProperty;
	console->Printf(KTxtCreateHandle);
	// Create a handle to the property
	User::LeaveIfError(myProperty.Attach(KMyPropertyCat,KMyPropertyName,EOwnerThread));

	// Byte-Array to be published
	TUint16 array[KArraySize] = {10,20,30,40};

	// Descriptor that the array contents are copied into
	TBuf16<KBufSize> buf;

	// Copy array contents into the descriptor
	buf.Copy(&array[0],sizeof(array));
	buf.SetLength(KArraySize);

	// Publish the property
	PublishL(myProperty,buf);

	// Change the array contents
	array[0] = 0x00ab;
	array[1] = 0x00bc;
	array[2] = 0x00cd;
	array[3] = 0x00de;
	array[4] = 0x00ef;
	array[5] = 0x00fa;

	// Copy array contents into the descriptor
	buf.Copy(&array[0],sizeof(array));
	buf.SetLength(KArraySize);
	console->Printf(KTxtEnter);

	// Wait for a key press
	while(console->Getch()!= EKeyEnter)
		{
		console->Printf(KTxtEnter);
		}

	// Publish the property
	PublishL(myProperty,buf);

	// Indicate the publisher will not publish any more values
	buf.Copy(KStop);
	console->Printf(KTxtEnter);

	// Wait for a key press
	while(console->Getch()!= EKeyEnter)
		{
		console->Printf(KTxtEnter);
		}

	// Publish the property
	PublishL(myProperty,buf);

	// Close the handle
	myProperty.Close();
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();

	TRAPD(createError, console = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen)));
	if (createError)
		return createError;

	TRAPD(mainError, DoExampleL());
	if (mainError)
		console->Printf(KTextFailed, mainError);
	console->Printf(KTextPressAnyKey);
	console->Getch();

	delete console;
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
