// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Demonstrates the pure event pattern of subscribing to an integer property
//



/**
 @file
*/

#include "subscribepe.h"

LOCAL_D CConsoleBase* console;

/**
Constructor
*/
CIntPropertyWatch::CIntPropertyWatch():CActive(EPriority)
	{
	}

/**
Adds the active object to the active scheduler and creates a handle to the integer property
@param aConsole The console interface
*/
void CIntPropertyWatch::ConstructL(CConsoleBase* aConsole)
	{
	User::LeaveIfError(iProperty.Attach(KMyPropertyCat,KMyPropertyName));
	iConsole = aConsole;
	CActiveScheduler::Add(this);
	}

/**
Constructs a CIntPropertyWatch object
@param aConsole The console interface
@return A CIntPropertyWatch object
*/
CIntPropertyWatch* CIntPropertyWatch::NewL(CConsoleBase* aConsole)
	{
	CIntPropertyWatch* self = new (ELeave) CIntPropertyWatch;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole);
	CleanupStack::Pop(self);
	return self;
	}

/**
Destructor
*/
CIntPropertyWatch::~CIntPropertyWatch()
	{
	iProperty.Close();
	Cancel();
	}

/**
Cancels any outstanding request
*/
void CIntPropertyWatch::DoCancel()
	{
	iProperty.Cancel();
	}

/**
Handles the subscription to the property
Subscribes to the property as long as there is no change in the value of the property
*/
void CIntPropertyWatch::RunL()
	{
	// Receive a notification when a value is published
	// The RProperty::Get() function call can be omitted as the published value is of no importance in the pure event pattern
	TInt value;
	TInt err = iProperty.Get(value);
	if (err == KErrNotFound)
		{
		// No need to re-subscribe to the property as it has been deleted by the publisher
		CActiveScheduler::Stop();
		}
	else if (err == KErrNone)
		{
		// Print the value of the property
		PrintProperty(value);
		// Re-subscribe to the property
		IssueRequest();
		}
	else
		{
		// Leave the function in case of any other error
		CActiveScheduler::Stop();
		User::Leave(err);
		}
	}

/**
Prints a property value
@param aValue The changed value of the property
*/
void CIntPropertyWatch::PrintProperty(TInt aValue)
	{
	iConsole->Printf(KTxtValChange,aValue);
	}

/**
Gets the value of the property if it is already defined
It also issues an outstanding request by re-subscribing to the property
*/
void CIntPropertyWatch::WatchL()
	{
	TInt value;
	// Checks if the property exists
	TInt res = iProperty.Get(value);
	if (res == KErrNotFound)
		{
		// Property not defined
		User::Leave(res);
		}
	// Prints the value of the property
	// This is only to illustrate that the value of the property is set by the publisher
	// This value has no significance as the pure event distribution pattern of publish and subscribe is being followed
	PrintProperty(value);
	IssueRequest();
	}

/**
Issues an outstanding request by subscribing to the property
*/
void CIntPropertyWatch::IssueRequest()
	{
	iProperty.Subscribe(iStatus);
	SetActive();
	}

void DoExampleL()
	{
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);

	console->Printf(KTxtPESub);

	// Create the integer property watch active object
	CIntPropertyWatch* obj = CIntPropertyWatch::NewL(console);
	CleanupStack::PushL(obj);

	// Subscribe to the property and start the scheduler
	obj->WatchL();
	CActiveScheduler::Start();

	CleanupStack::PopAndDestroy(obj);
	CleanupStack::PopAndDestroy(scheduler);
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();

	TRAPD(createError, console = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen)));
	if (createError)
		return createError;

	TRAPD(mainError, DoExampleL());
	if (mainError)
		console->Printf(KTextFailed, mainError);
	console->Printf(KTextPressAnyKey);
	console->Getch();

	delete console;
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
