// sigusr1.c
//
// Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:This example demonstrates asynchronous signal handling.
//


#include <stdio.h>
#include <stdlib.h>
#include <signal.h> //The header for signal functionality.
#include <unistd.h> //The header used for getpid().
#include <e32def.h>
#include <spawn.h>
#include <string.h>

int gch;            // The variable holding a character value, input by the user.
FILE *gfp = NULL;   // The file descriptor variable holding pointer to asyncFile.txt.
pid_t chldProcsID;  // The variable for holding the sigusr2 PID.
int gVal = TRUE;    // Variable used as a boolean value.
/**
 Custom handler function for SIGUSR1
 */
void SIGUSR1_handler(int signum)
    {
    int ret;
    const int delay1 = 1;  //Time delay of 1 sec.
    if (signum == SIGUSR1)
        {
        gVal = FALSE;
        printf("\nReceived the SIGUSR1 signal from the sigusr2 process\n");
        printf("Starting to read from the file\n");
        while((gch = fgetc(gfp))!='\n')
            {
            putchar(gch);  //Writing to stdout.
            }
        printf("\n");  
        sleep (delay1);     
        
        //Sending SIGUSR2 signal to sigusr2 process once the file content is written to the console.
        printf("Reading from the file now completed, sending SIGUSR2 to the sigusr2 process\n");
        
        //Raising SIGUSR2 signal using kill command.
        ret = kill(chldProcsID,SIGUSR2);
        if(ret)
            {
            printf("kill() failed to send signal, errno=%d\n",errno);
            }
        }
    }

/**
 Entry point for the asyncSignal process.
 The example demonstrates asynchronous signal handling.
 The main function does the following:
 1) obtains the PID of the sigusr1 process
 2) assigns a custom handler for the SIGUSR1 signal
 3) opens a file in read and write mode
 4) writes the file and spawns the sigusr2 process 
 5) waits for a SIGUSR1 signal from the sigusr2 process
 6) On reception of a SIGUSR1 signal, it reads the open file and displays its contents to the console.
 7) Once the file is read it sends a SIGUSR2 signal to the sigusr2 process, closes the open file descriptor and exits. 
 */

int main()
    {
    int ret;
    const int delay = 1;                                        //Delay of 1 sec provided to wait for signal.
    const int size = 50;                                        //No.of characters allowed in the filename.
    const int index = 3;                                        //No.of elements in the array.
    
    char *filename = "C:\\asyncFile.txt" ;                      //Filename of the file that holds names input by the user.
    char *childProcess = "Z:\\sys\\bin\\sigusr2.exe";           //Filename of sigusr2 process.
    char **argv = (char**)malloc(index*sizeof(char*));          //Variable holding values to be passed to the sigusr2 process.
    argv[0] = (char*)malloc(size*sizeof(char));
    argv[1] = (char*)malloc(index*sizeof(char));
    argv[2] = NULL;

    
    strcpy(argv[0], "z:\\sys\\bin\\sigusr2.exe");               //argv[0] holding name of the sigusr2 process.
    sprintf(argv[1],"%d",getpid());                             //argv[1] holding PID of sigusr1 process.
    
    //Setup the custom handler for SIGUSR1.
    signal(SIGUSR1,SIGUSR1_handler);
    
    printf("*****************************************************************\n");
    printf("*   Welcome to the asynchronous signal handling demonstration   *\n");
    printf("*****************************************************************\n");
    
    printf("* This example demonstrates asynchronous signal handling using SIGUSR1 and SIGUSR2 signals.\n");
    printf("* The example consists of two processes, the sigusr1 process and the sigusr2 process.\n");
    printf("* The sigusr1 process handles the SIGUSR1 signal and sends a SIGUSR2 signal to the sigusr2 process.\n");
    printf("* The sigusr2 process handles the SIGUSR2 signal and sends a SIGUSR1 signal to the sigusr1 process.\n");
    printf("* The sigusr1 process opens a file and writes some text in it.\n");
    printf("* The sigusr1 process then spawns the sigusr2 process and waits until SIGUSR1 is received from it.\n");
    printf("* Once the sigusr1 process obtains a SIGUSR1 signal, it starts reading from the file and displays its\ncontent on the console.\n");
    printf("* When all the file content is written, the sigusr1 process sends a SIGUSR2 signal to the sigusr2 process and prepares to exit.\n");
    printf("* On other side the sigusr2 process keeps waiting for the SIGUSR2 signal from the sigusr1 process.\n");
    printf("* On reception of the SIGUSR2 signal, the sigusr2 process prepares to exit.\n");
    
    printf("\nPress the Enter key to continue\n");
    getchar();
    
    printf("****************** In the sigusr1 process ********************\n"); 
  
         
    printf("\nOpening a file for read and write\n");
    if((gfp = fopen(filename,"w+"))!=NULL)
        {
        printf("Successfully opened the file\n");
        fprintf(gfp, "%s", "An asynchronous signal handling example using the SIGUSR1 and SIGUSR2 signals\n");
        rewind(gfp);
        printf("Writing to the file completed, preparing to start reading from the file\n");
        
        printf("Press the Enter key to spawn the sigusr2 process\n");
        getchar();
        
        //Spawning sigusr2 process.
        ret = posix_spawn(&chldProcsID,childProcess,NULL,NULL,argv,(char**)NULL);
        if(ret != 0)     
            {         
            printf("\n*** failure posix_spawn ***\n");        
            return EXIT_FAILURE;     
            }
        }
    
    printf("Waiting until SIGUSR1 is obtained from the sigusr2 process");
    while(gVal)
        {
        printf(".");
        sleep(delay);
        }
    
    if(!fclose(gfp))
        {
        remove(filename);
        gfp=NULL;
        printf("The file was closed successfully.\n");
        }
    else
        {
        printf("File close failed.\n");
        }
    
    printf("Press 'e'+Enter to exit from the sigusr1 process\n");
    
    while((gch=getchar())!= 'e')
        {
        if(gch == '\n')
            continue;
        else
            printf("The wrong option was selected, please try again!!!\n");
        }
    return EXIT_SUCCESS;
    }
