/*
 * Copyright  2008 Nokia Corporation.
 */

#ifndef DESCRIPTOREXEXAMPLES_H
#define DESCRIPTOREXEXAMPLES_H

#include <e32std.h>
#include <f32file.h>

//------------------------------------------------------------------------------
/**
* This class declares a protocol that examples use to render the results of
* example methods. The implementing class holds a buffer where text can be
* appended. GetViewBuffer() returns descriptor to internal buffer. UpdateView()
* renders the content to the destination device and may reset internal buffer.
*/
class MResultViewer
    {
    public:
        /**
         * Implementing method shall return a modifiable descriptor pointer
         * where text can be appended.
         */
        virtual TPtr GetViewBuffer()=0;

        /**
         * Implementing method shall render the content to output so that user
         * can view the results. As a side effect, the TPtr returned by method
         * GetViewBuffer() may come obsolete and should be queried again before
         * usage.
         */
        virtual void UpdateView()=0;
    };

//------------------------------------------------------------------------------
/**
* This class declares example methods that describe the usage of descriptors and
* related APIs.
*/
class CDescriptorExamples
    {
    public:

        /**
        * This constructor instantiates an object whose example methods are used
        * to demonstrate descriptors and related APIs.
        *
        * @param aViewer MResultViewer object where results are to be written.
        *                This doesn't take ownership of given viewer so it has
        *                to be deleted after this object is no more used.
        */
        CDescriptorExamples(MResultViewer *aViewer) : iViewer(aViewer) {};

        //----------------------------------------------------------------------
        // Methods below are implemented in DeclaringDescriptors.cpp

        /**
        * Examples in this method describes how to declare each descriptor type
        * as an automatic variable (object allocated to stack). Following issues
        * are covered:
        *
        * - how to declare each descriptor type to stack
        * - using code blocks to deallocate unneeded automatic variables
        */
        void ToStack();

        /**
        * Examples in this method describes how to allocate descriptors from
        * heap.
        */
        void ToHeapL();

        /**
        * This example method describes how to declare literals. Following
        * issues are covered:
        *
        * - declaring literal with macro _LIT
        * - declaring literal with macro _L
        * - declaring unicode characters in literals
        */
        void Literals();

        //----------------------------------------------------------------------
        // Methods below are implemented in ManipulatingDescriptors.cpp

        /**
        * This example demonstrates usage of non-modifying methods declared the
        * base class of all descriptor, TDesC.
        */
        void NonModifyingMethods();

        /**
        * This example demonstrates usage modifying methods declared the base
        * class of modifying descriptors, TDes.
        */
        void ModifyingMethodsL();

        /**
        * This example demonstrates how to convert string data encoded with one
        * character set to unicode and vice versa. Conversion of SMS data is
        * used as and example.
        */
        void CharacterConversionsL();

        /**
        * This example demonstrates how numbers in string format can be parsed
        * to binary representation.
        */
        void LexicalAnalysis();

        //----------------------------------------------------------------------
        // Methods below are implemented in OtherBuffers.cpp

        /**
        * The first example in this method demonstrates how circular buffer,
        * First In First Out (FIFO) data structure, is used manipulate character
        * data. Each item in the FIFO is instance of TText that is 16 bit
        * character code unicode build and 8 bit character in non-unicode build.
        * 
        * The second example in this method extends the usage to a bit more
        * complex type than a 8 or 16 bit number. An example class is used as an
        * item in structure.
        */
        void CircularBuffersL();

        /**
        * This method demonstrates how flat dynamic buffer (CBufFlat) is used to
        * store and alter binary data. 8-bit character data is used as an
        * example data.
        * 
        * Data in CBufFlat lies in one contiguous heap cell. CBufFlat extends
        * the heap cell used to store the contents automatically when so needed.
        * If heap cell can't extend a new cell is allocated, old data copied to
        * new cell and old heap cell deleted.
        *
        * Data is accessed and modified through base classes methods
        * CBufBase::Read(), CBufBase::Write(), CBufBase::InsertL() and
        * CBufBase::Delete(). Pointer access to data makes sence since data in
        * heap cell is continuous.
        */
        void FlatDynamicBuffersL();

        /** 
        * This example demonstrates how segmented dynamic buffers (CBufSeg) are
        * used to store and alter binary content. 8-bit character data is used
        * as an example data.
        *
        * CBufSeg splits contents to multiple heap cells having the same size.
        * These cells are called segments of the data. CBufSeg allocates new
        * segments automatically when data storage needs to be enlarged.
        * 
        * Data is accessed and modified through base classes methods
        * CBufBase::Read(), CBufBase::Write(), CBufBase::InsertL() and
        * CBufBase::Delete(). Pointer access is also possible but is difficult
        * since data isn't continuous (like in case of 
        */
        void SegmentedDynamicBuffersL();

        /**
        * This example method demonstrates how to declare and use package
        * buffers and package pointers.
        *
        * In typical cases the threads can't access to each others memory areas.
        * However, interthread communication provides a way to read and write to
        * other threads memory. The memory address given for reading or writing
        * must be a descriptor. The actual reading and writing is done by kernel
        * that will make sure that descriptor constraints are respected: buffer
        * is not read or written before its starting index or after end index.
        *
        * Package buffer capsulates any value object as an binary array that
        * makes it easy pass object or structure of objects to other thread.
        */
        void PackageBuffers();
        
                
        /**
        * Function demonstrates different usages of RBuf. 
        * It should be noted that the Create and Assign functions
        * leak memory if the memory for the RBuf isn't freed (i.e calling Close())
        * before calling them.
        * RBuf was introduced in Symbian OS 9 but was backported to Symbian OS 8.
        */
        void RBufDemonstrations();

    private: // attributes

        MResultViewer* iViewer; // not owned, just referred
    
    };

//------------------------------------------------------------------------------
/**
* This class is used to render data written through MResultViewer interface to a
* file.
*
* @see MResultViewer for general documentation about implemented interface.
*/
//PK:TODO:
#ifdef __SERIES60_3X__
_LIT( KDefaultLogFile, "c:\\Data\\descriptorEx.log.txt" );
#else
_LIT( KDefaultLogFile, "c:\\Nokia\\descriptorEx.log.txt" );
#endif

class CFileForwarder : public CBase, public MResultViewer
    {
    public:

        /**
         * Construct an instance of the MResultViewer that writes contents to
         * the file specified by logFileName parameter. The file is opened at
         * the construction time and kept open until object is destroyed.
         *
         * @param logFileName file name where contents are written.
         */
        void ConstructL(const TDesC &logFileName=KDefaultLogFile);

        /**
        * Returns a modifiable descriptor to the internal buffer.
        * @see MResultViewer::GetViewBuffer().
        */
        virtual TPtr GetViewBuffer();

        /**
        * Flushes the text to the file and resets the internal buffer.
        * @see MResultViewer::UpdateView().
        */
        virtual void UpdateView();

    private:

        /**
        * Destroys the object and closes the file opened at construction time.
        */
        virtual ~CFileForwarder();

    private:
        RFs          iFsSession;
        RFile        iFile;
        HBufC        *iBuf;
    };

#endif // DESCRIPTOREXEXAMPLES_H
