/*
 * Copyright  2008 Nokia Corporation.
 */

#ifndef STRING_RENDERER_H
#define STRING_RENDERER_H

#include <e32std.h>

// -----------------------------------------------------------------------------
/**
 * Rendering flags are used to specify how contents of objects are rendered. To
 * add specific handling, use bitwise-or to combine flags to each others.
 *
 * KRenderDefault renders the contenst of object in default manner. In case of
 *                descriptor it is rendered as string enclosed between '"'
 *                chars. In case of number it is rendered as is.
 *
 * KRenderContentAsBinary In case of descriptor, the data is rendered as numbers
 *                        instead of a string.
 *
 * KRenderCharacteristics If specified, the complex objects render also
 *                        characteristics in addition to content.
 */
const TInt KRenderDefault = 0x00;
const TInt KRenderContentAsBinary = 0x01;
const TInt KRenderCharacteristics = 0x02;

// -----------------------------------------------------------------------------
/**
 * Copy data from aSrc to the end of aDst.
 *
 * Descriptor api provides only method Copy() to copy narrow data to wide
 * descriptor. Appending is not possible so these functions can be used instead.
 * Overloaded 16 bit function is usefull where 8 bit and 16 bit calls are
 * templated or in macros.
 *
 * @param aSrc may be either 8 or 16 bit descriptor
 * @param aDst Unicode descriptor where character data is to be appended
 */
void Append( const TDesC8 &aSrc, TDes16 &aDst );
void Append( const TDesC16 &aSrc, TDes16 &aDst );

// -----------------------------------------------------------------------------
/**
 * Renders the contents of 8 or 16 bit descriptor as nuber array. The example
 * result: {4, 2, 2323, 12,2}
 */
void RenderAsNumbers( const TDesC8 &anArray, TDes16 &aOutput );
void RenderAsNumbers( const TDesC16 &anArray, TDes16 &aOutput );

// -----------------------------------------------------------------------------
/**
 * RenderObject() renders the contents of given object and possibly its
 *                characteristics. The format can be altered with aRenderFormat
 *                parameter.
 */
void RenderObject(const TDesC8 &aVariable,
                  TDes16 &aOutput,
                  TInt aRenderFormat=KRenderDefault);
void RenderObject(const TDes8 &aVariable,
                  TDes16 &aOutput,
                  TInt aRenderFormat=KRenderDefault);
void RenderObject(const TDesC16 &aVariable,
                  TDes16 &aOutput,
                  TInt aRenderFormat=KRenderDefault );
void RenderObject(const TDes16  &aVariable,
                  TDes16 &aOutput,
                  TInt aRenderFormat=KRenderDefault );
void RenderObject(const TInt &aVariable,
                  TDes16 &aOutput,
                  TInt aRenderFormat=KRenderDefault );

// -----------------------------------------------------------------------------
/**
 * Renders a header string to the output. Example:
 *
 *   TBuf<128> output;
 *   RenderHeader( _L("Hello"), output );
 *
 * renders string "\n--- Hello ---\n" to the output.
 */
void RenderHeader(const TDesC &aHeader, TDes &aOutput);

// -----------------------------------------------------------------------------
/**
 * This macro is used to render a descriptor variable name and its contents and
 * possibly the characteristics.
 * The macro usage below
 *
 *   TBuf<128> output;
 *   TBuf<6> data(_L("Hello"));
 *   RenderVariableFormatted( data, output, KRenderDefault );
 *
 * appends string
 *
 *   data="Hello"
 *
 * to output and macro below
 *
 *   RenderVariableFormatted( data, output, KRenderCharacteristics );
 *
 * appends string
 *
 *   data="Hello" (len=5, maxlen=6);
 */
#define RenderVariableFormatted(aVariableName, aOutput, aRenderFormat) \
    aOutput.Append(_L(#aVariableName));\
    aOutput.Append('=');\
    RenderObject( aVariableName, aOutput, aRenderFormat );\
    aOutput.Append('\n');\

// -----------------------------------------------------------------------------
_LIT( KForward, " -> " );
/**
 * This macro is used to render a declaration that does return a
 * result. The declaration and the results are then printed to output.
 * The example below
 *
 *   TBuf<128> output;
 *   RenderResultFormatted( 10 + 20 + 5, output, KRenderDefault );
 *
 * appends string
 *
 *   10 + 20 + 5 -> 25
 *
 * to output and code
 *
 *   TBuf<64> source(_L("abcdefghij"));
 *   RenderResultFormatted( source.Mid(2,4), output, KRenderCharacteristics );
 *
 * appends string
 *
 *   source.Mid(2,4) -> "cdef" (len=4)
 *
 * to output.
 *
 * @param aDeclaration Any declaration that compiles and evaluates to
 *                     any descriptor or TInt value.
 * @param aOutput TDes16 derived class where data is to be written.
 */
#define RenderResultFormatted(aDeclaration, aOutput, aRenderFormat) \
    aOutput.Append(_L(#aDeclaration));\
    aOutput.Append(KForward);\
    RenderObject( aDeclaration, aOutput, aRenderFormat );\
    aOutput.Append('\n');\

// -----------------------------------------------------------------------------
/**
 * This macro is used to execute a piece of code that modifies some
 * variable and then print the code and the results as is to the
 * output. In detail the macro does
 *
 * 1. execute aDeclaration as is
 * 2. prints aDeclaration as a string to the output
 * 3. prints aVariable to the output
 *
 * The example code:
 *
 *   TBuf<128> output;
 *   TBuf<10> data;
 *   ExecuteAndRenderVariableFormatted( data.Fill('x', 3), data, output,
 *                                      KRenderDefault );
 *
 * Executes the first parameter as
 *
 *      data.Fill('x', 3);
 *
 * and prints the same declaration as a string and the aVariable to the
 * output. The content of the output after this macro is:
 *
 *      data.Fill('x', 3) -> data="xxx"
 *
 * @param aDeclaration Any declaration that compiles (should modify aVariable)
 * @param aVariable A variable that is to be rendered after aDeclaration
 *                  is executed (have to be 8 or 16 bit descriptor or TInt)
 * @param aOutput TDes16 derived class where data is to be written.
 */
#define ExecuteAndRenderVariableFormatted(aDeclaration, \
                                          aVariable, \
                                          aOutput, \
                                          aRenderFormat) \
    aDeclaration;\
    aOutput.Append(_L(#aDeclaration));\
    aOutput.Append(KForward);\
    aOutput.Append(_L(#aVariable));\
    aOutput.Append('=');\
    RenderObject( aVariable, aOutput, aRenderFormat );\
    aOutput.Append('\n');\

#endif // STRING_RENDERER_H
