/*
 * Copyright  2008 Nokia Corporation.
 */

// INCLUDE FILES
#include <coemain.h>
#include <eikenv.h>
#include <gdi.h>
#include <txtrich.h>
#include <aknsbasicbackgroundcontrolcontext.h>

#include "DescriptorExContainer.h"


// -----------------------------------------------------------------------------
// Two phase constructor. Constructs container containing one non editable
// editor component.
// -----------------------------------------------------------------------------
void CDescriptorExContainer::ConstructL(const TRect& aRect)
    {
    CreateWindowL();

    // Create editable editor control
    iTextBox = new (ELeave) CEikEdwin();
    iTextBox->SetContainerWindowL( *this );
    iTextBox->ConstructL( CEikEdwin::ENoAutoSelection |
                          CEikEdwin::EDisplayOnly |
                          CEikEdwin::EReadOnly |
                          CEikEdwin::EAvkonEditor |
                          CEikEdwin::EInclusiveSizeFixed);
    iTextBox->CreateScrollBarFrameL()->SetScrollBarVisibilityL(
            CEikScrollBarFrame::EOff, CEikScrollBarFrame::EOn);
    
    iTextBox->UpdateScrollBarsL();
    
    // http://wiki.forum.nokia.com/index.php/TSS000578_-_How_to_ensure_that_the_scroll_bar_background_is_drawn_properly%3F
    // Skin support
    iSkinContext = CAknsBasicBackgroundControlContext::NewL(KAknsIIDQsnBgAreaMainAppsGrid,aRect,EFalse);         
    
    // Activate the window, which makes it ready to be drawn
    ActivateL();

    // Set the windows size
    // NOTE: To call this after ActivateL() is solution to get
    // scrollbarframe width in SizeChanged() method.
    SetRect(aRect);
    }

// -----------------------------------------------------------------------------
// Destroys all resources owned by this object.
// -----------------------------------------------------------------------------
CDescriptorExContainer::~CDescriptorExContainer()
    {
    delete iTextBox;
    delete iSkinContext;
    }

// -----------------------------------------------------------------------------
// Sets the data shown in editor and sets the focus on editor.
// -----------------------------------------------------------------------------
void CDescriptorExContainer::SetTextL(const TDesC& aText)
{
    iTextBox->SetTextL( &aText );
    iTextBox->SetFocus( ETrue );
    
    iTextBox->HandleTextChangedL();
    iTextBox->UpdateScrollBarsL();
    
}

// -----------------------------------------------------------------------------
// Return number of objects in this container
// -----------------------------------------------------------------------------
TInt CDescriptorExContainer::CountComponentControls() const
    {
    return 1; // return nbr of controls inside this container
    }

// -----------------------------------------------------------------------------
// Handle up and down keys to scroll the edwin control.
// 
// For some odd reason the edwin control doesn't process keyevent expect in
// case that cursor is in the beginning or end of the document.
// -----------------------------------------------------------------------------
TKeyResponse CDescriptorExContainer::OfferKeyEventL(const TKeyEvent& aKeyEvent,
                                                    TEventCode aType)
{
    TKeyResponse result = EKeyWasNotConsumed;
    switch(aKeyEvent.iScanCode)
    {
    case EStdKeyRightArrow:
    case EStdKeyDownArrow:
        {
            result = EKeyWasConsumed;
            if( aType == EEventKeyDown )
            {
                iTextBox->MoveCursorL( TCursorPosition::EFLineDown, EFalse );
            }
            break;
        }

    case EStdKeyLeftArrow:
    case EStdKeyUpArrow:
        {
            result = EKeyWasConsumed;
            if( aType == EEventKeyDown )
            {
                iTextBox->MoveCursorL( TCursorPosition::EFLineUp, EFalse );
            }
            break;
        }
    }
    return result;
}

// -----------------------------------------------------------------------------
// Return component at requested index.
// -----------------------------------------------------------------------------
CCoeControl* CDescriptorExContainer::ComponentControl(TInt aIndex) const
    {
    switch ( aIndex )
        {
        case 0:
            return iTextBox;
        default:
            return NULL;
        }
    }

TInt CDescriptorExContainer::GetScrollbarWidth() const
    {
    TInt scrollbarWidth = iTextBox->ScrollBarFrame()->
        ScrollBarBreadth(CEikScrollBar::EVertical);
    // If scrollbars are not drawn yet, the width remains zero. In this
    // case, an intentionally magical number is returned.
    if (scrollbarWidth == 0)
        {
        scrollbarWidth = 8;
        }
    
    return scrollbarWidth;
    }


void CDescriptorExContainer::SizeChanged()
    {
    TRect rect = Rect();
    TInt scrollbarWidth = GetScrollbarWidth();
    iTextBox->SetExtent(TPoint(0, 0),
            TSize(rect.Width() - scrollbarWidth, rect.Height()));

    iTextBox->UpdateScrollBarsL();
    
    // Update new size for the skin
    iSkinContext->SetRect(rect);
    }


TTypeUid::Ptr CDescriptorExContainer::MopSupplyObject(TTypeUid aId)
    {
    // That is for the skin support
    if( aId.iUid == MAknsControlContext::ETypeId && iSkinContext != NULL )
        {
        return MAknsControlContext::SupplyMopObject( aId, iSkinContext );
        }
    return CCoeControl::MopSupplyObject( aId );    
    }

