// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Contains definition of functions defined in the CMyStringReverse class.
// 
//

/**
 @file
*/

#include "mystringreverse.h"

/**
Creates an object of the CMyStringReverse class.
@param aConsole The user console.
*/
CMyStringReverse* CMyStringReverse::NewL(CConsoleBase* aConsole)
	{
	CMyStringReverse* self = new (ELeave)CMyStringReverse(aConsole);
	self->AddToScheduler();
	return self;
	}

/**
Adds this active object to the active scheduler
*/
void CMyStringReverse::AddToScheduler()
	{
	CActiveScheduler::Add(this);
	}

/**
Constructor.
@param aConsole The user console.
*/
CMyStringReverse::CMyStringReverse(CConsoleBase* aConsole):
														// Constructor of the base class.
														CActive(CActive::EPriorityUserInput),
														// Create the linked list.
														iString(_FOFF(TLetter,iDLink)),
														// Initialize iterators.
														iIteratorString(iString),
														iConsole(aConsole)
	{
	}

/**
Handles the key press events for this active object.
The following actions are defined for a key press event:
- Call the iString.AddFirst() function if '1' is pressed.
- Call the iString.AddLast() function if '2' is pressed.
- Stop the active scheduler if any other number keys are pressed.
- Add the character to iString if any alphabetic keys are pressed
  depending on the previous numeric key press.
*/
void CMyStringReverse::RunL()
	{
	// Get the key code.
	TUint8 option = iConsole->KeyCode();
	// Print the selected option.
	_LIT(KTextFormat,"%c\n");
	iConsole->Printf(KTextFormat,option);

	// Check if a numeric key is pressed.
	TInt number = option - (TUint)'0';
	if(number <10 && number > -1)
		{
		// Handle the numeric key press.
		switch(number)
			{
			case 1:
				{
				// The iString.AddFirst() function needs to be called.
				// Read the character to be added to the list.
				iTask = ETaskStringAddFirst;
				ReadChar();
				}
				break;
			case 2:
				{
				// The iString.AddLast() function needs to be called.
				// Read the character to be added to the list.
				iTask = ETaskStringAddLast;
				ReadChar();
				}
				break;
			default:
				// Stop the active scheduler.
				CActiveScheduler::Stop();
			}
		}
	else
		{
		// An alphabetic key is pressed.
		// Check the action to be performed.
		switch(iTask)
			{
			case ETaskNone:
				_LIT(KInvalidOption,"Invalid Option!\n");
				iConsole->Printf(KInvalidOption);
				break;
			// The iString.AddFirst() function needs to be called.
			case ETaskStringAddFirst:
				{
				// Create an object of the TLetter class.
				TLetter* letter = new (ELeave) TLetter(option);
				// Add the TLetter object into the list.
				iString.AddFirst(*letter);
				iTask = ETaskNone;
				// Print the data in iString and iReverseString.
				PrintStrings();
				}
				break;
			case ETaskStringAddLast:
				{
				// Create an object of the TLetter class.
				TLetter* letter = new (ELeave) TLetter(option);
				// Add the TLetter object into the list.
				iString.AddLast(*letter);
				iTask = ETaskNone;
				// Print the data in iString and iReverseString.
				PrintStrings();		
				}
				break;
			default:
				iTask = ETaskNone;
				break;
			}
		// Generate an asynchronous read request.
		ReadOption();
		}
	}

/**
Issues an outstanding request to get a keystroke from the console.
*/
void CMyStringReverse::ReadFunc()
	{
	// Wait for a key press event.
	iConsole->Read(iStatus);
	SetActive();
	}

/**
Reads the option entered by the user.
*/
void CMyStringReverse::ReadOption()
	{
	// Print the menu.
	_LIT(KTextMenu,"\nEnter\n1-> Add a character to the beginning of the string\n2-> Add a character to the end of the string\nAny other NUMBER to stop\n");
	iConsole->Printf(KTextMenu);
	// Generate an asynchronous read request.
	ReadFunc();
	}

/**
Reads the character entered by the user.
*/
void CMyStringReverse::ReadChar()
	{
	_LIT(KTextReadChar,"\nEnter a character\n");
	iConsole->Printf(KTextReadChar);
	// Generate an asynchronous read request.
	ReadFunc();
	}

/**
Prints the data in iString and iReverseString.
*/
void CMyStringReverse::PrintStrings()
	{
	_LIT(KTextSrcString,"Source String: ");
	iConsole->Printf(KTextSrcString);
	// Initialize the iterator.
	iIteratorString.SetToFirst();

	// Iterate iString.
	while(iIteratorString != NULL)
		{
		// Get the TLetter object pointed to by the iterator.
		TLetter letter = *iIteratorString;
		// Print the character value of the TLetter object.
		_LIT(KTextChar,"%c");
		iConsole->Printf(KTextChar,TUint(letter.iChar));
		// Set the iterator to point to the next element.
		iIteratorString++;
		}

	_LIT(KTextRevString,"\nReverse of the String: ");
	iConsole->Printf(KTextRevString);
	// Initialize the iterator.
	iIteratorString.SetToLast();

	// Iterate iReverseString.
	while(iIteratorString != NULL)
		{
		// Get the TLetter object pointed to by the iterator.
		TLetter letter = *iIteratorString;
		// Print the character value of the TLetter object.
		_LIT(KTextChar,"%c");
		iConsole->Printf(KTextChar,TUint(letter.iChar));
		// Set the iterator to point to the next element.
		iIteratorString--;
		}
	iConsole->Printf(KTextNewLine);
	}

/**
Cancel any outstanding request.
*/
void CMyStringReverse::DoCancel()
	{
	if(IsActive())
		{
		// Cancel any outstanding read requests.
		iConsole->ReadCancel();
		}
	}

/**
Destructor.
The TLetter objects were allocated memory at runtime.
Iterate over the list to delete them.
*/
CMyStringReverse::~CMyStringReverse()
	{
	// Delete the elements from the list.		
		iIteratorString.SetToFirst();
		// Iterate the list and delete all TLetter objects.
		TLetter* ptr = iIteratorString++;
		while (ptr != NULL)
			{
			delete ptr;
			ptr = iIteratorString++;
			}	
	}

